/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces;

import java.io.ByteArrayInputStream;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;

import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamSource;

import org.apache.neethi.Policy;
import org.apache.xerces.dom.DocumentImpl;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.PolicyDocumentParser;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util.PolicyReader;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier.Scheme;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature.MetadataType;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyDocument;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyException;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyFactory;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

public final class WSPolicyFactoryImpl implements WSPolicyFactory.Provider {

    public static final String NAME = "ws-policy-15-neethi";

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ExtensionFactory$Provider#spi_getName()
     */
    public String spi_getName() {
        return NAME;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyFactory$Provider#spi_dereferenceLocation(org.xml.sax.EntityResolver,
     *      java.lang.String)
     */
    public InputSource spi_dereferenceLocation(EntityResolver entityResolver, String location) throws WSPolicyException {

        // policy expression is defined remotely. if optional feature, ws-mex is
        // enabled, we run the request through it in case policy is resolved via
        // soap request/response. otherwise retrieve using our entity resolver

        Server server = SystemContext.getContext().getSystem().getServer();
        WSMexFeature metadataFeature = server.getFeature(WSMexFeature.class);

        InputSource is = null;
        if (metadataFeature == null) {
            try {
                is = entityResolver.resolveEntity(null, location);
            } catch (Exception ex) {
                throw new WSPolicyException("Error dereferencing policy from URI '" + location + "'. " + ex);
            }
        } else {
            try {
                is = metadataFeature.getMetadata(MetadataType.POLICY, location);
            } catch (Exception ex) {
                throw new WSPolicyException("Error dereferencing policy from URI '" + location + "'. " + ex);
            }
        }

        return is;

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyFactory$Provider#spi_readPolicy(org.xml.sax.EntityResolver, javax.xml.transform.Source)
     */
    public WSPolicyDocument spi_readPolicy(EntityResolver entityResolver, Source source) throws WSPolicyException {

        if (source == null) {
            return new WSPolicyDocumentImpl();
        }

        InputSource is = null;

        if (source instanceof SAXSource) {
            is = ((SAXSource)source).getInputSource();
        } else if (source instanceof StreamSource) {
            is = new InputSource(source.getSystemId());
            is.setByteStream(((StreamSource)source).getInputStream());
        } else if (source instanceof DOMSource) {
            try {
                StringWriter sw = new StringWriter();
                DOMSerializer.serializeNode(((DOMSource)source).getNode(), sw, "UTF-8", false);
                is = new InputSource(source.getSystemId());
                is.setByteStream(new ByteArrayInputStream(sw.toString().getBytes("UTF-8")));
            } catch (Exception ex) {
                throw new WSPolicyException(ex);
            }
        }
        
        // parse the source into our policy document impl which will generate
        // an instance of WSPolicyAssertionImpl for each element not defined
        // within ws-policy namespace

        WSPolicyDocumentImpl policyDocument = null;
        try {
            PolicyDocumentParser pp = new PolicyDocumentParser(entityResolver);
            pp.parse(is);
            policyDocument = (WSPolicyDocumentImpl)pp.getDocument();
            policyDocument.setTargetNamespace(source.getSystemId());
        } catch (Exception ex) {
            throw new WSPolicyException("Error reading policy. " + ex);
        }

        // iterate over policy reference elements and normalize each url. see
        // below for comments
        
        NodeList nodeList = null;
        Element root = policyDocument.getDocumentElement();
        nodeList = root.getElementsByTagNameNS(WSPolicyDocument.TYPE, "PolicyReference");
        for (int i = 0; i < nodeList.getLength(); i++) {
            Element policyReference = (Element)nodeList.item(i);
            String location = policyReference.getAttribute("URI");
            if (location.equals("")) {
                throw new WSPolicyException("Error reading policy. Expression contains a"
                        + " PolicyReference element with an empty URI.");
            }
            policyReference.setAttribute("URI", normalizeLocation(source.getSystemId(), location));
        }

        // now parse into a neethi policy object using our reader impl which
        // wraps each WSPolicyAssertionImpl element with a 'wrapper' that impls
        // the neethi Assertion interface, i.e. WSPolicyAssertionWrapper. note
        // that the assertion remains attached to the policy document.
        
        Policy policy = PolicyReader.readPolicyDocument(entityResolver, policyDocument);
        
        // set reference to neethi policy object on document element, i.e. the
        // policy expression element. policy expression element uses it as the
        // underlying policy implementation, i.e. to perform policy operations

        ((WSPolicyExpressionImpl)policyDocument.getDocumentElement()).setUnderlying(policy);
        
        // set the document as read only. policy operations always result
        // in the creation of a new document

        ((DocumentImpl)policyDocument).setReadOnly(true, true);
        
        // we're done. return the generated document which wraps the neethi
        // policy object. the returned object can be used as a document, e.g.
        // to serialize the expression, and as a policy expression, e.g. to
        // perform policy operations
        
       return policyDocument;

    }

    private String normalizeLocation(String systemID, String location) throws WSPolicyException {

        if (location.startsWith("#")) {

            // decode fragment to remove any escaped octets before attempting to parse
            // into a fragmentid

            String fragment;
            try {
                fragment = new URI(location).getFragment();
            } catch (URISyntaxException ue) {
                throw new WSPolicyException("Error attempting to resolve referenced policy at URI '"
                        + location
                        + "'. "
                        + ue);
            }

            // location is a fragment, i.e. uri references a deployed policy expression

            FragmentIdentifier fragmentIdentifier = null;
            try {

                // attempt to parse fragment into a fragment id, i.e. if value fully
                // qualified. note that if parent expression is embedded within a
                // runtime engine definition, i.e. parent document is not wsdl, then
                // this is the only allowable form

                fragmentIdentifier = FragmentIdentifier.valueOf(fragment);

            } catch (IllegalArgumentException ie) {

                if (systemID == null) {
                    throw new WSPolicyException("Error attempting to resolve referenced policy at URI '"
                            + location
                            + "'. SystemID is null.");
                }

                // we assume fragment is just policy id and remainder of fragmentid is
                // implied using target namespace of parent document, i.e. wsdl doc

                fragmentIdentifier = new FragmentIdentifier(systemID, Scheme.EOA, "policy", fragment);

            }

            // create location with '#' prepended and escape any illegal chars within
            // framgent, e.g. the '#' character which occurs after document uri
            
            try {
                return new URI(null, null, fragmentIdentifier.toString()).toString();
            } catch (URISyntaxException ue) {
                throw new WSPolicyException("Error normalizing policy reference URI. " + ue.getMessage());
            }

        } else {

            // location is a url, i.e. policy is defined remotely. leave value
            // as is

            return location;

        }

    }

}
