/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;

import org.apache.xerces.parsers.DOMParser;
import org.apache.xerces.xni.Augmentations;
import org.apache.xerces.xni.NamespaceContext;
import org.apache.xerces.xni.XMLAttributes;
import org.apache.xerces.xni.XMLLocator;
import org.apache.xerces.xni.XNIException;
import org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces.WSPolicyDocumentImpl;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyFactory;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyDocument;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyException;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * Parses an input source into a policy document. This class is NOT threadsafe, but is
 * reusable.
 * 
 */
public final class PolicyDocumentParser extends DOMParser implements ErrorHandler, DocumentContext {

    private static Log log = SystemContext.getContext().getSystem().getLog(PolicyDocumentParser.class);

    private PolicyDocumentValidator validator = null;
    private ArrayList<String> errorMessages = new ArrayList<String>();

    public PolicyDocumentParser(EntityResolver entityResolver) throws WSPolicyException {

        setEntityResolver(entityResolver);

        setErrorHandler(this);

        try {
            setFeature(Constants.NAMESPACES_FEATURE, true);
            setProperty(Constants.DOCUMENT_CLASS_NAME_PROPERTY, Constants.POLICY_DOCUMENT_CLASS_NAME);
        } catch (Exception ex) {
            throw new WSPolicyException(ex);
        }

        this.validator = PolicyDocumentValidator.getInstance();

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.dfault.util.DocumentContext#getDeploymentContext()
     */
    public DeploymentContext getDeploymentContext() {
        return (DeploymentContext)getEntityResolver();
    }

    /**
     * Parses inputSource into a Schema document
     * 
     * @param inputSource
     * @throws SAXException
     * @throws IOException
     */
    @Override
    public void parse(InputSource inputSource) throws SAXException, IOException {

        errorMessages.clear();

        super.parse(inputSource);

        // validation is not enabled on super, but we need to check for
        // parsing errors

        if (errorMessages.size() > 0) {
            Iterator<String> messageIterator = errorMessages.iterator();
            StringBuilder errorMessage = new StringBuilder();
            while (messageIterator.hasNext()) {
                errorMessage.append(messageIterator.next() + "  ");
            }
            throw new SAXException(errorMessage.toString());
        }

        // we validate post-parse. if schema validation feature is enabled
        // on super pre-parse, we end up getting psvi on dom, i.e. default
        // values for attributes etc ...

        validator.validate(new DOMSource(getDocument()));
        
        // note that document is marked as read only by factory after all
        // policy reference uris have been normalized

    }

    @Override
    public void startDocument(XMLLocator locator, String encoding, NamespaceContext namespaceContext, Augmentations augs) throws XNIException {
        super.startDocument(locator, encoding, namespaceContext, augs);
        ((WSPolicyDocumentImpl)fDocument).setDocumentContext(this);
    }

    @Override
    public void startElement(org.apache.xerces.xni.QName element, XMLAttributes attributes, Augmentations augs) throws XNIException {
        if (log.isTraceEnabled()) {
            log.trace("start element: " + new QName(element.uri, element.localpart));
        }
        super.startElement(element, attributes, augs);
    }

    @Override
    public void endElement(org.apache.xerces.xni.QName element, Augmentations augs) throws XNIException {
        if (log.isTraceEnabled()) {
            log.trace("end element: " + new QName(element.uri, element.localpart));
        }
        super.endElement(element, augs);
    }
    
    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.wsdl.wsdl11.xerces.util.DocumentContext#getFactory()
     */
    public PolicyFactory getFactory() {
        return (PolicyFactory)SystemContext.getContext().getSystem().getPolicyFactory(WSPolicyDocument.TYPE);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.open.eoa.ext.wsdl.wsdl11.xerces.util.DocumentContext#getParent()
     */
    public Element getParent() {
        if (fCurrentNode != null && fCurrentNode.getNodeType() == Node.ELEMENT_NODE) {
            return (Element)fCurrentNode;
        }
        return null;
    }

    /** ************************* ErrorHandler Impl ************************************* */

    public void error(SAXParseException spe) throws SAXException {
        errorMessages.add(spe.getMessage());
    }

    public void warning(SAXParseException spe) throws SAXException {
        errorMessages.add(spe.getMessage());
    }

    public void fatalError(SAXParseException spe) throws SAXException {
        errorMessages.add(spe.getMessage());
    }

}
