/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.policy.wsp15.xerces;

import java.net.URI;
import java.net.URISyntaxException;

import javax.xml.transform.sax.SAXSource;

import org.apache.neethi.Policy;
import org.apache.neethi.PolicyRegistry;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.RootComponent;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyDocument;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature.MetadataType;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyException;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyFactory;
import org.bluestemsoftware.specification.eoa.ext.policy.wsp15.WSPolicyDocument;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

/**
 * Resolves policy references embedded within a policy expression. References are resolved when
 * parent expression is normalized. If reference URI is a fragment, referenced expression is a
 * deployed eoa policy component (URI fragment was augmented within our policy factory impl).
 * Otherwise URI is a URL, i.e. expression is defined remotely, in which case we retrieve via
 * ws-mex feature if enabled otherwise via entity resolver.
 */
public final class PolicyRegistryImpl implements PolicyRegistry {

    private static PolicyRegistryImpl singleton = new PolicyRegistryImpl();

    public static final PolicyRegistryImpl getInstance() {
        return singleton;
    }
    
    /*
     * (non-Javadoc)
     * @see org.apache.neethi.PolicyRegistry#lookup(java.lang.String)
     */
    public Policy lookup(String key) {
        return lookup(null, key);
    }
    
    public Policy lookup(EntityResolver entityResolver, String policyURI) {

        if (policyURI == null) {
            throw new IllegalArgumentException("policyURI null");
        }

        if (policyURI.startsWith("#")) {

            // decode fragment to remove any escaped octets before attempting to parse
            // into a fragmentid

            String fragment;
            try {
                fragment = new URI(policyURI).getFragment();
            } catch (URISyntaxException ue) {
                throw new PolicyRegistryException("Error dereferencing policy with URI '" + policyURI + "'. " + ue);
            }

            FragmentIdentifier fragmentIdentifier = null;
            try {
                fragmentIdentifier = FragmentIdentifier.valueOf(fragment);
            } catch (IllegalArgumentException ie) {
                throw new PolicyRegistryException("Error dereferencing policy with URI '" + policyURI + "'. " + ie);
            }

            RootComponent rootComponent = SystemContext.getContext().getSystem().getComponent(fragmentIdentifier);

            if (rootComponent == null) {
                return null;
            }

            PolicyExpression pe = null;
            pe = ((org.bluestemsoftware.specification.eoa.component.policy.Policy)rootComponent)
                    .getRuntimeProvider().getPolicyExpression();

            return ((WSPolicyExpressionImpl)pe).getUnderlying();

        } else {

            // policy expression is defined remotely. if optional feature, ws-mex is 
            // enabled, we run the request through it in case policy is resolved via
            // soap request/response. otherwise retrieve using entity resolver

            Server server = SystemContext.getContext().getSystem().getServer();
            WSMexFeature metadataFeature = server.getFeature(WSMexFeature.class);

            InputSource is = null;
            if (metadataFeature == null) {
                if (entityResolver == null) {
                    throw new PolicyRegistryException("No entity resolver supplied");
                }
                try {
                    is = entityResolver.resolveEntity(null, policyURI);
                } catch (Exception ex) {
                    throw new PolicyRegistryException("Error dereferencing policy from URI '"
                            + policyURI
                            + "'. "
                            + ex);
                }
            } else {
                try {
                    is = metadataFeature.getMetadata(MetadataType.POLICY, policyURI);
                } catch (Exception ex) {
                    throw new PolicyRegistryException("Error dereferencing policy from URI '"
                            + policyURI
                            + "'. "
                            + ex);
                }
            }
            
            PolicyExpression policyExpression = null;
            try {
                PolicyFactory pf = SystemContext.getContext().getSystem().getPolicyFactory(WSPolicyDocument.TYPE);
                PolicyDocument pd = pf.readPolicy(entityResolver, new SAXSource(is));
                policyExpression = pd.getPolicyExpression();
            } catch (PolicyException de) {
                throw new PolicyRegistryException("Error dereferencing policy with URI '" + policyURI + "'. " + de);
            }

            return ((WSPolicyExpressionImpl)policyExpression).getUnderlying();

        }

    }

    /*
     * (non-Javadoc)
     * @see org.apache.neethi.PolicyRegistry#register(java.lang.String, org.apache.neethi.Policy)
     */
    public void register(String policyURI, Policy policy) {
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * @see org.apache.neethi.PolicyRegistry#remove(java.lang.String)
     */
    public void remove(String policyURI) {
        throw new UnsupportedOperationException();
    }

    static class PolicyRegistryException extends RuntimeException {

        private static final long serialVersionUID = 1L;

        public PolicyRegistryException() {
            super();
        }

        public PolicyRegistryException(String message, Throwable cause) {
            super(message, cause);
        }

        public PolicyRegistryException(String message) {
            super(message);
        }

        public PolicyRegistryException(Throwable cause) {
            super(cause);
        }

    }

}
