/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http;

import java.io.IOException;
import java.net.ProtocolException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.Servlet;
import javax.xml.namespace.QName;

import org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule.WSSpecification;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.HostInfo;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClient;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientResponse;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransportModule;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.rt.TransportFault;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public final class HTTPTransport {

    private static final Log log = SystemContext.getContext().getSystem().getLog(HTTPTransportModule.class);

    public static HostInfo ANY_HOST = new HostInfo("_AnyHost");

    private static final long serialVersionUID = 1L;
    private Map<String, WSSpecification> supportedVersions;
    private Map<HostInfo, HTTPClient> clients;

    Servlet servlet;
    Servlet selfServlet;

    public HTTPTransport(Map<String, WSSpecification> supportedVersions, Servlet servlet, Servlet selfServlet,
            Map<HostInfo, HTTPClient> clients) {
        this.supportedVersions = supportedVersions;
        this.servlet = servlet;
        this.selfServlet = selfServlet;
        this.clients = clients;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
    }

    public void destroy() {
        if (servlet != null) {
            servlet.destroy();
        }
        if (selfServlet != null) {
            selfServlet.destroy();
        }
        if (clients != null) {
            clients.clear();
        }
    }

    Map<String, WSSpecification> getSupportedVersions() {
        return supportedVersions;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransport$Provider#spi_doSend(org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientRequest)
     */
    public HTTPClientResponse spi_doSend(HTTPClientRequest request, boolean handleRedirects) throws TransportFault {

        String host = null;
        int port = 0;
        try {
            URI uri = new URI(request.getAddress());
            host = uri.getHost();
            port = uri.getPort();
            port = port == -1 ? 80 : port; // default
        } catch (URISyntaxException fatchance) {
        }

        // first attempt to retrieve the most specific configuration
        // for indicated host, i.e. with port

        HostInfo hostInfo = new HostInfo(host, port);
        HTTPClient client = clients.get(hostInfo);

        // if that fails, attempt to retreive the next most specific
        // configuration, i.e. sans port

        if (client == null) {
            if (log.isDebugEnabled()) {
                log.debug("no client configured for " + hostInfo);
            }
            hostInfo = new HostInfo(host);
            client = clients.get(hostInfo);
        }

        // if that fails, we use the default client configuration
        // which is host agnostic

        if (client == null) {
            if (log.isDebugEnabled()) {
                log.debug("no client configured for " + hostInfo);
            }
            hostInfo = ANY_HOST;
            client = clients.get(hostInfo);
        }

        if (log.isDebugEnabled()) {
            if (hostInfo == ANY_HOST) {
                log.debug("using default, non-authenticating client to send message");
            } else {
                log.debug("using client configured for " + hostInfo + " to send message");
            }
        }

        HTTPClientResponse response = null;
        try {
            response = client.doSend(request);
            int sc = response.getStatusCode();
            if (handleRedirects && (sc >= 300 && sc < 400)) {
                response = handleRedirect(new ArrayList<String>(), request, response);
            }
        } catch (IOException ie) {

            if (log.isDebugEnabled()) {
                log.debug("Error sending request " + request + ". " + ie);
            }

            // if IOException is an instanceof ProtocolException, then it's
            // probably not recoverable. otherwise it MAY be transient in
            // nature. note that we convert cause to a string such that
            // fault reason will contain stack trace and causes. the fault
            // is handled internally, so this should be o.k.

            QName faultName;
            if (ie instanceof ProtocolException) {
                faultName = TransportFault.PROTOCOL_ERROR;
            } else {
                faultName = TransportFault.TRANSPORT_ERROR;
            }

            throw new TransportFault(faultName, ie.toString());

        }

        return response;
    }

    private HTTPClientResponse handleRedirect(List<String> redirects, HTTPClientRequest request, HTTPClientResponse response) throws TransportFault, IOException {

        // TODO: we need to configure the max number of attempts via
        // ws-transport policy

        if (redirects.size() > 3) {
            throw new TransportFault(TransportFault.TRANSPORT_ERROR,
                    "Maximum number of redirect attempts reached. Redirection trail: " + redirects + ".");
        }

        // if no valid location was provided for the redirection,
        // interpret as a 404 and throw fault (with a more
        // meaningful description other than 'Not Found'.

        String redirect = response.getHeaders().get("location");
        if (redirect == null || redirect.equals("")) {
            String error = null;
            if (redirects.size() > 0) {
                error = "Request sent to '"
                        + request.getAddress()
                        + "' was redirected with code "
                        + response.getStatusCode()
                        + " but response failed to contain a valid redirect location. Redirection trail: "
                        + redirects;
            } else {
                error = "Request sent to '"
                        + request.getAddress()
                        + "' was redirected with code "
                        + response.getStatusCode()
                        + " but response failed to contain a valid redirect location.";
            }
            throw new TransportFault(TransportFault.PROTOCOL_ERROR, error);
        }

        if (log.isDebugEnabled()) {
            log.debug("handling redirect to location '" + redirect + "'.");
        }

        // we could conceivably loop if redirected to a location
        // already attempted

        if (redirects.contains(redirect)) {
            redirects.add(redirect);
            throw new TransportFault(TransportFault.PROTOCOL_ERROR,
                    "Circular redirection detected. Redirection trail: " + redirects);
        } else {
            redirects.add(redirect);
        }

        // we may be redirected to a different host. so check to
        // see if a client has been specifically configured

        String host = null;
        try {
            URI uri = new URI(redirect);
            host = uri.getHost();
        } catch (URISyntaxException fatchance) {
        }

        HTTPClient client = clients.get(host);
        if (client == null) {
            client = clients.get(ANY_HOST);
        }

        // everything within original request is valid, except for the
        // to address. continue to redirect until either max number of
        // attempts is reached, or a non-redirected code is returned

        request = new HTTPClientRequest(redirect, request.getMethod(), request.getRequestHeaders(), request
                .getRequestEntity());
        response = client.doSend(request);

        int sc = response.getStatusCode();
        if (sc >= 300 && sc < 400) {
            return handleRedirect(redirects, request, response);
        } else {
            return response;
        }

    }

}
