/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http;

import java.io.StringWriter;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.Servlet;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMUtils;
import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http.util.ClientAuthInfoImpl;
import org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http.util.Constants;
import org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http.util.HTTPTransportServlet;
import org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http.util.ServerAuthInfoImpl;
import org.bluestemsoftware.specification.eoa.DeploymentContext;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.binding.Binding;
import org.bluestemsoftware.specification.eoa.component.engine.rt.BindingModule;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointOperationReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule;
import org.bluestemsoftware.specification.eoa.component.engine.rt.ServiceReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.TransportModule;
import org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule.WSSpecification;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction.Direction;
import org.bluestemsoftware.specification.eoa.component.policy.rt.ActionPolicy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyAlternative;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyAssertion;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyDocument;
import org.bluestemsoftware.specification.eoa.component.policy.rt.PolicyExpression;
import org.bluestemsoftware.specification.eoa.component.policy.rt.UnsupportedPolicyException;
import org.bluestemsoftware.specification.eoa.component.service.Endpoint;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.binding.http.HTTPBinding;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.HostInfo;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.ClearTextPassword;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.Credential;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.CredentialStore;
import org.bluestemsoftware.specification.eoa.ext.feature.auth.cstore.CredentialStoreFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClient;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.client.HTTPClientFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerFeatureException.UniquePathException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA.WSA10;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPAuthenticationScheme;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransportFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransportFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransportModule;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HyperTextTransferProtocol;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.WSHTTP;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.WSHTTP.WSHTTP10;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.WSHTTP.WSHTTP10.Policy.AuthInfo;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyException;
import org.bluestemsoftware.specification.eoa.ext.policy.PolicyFactory;
import org.bluestemsoftware.specification.eoa.ext.service.HTTPEndpoint;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public final class HTTPTransportFeatureImpl implements HTTPTransportFeature.Provider {
    private static final Log log = SystemContext.getContext().getSystem().getLog(HTTPTransportFeature.class);

    private HTTPTransportFeature consumer;
    private DeploymentContext deploymentContext;
    private javax.xml.validation.Schema policySchema;
    private CredentialStore credentialStore = null;
    private Map<String, HTTPTransport> transports = new HashMap<String, HTTPTransport>();
    private Element configuration;

    public static final String IMPL = HTTPTransportFeatureImpl.class.getName();

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_getPolicyVocabularyNamespaces()
     */
    public Set<String> spi_getPolicyVocabularyNamespaces() {
        Set<String> pvns = new HashSet<String>();
        pvns.add(HTTPTransportFeature.PVNS_10);
        return pvns;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_setConfiguration(org.w3c.dom.Element)
     */
    public void spi_setConfiguration(Element configuration) {
        this.configuration = configuration;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = (HTTPTransportFeature)consumer;
        this.deploymentContext = consumer.getExtensionFactory().getFactoryContext();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c.dom.Element)
     */
    public void spi_init() throws HTTPTransportFeatureException {

        log.debug("init begin");

        if (configuration == null) {
            String loc = "classpath:///schema/http.bluestemsoftware.org.open.eoa.ext.feature.ws.transport.http.config.1.0.xml";
            try {
                Resource resource = deploymentContext.getResource(loc);
                InputSource inputSource = new InputSource(resource.getInputStream());
                inputSource.setSystemId(loc);
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder documentBuilder = factory.newDocumentBuilder();
                documentBuilder.setEntityResolver(deploymentContext);
                configuration = documentBuilder.parse(inputSource).getDocumentElement();
            } catch (Exception ex) {
                throw new HTTPTransportFeatureException("Error retrieving default configuration. " + ex);
            }
        }

        validateConfiguration(new DOMSource(configuration));

        QName childName = new QName(Constants.FEATURE_SCHEMA_NS, "credentialStore");
        Element cstoreElement = DOMUtils.getChildElement(configuration, childName);
        String id = DOMUtils.getText(cstoreElement);

        Server server = SystemContext.getContext().getSystem().getServer();
        CredentialStoreFeature credentialStoreFeature = server.getFeature(CredentialStoreFeature.class);
        credentialStore = credentialStoreFeature.getCredentialStore(id);

        if (credentialStore == null) {
            throw new HTTPTransportFeatureException("Error enabling feature. Credential store '"
                    + id
                    + "' is undefined.");
        }

        log.debug("init end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public void spi_destroy() {
        log.debug("destroy begin");
        // transport modules destroyed separately from feature
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_createDefaultTransportModule()
     */
    public TransportModule spi_createDefaultTransportModule(EndpointReference epr) throws FeatureException {

        ServiceReference serviceReference = epr.getParent();

        HTTPTransport httpTransport = transports.get(epr.getFragmentIdentifier().toString());

        // note that modules enabled/configured via policy have already been created by the
        // time this method is invoked by system, i.e. we will only create default transport
        // if it does not already exist

        if (httpTransport == null) {

            Binding binding = epr.getEndpoint().getBindingReference().getReferencedComponent();

            if (!(binding.getUnderlyingProtocol() instanceof HyperTextTransferProtocol)) {
                throw new HTTPTransportFeatureException("Error creating default transport module of type "
                        + HTTPTransportFeature.TYPE
                        + ". Binding "
                        + binding.getName()
                        + " employed by referenced endpoint must implement "
                        + HyperTextTransferProtocol.class
                        + ".");
            }

            if (serviceReference.isMyService()) {
                try {
                    httpTransport = processMyEndpoint(epr, null, null);
                } catch (UnsupportedPolicyException fatchance) {
                }
            } else {
                try {

                    // wsa policy is defined with policy subject endpoint, meaning
                    // wsa binding module on all actions defined within endpoint
                    // are identical. we use it to retrieve the callback address to
                    // pick up address which may have been overridden by user via
                    // policy. otherwise, default reply to generated by wsa is used

                    URI replyTo = null;

                    EndpointActionReference ear = epr.getEndpointActionReferences().iterator().next();
                    if (ear == null) {

                        log.warn("unable to configure callback address. endpoint reference "
                                + epr.getFragmentIdentifier()
                                + " defines no actions.");

                    } else {

                        // because we are messaging protocol agnostic, we cannot retrieve
                        // ws-addressing binding module by type (which is binding type
                        // specific). but we can look for one that implements the core
                        // ws-addressing spec, which is not protocol specific

                        WSA wsa = null;
                        for (BindingModule bindingModule : ear.getBindingModules()) {
                            for (WSSpecification s : bindingModule.getSupportedVersions().values()) {
                                if (s instanceof WSA10) {
                                    wsa = (WSA)s;
                                    break;
                                }
                            }
                        }

                        // currently we support only one version of wsa, but eoa spec
                        // allows support for future versions of wsa spec

                        if (wsa == null) {
                            Binding b = epr.getEndpoint().getBindingReference().getReferencedComponent();
                            if (!b.getBindingType().equals(HTTPBinding.BINDING_TYPE)) {
                                log.warn("unable to configure callback address. ws-addressing module"
                                        + " not enabled on endpoint reference "
                                        + epr.getFragmentIdentifier()
                                        + ".");
                            }
                        } else {
                            WSSpecification.Policy wsaPolicy = wsa.getPolicy().iterator().next();
                            Boolean anonResponses = ((WSA10.Policy)wsaPolicy).isAnonymousResponses();
                            if (anonResponses != Boolean.TRUE) {
                                String temp = ((WSA10.Policy)wsaPolicy).getReplyToAddress();
                                if (temp == null) {
                                    if (anonResponses == null) {
                                        // user is forcing anon responses via private policy
                                    } else {
                                        throw new HTTPTransportFeatureException(
                                                "Error creating default transport module of type "
                                                        + HTTPTransportFeature.TYPE
                                                        + " on endpoint reference "
                                                        + epr.getFragmentIdentifier()
                                                        + ". ws-addressing policy implies non-anonymous responses,"
                                                        + " but no callback address defined.");
                                    }
                                } else {
                                    replyTo = new URI(temp);
                                }
                            }
                        }

                    }

                    httpTransport = processPartnerEndpoint(epr, replyTo, null, null);

                } catch (Exception ex) {
                    throw new HTTPTransportFeatureException("Error creating default transport module of type "
                            + HTTPTransportFeature.TYPE
                            + ". "
                            + ex);
                }
            }

        }

        HTTPTransportModule.Provider provider = new HTTPTransportModuleImpl(httpTransport);
        return new HTTPTransportModule(provider, consumer);

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransportFeature$Provider#spi_createFeatureModules(org.bluestemsoftware.specification.eoa.component.engine.EndpointActionReference,
     *      org.bluestemsoftware.specification.eoa.component.policy.ActionPolicy,
     *      org.bluestemsoftware.specification.eoa.component.policy.ActionPolicy)
     */
    public TransportModule spi_createTransportModule(EndpointActionReference endpointActionReference, ActionPolicy privatePolicy, ActionPolicy publicPolicy) throws FeatureException, UnsupportedPolicyException {
        log.debug("configuring transport module");

        EndpointOperationReference eor = (EndpointOperationReference)endpointActionReference.getParent();
        EndpointReference epr = (EndpointReference)eor.getParent();

        HTTPTransport httpTransport = transports.get(epr.getFragmentIdentifier().toString());

        if (httpTransport == null) {

            ServiceReference serviceReference = (ServiceReference)epr.getParent();
            Binding binding = epr.getEndpoint().getBindingReference().getReferencedComponent();

            if (!(binding.getUnderlyingProtocol() instanceof HyperTextTransferProtocol)) {
                throw new HTTPTransportFeatureException("Error creating transport module of type "
                        + HTTPTransportFeature.TYPE
                        + ". Binding "
                        + binding.getName()
                        + " employed by referenced endpoint must implement "
                        + HyperTextTransferProtocol.class
                        + ".");
            }

            // we only are concerned about policy with policy subject endpoint. any other
            // subject is not supported

            if (privatePolicy.getMessagePolicy() != null) {
                throw new UnsupportedPolicyException(privatePolicy.getMessagePolicy(),
                        "Policy subject MESSAGE not supported");
            }
            if (privatePolicy.getOperationPolicy() != null) {
                throw new UnsupportedPolicyException(privatePolicy.getOperationPolicy(),
                        "Policy subject OPERATION not supported");
            }
            if (privatePolicy.getServicePolicy() != null) {
                throw new UnsupportedPolicyException(privatePolicy.getServicePolicy(),
                        "Policy subject SERVICE not supported");
            }

            if (publicPolicy.getMessagePolicy() != null) {
                throw new UnsupportedPolicyException(publicPolicy.getMessagePolicy(),
                        "Policy subject MESSAGE not supported");
            }
            if (publicPolicy.getOperationPolicy() != null) {
                throw new UnsupportedPolicyException(publicPolicy.getOperationPolicy(),
                        "Policy subject OPERATION not supported");
            }
            if (publicPolicy.getServicePolicy() != null) {
                throw new UnsupportedPolicyException(publicPolicy.getServicePolicy(),
                        "Policy subject SERVICE not supported");
            }

            if (serviceReference.isMyService()) {

                httpTransport = processMyEndpoint(epr, privatePolicy, publicPolicy);

            } else {

                // because we are messaging protocol agnostic, we cannot retrieve
                // ws-addressing binding module by type (which is binding type
                // specific). but we can look for one that implements the core
                // ws-addressing spec, which is not protocol specific

                WSA wsa = null;
                for (BindingModule bindingModule : endpointActionReference.getBindingModules()) {
                    for (WSSpecification s : bindingModule.getSupportedVersions().values()) {
                        if (s instanceof WSA10) {
                            wsa = (WSA)s;
                            break;
                        }
                    }
                }

                // retrieve callback address. if a wsa binding module exists and
                // policy requires non-anonymous responses, or is non-specific
                // and binding defaulted to non-anonymous responses, then the
                // replyTo address on policy is non-null, in which case we config
                // a servlet to receive asnyc responses

                URI cba = null;
                if (wsa != null) {
                    WSSpecification.Policy wsaPolicy = wsa.getPolicy().iterator().next();
                    String replyToAddress = ((WSA10.Policy)wsaPolicy).getReplyToAddress();
                    if (replyToAddress != null) {
                        try {
                            cba = new URL(replyToAddress).toURI();
                        } catch (Exception ex) {
                            throw new HTTPTransportFeatureException("Error creating transport module. " + ex);
                        }
                    } else {
                        log.debug("non-anonymous responses required.");
                    }
                } else {
                    log.debug("no ws-addressing binding module enabled");
                }

                httpTransport = processPartnerEndpoint(epr, cba, privatePolicy, publicPolicy);

            }

        }

        HTTPTransportModule.Provider provider = new HTTPTransportModuleImpl(httpTransport);
        TransportModule httpTransportModule = new HTTPTransportModule(provider, consumer);

        log.debug("transport module configured");
        return httpTransportModule;

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_destroyFeatureModule(org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule)
     */
    public void spi_destroyFeatureModule(FeatureModule featureModule) {
        Extension.Provider p = ((Extension)featureModule).getExtensionProvider();
        Server server = SystemContext.getContext().getSystem().getServer();
        HTTPServerFeature httpServer = server.getFeature(HTTPServerFeature.class);
        httpServer.removeServlet(((HTTPTransportModuleImpl)p).httpTransport.servlet);
    }

    private HTTPTransport processMyEndpoint(EndpointReference epr, ActionPolicy privatePolicy, ActionPolicy publicPolicy) throws HTTPTransportFeatureException, UnsupportedPolicyException {

        if (log.isDebugEnabled()) {
            log.debug("processMyEndpoint begin");
            log.debug("processing " + epr.getFragmentIdentifier().getSchemeSpecificPart());
        }

        ServerAuthInfoImpl serverAuthInfo = null;
        Map<HostInfo, ClientAuthInfoImpl> clientAuthInfos = new HashMap<HostInfo, ClientAuthInfoImpl>();

        Endpoint myEndpoint = null;
        String myScheme = null;
        String contextPath = null;
        try {
            myEndpoint = epr.getEndpoint();
            URI temp = new URI(myEndpoint.getAddress());
            myScheme = temp.getScheme();
            contextPath = temp.getPath();
        } catch (Exception ex) {
            throw new HTTPTransportFeatureException("Error creating transport module. " + ex);
        }

        if (!myScheme.equals("http")) {
            throw new HTTPTransportFeatureException("Error creating transport module. Address "
                    + myEndpoint.getAddress()
                    + " uses an unrecognized scheme. Expected 'http'");
        }

        // address used by remote clients should end with a slash to avoid redirects
        // by http server, as required by servlet spec. the trailing slash needs to be
        // removed, however, when used as a path spec for servlet

        if (contextPath.endsWith("/")) {
            contextPath = contextPath.substring(0, contextPath.length() - 1);
        }

        // merge private effective expression with public effective expression, either
        // of which could be null. note also that we only support a single alternative

        PolicyExpression pe = null;
        if (privatePolicy != null && privatePolicy.getEndpointPolicy() != null) {
            pe = privatePolicy.getEndpointPolicy().getExpression();
            if (pe.getAlternatives().size() != 1) {
                throw new UnsupportedPolicyException(privatePolicy.getEndpointPolicy(),
                        "Policy is invalid. Multiple alternatives not supported");
            }
        }

        if (publicPolicy != null && publicPolicy.getEndpointPolicy() != null) {
            throw new UnsupportedPolicyException(publicPolicy.getEndpointPolicy(),
                    "Policy is invalid. Public transport policy not supported.");
        }

        // we convert the whttp extension attributes into public policy, which
        // is how wsdl spec should have handled it, i.e. deferred it to ...

        if (!(myEndpoint instanceof HTTPEndpoint)) {
            throw new HTTPTransportFeatureException("Error creating transport module. EndpointReference "
                    + epr.getFragmentIdentifier()
                    + " is invalid. Expected referenced endpoint to be an instance of "
                    + HTTPEndpoint.class.getName()
                    + ".");
        } else {
            String s = ((HTTPEndpoint)myEndpoint).getAuthenticationScheme();
            String r = ((HTTPEndpoint)myEndpoint).getAuthenticationRealm();
            ComponentContext context = myEndpoint.getRootComponent().getComponentContext();
            String policyType = null;
            if (pe != null) {
                policyType = ((PolicyDocument)pe.getOwnerDocument()).getExtensionType();
            }
            PolicyExpression temp = createPublicPolicy(context, policyType, HostInfo.MY_HOST, s, r);
            if (temp != null) {
                if (pe == null) {
                    pe = temp;
                } else {
                    pe = pe.mergeExpression(temp);
                }
            }
        }

        if (pe != null) {
            pe = pe.normalizeExpression();
        }

        List<Element> assertions = new ArrayList<Element>();
        if (pe != null) {
            PolicyAlternative alternative = pe.getAlternatives().iterator().next();
            for (PolicyAssertion assertion : alternative) {
                if (!assertion.getNamespaceURI().equals(HTTPTransportFeature.PVNS_10)) {
                    continue;
                }
                try {
                    validatePolicy(new DOMSource(assertion));
                } catch (PolicyValidationException ex) {
                    throw new UnsupportedPolicyException(ex.getMessage());
                }
                assertions.add(assertion);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("assertions in scope:");
            Map<String, Object> parameters = new HashMap<String, Object>();
            parameters.put(DOMSerializer.FORMAT_PRETTY_PRINT, Boolean.TRUE);
            parameters.put(DOMSerializer.XML_DECLARATION, Boolean.FALSE);
            StringBuilder sb = new StringBuilder();
            sb.append(System.getProperty("line.separator"));
            sb.append(System.getProperty("line.separator"));
            for (Element assertion : assertions) {
                StringWriter sw = new StringWriter();
                try {
                    DOMSerializer.serializeNode(assertion, sw, "UTF-8", parameters);
                    sb.append(sw.toString());
                    sb.append(System.getProperty("line.separator"));
                } catch (Exception ex) {
                    log.error("Failed to serialze assertion. " + ex);
                    break;
                }
            }
            log.debug(sb.toString());
        }

        // we need to accomodate for the fact that more than one assertion may be defined
        // for same host, i.e. to supplement a parameter not defined on endpoint, etc ...
        // hold values in auth info objects and throw an exception if values mismatched

        Map<HostInfo, AuthInfo> authInfos = new HashMap<HostInfo, AuthInfo>();
        for (Element assertion : assertions) {
            String host = assertion.getAttribute("host");
            int port = -1;
            if (!assertion.getAttribute("port").equals("")) {
                port = Integer.parseInt(assertion.getAttribute("port"));
            }
            if (host.equals(HostInfo.MY_HOST)) {
                if (port != -1) {
                    throw new UnsupportedPolicyException("Policy is invalid. Port specified for host '"
                            + host
                            + "' must be '-1' which implies all ports.");
                }
            }
            HostInfo hostInfo = new HostInfo(host, port);
            QName childName = new QName(HTTPTransportFeature.PVNS_10, "none");
            Element noneElement = DOMUtils.getChildElement(assertion, childName);
            boolean noAuthentication = false;
            if (noneElement != null) {
                noAuthentication = true;
            }
            childName = new QName(HTTPTransportFeature.PVNS_10, "realm");
            Element realmElement = DOMUtils.getChildElement(assertion, childName);
            String realm = null;
            if (realmElement != null) {
                realm = DOMUtils.getText(realmElement);
            }
            childName = new QName(HTTPTransportFeature.PVNS_10, "scheme");
            Element schemeElement = DOMUtils.getChildElement(assertion, childName);
            HTTPAuthenticationScheme scheme = null;
            if (schemeElement != null) {
                String temp = DOMUtils.getText(schemeElement);
                scheme = HTTPAuthenticationScheme.valueOf(temp.toUpperCase());
            }
            childName = new QName(HTTPTransportFeature.PVNS_10, "user");
            Element userElement = DOMUtils.getChildElement(assertion, childName);
            String user = null;
            if (userElement != null) {
                user = DOMUtils.getText(userElement);
            }
            AuthInfo authInfo = new AuthInfo(realm, scheme, user);
            authInfos.put(hostInfo, authInfo);
            if (host.equals(HostInfo.MY_HOST)) {
                if (user != null) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. Misplaced parameter. User name specified for host '"
                                    + host
                                    + "'. Users for '"
                                    + HostInfo.MY_HOST
                                    + "' are configured within a local realm.");
                }
                try {
                    if (serverAuthInfo == null) {
                        serverAuthInfo = new ServerAuthInfoImpl();
                    }
                    serverAuthInfo.setScheme(scheme);
                    serverAuthInfo.setRealm(realm);
                    if (noAuthentication) {
                        serverAuthInfo.setNoAuthentication();
                    }
                } catch (IllegalArgumentException ie) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. Conflicting parameter value specified. " + ie);
                }
            } else {
                ClientAuthInfoImpl clientAuthInfo = clientAuthInfos.get(hostInfo);
                if (clientAuthInfo == null) {
                    clientAuthInfo = new ClientAuthInfoImpl();
                    clientAuthInfos.put(hostInfo, clientAuthInfo);
                }
                try {
                    clientAuthInfo.setScheme(scheme);
                    clientAuthInfo.setRealm(realm);
                    clientAuthInfo.setUserName(user);
                    if (noAuthentication) {
                        clientAuthInfo.setNoAuthentication();
                    }
                } catch (IllegalArgumentException ie) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. Conflicting parameter value specified. " + ie);
                }
            }
        }

        WSSpecification.Policy policy = new WSHTTP10.Policy(authInfos);
        Set<WSSpecification.Policy> policySet = new HashSet<WSSpecification.Policy>();
        policySet.add(policy);
        WSHTTP wshttp = new WSHTTP10(policySet);

        // now that we've verified none assertion is mutually exclusive of other
        // auth info parms, if no authentication is required set object to null
        // as required by httpServer api

        if (serverAuthInfo != null && serverAuthInfo.noAuthentication()) {
            serverAuthInfo = null;
        }

        // add a servlet to handle inbound requests

        if (log.isDebugEnabled()) {
            log.debug("creating servlet to handle inbound requests");
            log.debug("pathSpec: " + contextPath);
            log.debug("authInfo:");
            log.debug(serverAuthInfo == null ? "non-authenticating" : serverAuthInfo.toString());
        }

        Server eoaServer = SystemContext.getContext().getSystem().getServer();

        Servlet servlet = null;
        HTTPServerFeature httpServer = null;
        try {
            httpServer = eoaServer.getFeature(HTTPServerFeature.class);
            servlet = new HTTPTransportServlet(epr, Direction.IN, false);
            httpServer.addServlet(servlet, contextPath, serverAuthInfo);
        } catch (UniquePathException ue) {
            throw new HTTPTransportFeatureException("Failed to configure servlet. Path "
                    + contextPath
                    + " is not unique.");
        } catch (Exception ex) {
            throw new HTTPTransportFeatureException("Failed to configure servlet. " + ex);
        }

        // if a callback address is configured for my endpoint, i.e. to handle
        // async responses from invocation of self. map callback path to new
        // servlet that handles inbound responses. note that wsa policy is defined
        // with policy subject endpoint, meaning wsa binding module on all actions
        // defined within endpoint are identical

        EndpointActionReference ear = epr.getEndpointActionReferences().iterator().next();

        Servlet callbackServlet = null;

        if (ear == null) {

            // the asynchronous response will fail at runtime with a
            // transport fault

            log.warn("unable to configure callback servlet. endpoint reference "
                    + epr.getFragmentIdentifier()
                    + " defines no actions.");
        } else {

            // because we are messaging protocol agnostic, we cannot retrieve
            // ws-addressing binding module by type (which is binding type
            // specific). but we can look for one that implements the core
            // ws-addressing spec, which is not protocol specific

            WSA wsa = null;
            for (BindingModule bindingModule : ear.getBindingModules()) {
                for (WSSpecification s : bindingModule.getSupportedVersions().values()) {
                    if (s instanceof WSA10) {
                        wsa = (WSA)s;
                        break;
                    }
                }
            }

            // currently we support only one version of wsa, but eoa spec
            // allows support for future versions of wsa spec. if no
            // ws-a policy defined, quietly opt out, i.e. some bindings,
            // e.g. http binding, do not require it

            if (wsa != null) {

                // note that locally hosted engines do not use a fault to epr,
                // i.e. normal and fault responses are returned to same epr

                WSSpecification.Policy wsaPolicy = wsa.getPolicy().iterator().next();
                String replyTo = ((WSA10.Policy)wsaPolicy).getReplyToAddress();

                if (replyTo != null) {

                    String callbackPath = null;
                    try {
                        URI temp = new URI(replyTo);
                        callbackPath = temp.getPath();
                    } catch (Exception ex) {
                        throw new HTTPTransportFeatureException("Error creating transport module. " + ex);
                    }

                    // address used by remote clients should end with a slash to
                    // avoid redirects by http server, as required by servlet spec.
                    // the trailing slash needs to be removed, however, when used
                    // as a path spec for servlet

                    if (callbackPath.endsWith("/")) {
                        callbackPath = callbackPath.substring(0, callbackPath.length() - 1);
                    }

                    // map context path to new servlet. note that because we are
                    // invoked by self, no authentication is performed. within
                    // servlet, however, we verify that host on originating ep
                    // is localhost or my host as a precaution

                    // TODO: we need a test for this within this project (it's
                    // tested within binding test suites)

                    try {
                        callbackServlet = new HTTPTransportServlet(epr, Direction.OUT, true);
                        httpServer.addServlet(callbackServlet, callbackPath, null);
                    } catch (UniquePathException ue) {
                        throw new HTTPTransportFeatureException("Failed to configure servlet. Path "
                                + callbackPath
                                + " is not unique.");
                    } catch (Exception ex) {
                        throw new HTTPTransportFeatureException("Failed to configure servlet. " + ex);
                    }

                } else {

                    // policy requires anonymous responses. replies are sent
                    // over back channel (no need to configure servlet for
                    // callbacks)

                    if (log.isDebugEnabled()) {
                        log.debug("public policy requires anonymous responses. not configuring"
                                + " servlet for callbacks (for self invocation)");
                    }

                }

            }

        }

        // add client(s) to handle async responses

        HTTPClientFeature httpClientFeature = eoaServer.getFeature(HTTPClientFeature.class);

        Map<HostInfo, HTTPClient> clients = new HashMap<HostInfo, HTTPClient>();
        try {
            for (Map.Entry<HostInfo, ClientAuthInfoImpl> entry : clientAuthInfos.entrySet()) {

                HostInfo hostInfo = entry.getKey();
                ClientAuthInfoImpl authInfo = entry.getValue();

                // now that we've verified none assertion is mutually exclusive of other
                // auth info parms, if no authentication is required set object to null
                // as required by httpClient api

                if (authInfo.noAuthentication()) {
                    authInfo = null;
                }

                if (authInfo != null) {
                    if (authInfo.noAuthentication() && authInfo.getScheme() == null) {
                        throw new UnsupportedPolicyException("Policy is invalid."
                                + " Error configuring client for host '"
                                + hostInfo.getHost()
                                + "' and port '"
                                + hostInfo.getPort()
                                + "'. No 'scheme' parameter defined.");
                    }
                    String realm = null;
                    if (authInfo.getRealm() == null) {
                        throw new UnsupportedPolicyException("Policy is invalid."
                                + " Error configuring client for host '"
                                + hostInfo.getHost()
                                + "' and port '"
                                + hostInfo.getPort()
                                + "'. No 'realm' parameter defined.");
                    } else {
                        realm = authInfo.getRealm();
                    }
                    String user = null;
                    if (authInfo.getUserName() == null) {
                        throw new UnsupportedPolicyException("Policy is invalid."
                                + " Error configuring client for host '"
                                + hostInfo.getHost()
                                + "' and port '"
                                + hostInfo.getPort()
                                + "'. No 'user' parameter defined.");
                    } else {
                        user = authInfo.getUserName();
                    }
                    Credential credential = credentialStore.getCredential(hostInfo, realm, user);
                    if (credential == null) {
                        throw new UnsupportedPolicyException("Policy is invalid."
                                + "  No credential found within credential store for user '"
                                + user
                                + "' defined on host '"
                                + hostInfo.getHost()
                                + "' and port '"
                                + hostInfo.getPort()
                                + "' within realm '"
                                + realm
                                + "'.");
                    }
                    String password = null;
                    if (!(credential.getPassword() instanceof ClearTextPassword)) {
                        throw new UnsupportedPolicyException("Policy is invalid."
                                + " Password retrieved from credential store for user '"
                                + user
                                + "' defined on host '"
                                + hostInfo.getHost()
                                + "' and port '"
                                + hostInfo.getPort()
                                + "' within realm '"
                                + realm
                                + "' is not recoverable, i.e. password is a one-way hash.");
                    } else {
                        password = ((ClearTextPassword)credential.getPassword()).getText();
                        authInfo.setPassword(password);
                    }
                }

                if (log.isDebugEnabled()) {
                    log.debug("adding client to return async responses.");
                    log.debug("host: " + hostInfo.getHost());
                    log.debug("port: " + hostInfo.getPort());
                    log.debug("authInfo:");
                    log.debug(authInfo == null ? null : authInfo.toString());
                }
                HTTPClient client = httpClientFeature.createHTTPClient(hostInfo, authInfo);
                clients.put(hostInfo, client);

            }
        } catch (HTTPClientFeatureException ex) {
            throw new HTTPTransportFeatureException("Error configuring client to handle async responses. " + ex);
        }

        // add default client, i.e. a non-authenticating client which will be
        // used to return responses when a host was not specifically configured
        // via policy

        try {
            clients.put(HTTPTransport.ANY_HOST, httpClientFeature.createHTTPClient());
        } catch (HTTPClientFeatureException he) {
            throw new HTTPTransportFeatureException("Error configuring default client to handle async responses. "
                    + he);
        }

        Map<String, WSSpecification> supportedVersions = new HashMap<String, WSSpecification>();
        supportedVersions.put(WSHTTP10.NAMESPACE_URI, wshttp);
        HTTPTransport httpTransport = new HTTPTransport(supportedVersions, servlet, callbackServlet, clients);
        transports.put(epr.getFragmentIdentifier().toString(), httpTransport);

        log.debug("processMyEndpoint end");
        return httpTransport;

    }

    private HTTPTransport processPartnerEndpoint(EndpointReference epr, URI cba, ActionPolicy privatePolicy, ActionPolicy publicPolicy) throws HTTPTransportFeatureException, UnsupportedPolicyException {

        if (log.isDebugEnabled()) {
            log.debug("processPartnerEndpoint begin");
            log.debug("processing " + epr.getFragmentIdentifier().getSchemeSpecificPart());
        }

        ServerAuthInfoImpl serverAuthInfo = null;
        ClientAuthInfoImpl clientAuthInfo = null;

        Endpoint partnerEndpoint = null;
        String partnerScheme = null;
        String partnerHost = null;
        int partnerPort = 0;
        try {
            partnerEndpoint = epr.getEndpoint();
            URI temp = new URI(partnerEndpoint.getAddress());
            partnerScheme = temp.getScheme();
            partnerHost = temp.getHost();
            partnerPort = temp.getPort();
            partnerPort = partnerPort == -1 ? 80 : partnerPort;
        } catch (Exception ex) {
            throw new HTTPTransportFeatureException("Error creating transport module. " + ex);
        }

        if (!partnerScheme.equals("http")) {
            throw new HTTPTransportFeatureException("Error creating transport module. Address "
                    + partnerEndpoint.getAddress()
                    + " uses an urecognized scheme. Expected 'http'");
        }

        // merge private effective expression with public effective expression, either
        // of which could be null. note also that we only support a single alternative

        PolicyExpression pe = null;
        if (privatePolicy != null && privatePolicy.getEndpointPolicy() != null) {
            pe = privatePolicy.getEndpointPolicy().getExpression();
            if (pe.getAlternatives().size() != 1) {
                throw new UnsupportedPolicyException(privatePolicy.getEndpointPolicy(),
                        "Policy is invalid. Multiple alternatives not supported");
            }
        }

        if (publicPolicy != null && publicPolicy.getEndpointPolicy() != null) {
            throw new UnsupportedPolicyException(publicPolicy.getEndpointPolicy(),
                    "Policy is invalid. Public transport policy not supported.");
        }

        // we convert the whttp extension attributes into public policy, which
        // is how wsdl spec should have handled it, i.e. deferred it to ...

        if (!(partnerEndpoint instanceof HTTPEndpoint)) {
            throw new HTTPTransportFeatureException("Error creating transport module. EndpointReference "
                    + epr.getFragmentIdentifier()
                    + " is invalid. Expected referenced endpoint to be an instance of "
                    + HTTPEndpoint.class.getName()
                    + ".");
        } else {
            String s = ((HTTPEndpoint)partnerEndpoint).getAuthenticationScheme();
            String r = ((HTTPEndpoint)partnerEndpoint).getAuthenticationRealm();
            ComponentContext context = partnerEndpoint.getRootComponent().getComponentContext();
            String policyType = null;
            if (pe != null) {
                policyType = ((PolicyDocument)pe.getOwnerDocument()).getExtensionType();
            }
            PolicyExpression temp = createPublicPolicy(context, policyType, partnerHost, s, r);
            if (temp != null) {
                if (pe == null) {
                    pe = temp;
                } else {
                    pe = pe.mergeExpression(temp);
                }
            }
        }

        if (pe != null) {
            pe = pe.normalizeExpression();
        }

        List<Element> assertions = new ArrayList<Element>();
        if (pe != null) {
            PolicyAlternative alternative = pe.getAlternatives().iterator().next();
            for (PolicyAssertion assertion : alternative) {
                if (!assertion.getNamespaceURI().equals(HTTPTransportFeature.PVNS_10)) {
                    continue;
                }
                try {
                    validatePolicy(new DOMSource(assertion));
                } catch (PolicyValidationException ex) {
                    throw new UnsupportedPolicyException(ex.getMessage());
                }
                assertions.add(assertion);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("assertions in scope:");
            Map<String, Object> parameters = new HashMap<String, Object>();
            parameters.put(DOMSerializer.FORMAT_PRETTY_PRINT, Boolean.TRUE);
            parameters.put(DOMSerializer.XML_DECLARATION, Boolean.FALSE);
            StringBuilder sb = new StringBuilder();
            sb.append(System.getProperty("line.separator"));
            sb.append(System.getProperty("line.separator"));
            for (Element assertion : assertions) {
                StringWriter sw = new StringWriter();
                try {
                    DOMSerializer.serializeNode(assertion, sw, "UTF-8", parameters);
                    sb.append(sw.toString());
                    sb.append(System.getProperty("line.separator"));
                } catch (Exception ex) {
                    log.error("Failed to serialze assertion. " + ex);
                    break;
                }
            }
            log.debug(sb.toString());
        }

        // we need to accomodate for the fact that more than one assertion may be defined
        // for same host, i.e. to supplement a parameter not defined on endpoint, etc ...
        // hold values in auth info objects and throw an exception if values mismatched

        Map<HostInfo, AuthInfo> authInfos = new HashMap<HostInfo, AuthInfo>();
        for (Element assertion : assertions) {
            String host = assertion.getAttribute("host");
            int port = -1;
            if (!assertion.getAttribute("port").equals("")) {
                port = Integer.parseInt(assertion.getAttribute("port"));
            }
            HostInfo hostInfo = new HostInfo(host, port);
            QName childName = new QName(HTTPTransportFeature.PVNS_10, "none");
            Element noneElement = DOMUtils.getChildElement(assertion, childName);
            boolean noAuthentication = false;
            if (noneElement != null) {
                noAuthentication = true;
            }
            childName = new QName(HTTPTransportFeature.PVNS_10, "realm");
            Element realmElement = DOMUtils.getChildElement(assertion, childName);
            String realm = null;
            if (realmElement != null) {
                realm = DOMUtils.getText(realmElement);
            }
            childName = new QName(HTTPTransportFeature.PVNS_10, "scheme");
            Element schemeElement = DOMUtils.getChildElement(assertion, childName);
            HTTPAuthenticationScheme scheme = null;
            if (schemeElement != null) {
                String temp = DOMUtils.getText(schemeElement);
                scheme = HTTPAuthenticationScheme.valueOf(temp.toUpperCase());
            }
            childName = new QName(HTTPTransportFeature.PVNS_10, "user");
            Element userElement = DOMUtils.getChildElement(assertion, childName);
            String user = null;
            if (userElement != null) {
                user = DOMUtils.getText(userElement);
            }
            AuthInfo authInfo = new AuthInfo(realm, scheme, user);
            authInfos.put(hostInfo, authInfo);
            if (host.equals(HostInfo.MY_HOST)) {
                if (port != -1) {
                    throw new UnsupportedPolicyException("Policy is invalid. Port '"
                            + port
                            + "' specified for host '"
                            + host
                            + "' must be '-1' which implies all ports.");
                }
                if (user != null) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. Misplaced parameter. User name specified for host '"
                                    + host
                                    + "'. Users for '"
                                    + HostInfo.MY_HOST
                                    + "' are configured within a local realm.");
                }
                try {
                    if (serverAuthInfo == null) {
                        serverAuthInfo = new ServerAuthInfoImpl();
                    }
                    serverAuthInfo.setScheme(scheme);
                    serverAuthInfo.setRealm(realm);
                    if (noAuthentication) {
                        serverAuthInfo.setNoAuthentication();
                    }
                } catch (IllegalArgumentException ie) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. Conflicting parameter value specified. " + ie);
                }
            } else if (host.equals(partnerHost) || host.equals("partnerhost")) {
                if (port > -1 && port != partnerPort) {
                    throw new UnsupportedPolicyException("Policy is invalid. Port '"
                            + port
                            + "' specified for host '"
                            + partnerHost
                            + "' is invalid. Expected "
                            + partnerPort
                            + ".");
                }
                if (clientAuthInfo == null) {
                    clientAuthInfo = new ClientAuthInfoImpl();
                }
                try {
                    clientAuthInfo.setScheme(scheme);
                    clientAuthInfo.setRealm(realm);
                    clientAuthInfo.setUserName(user);
                    if (noAuthentication) {
                        clientAuthInfo.setNoAuthentication();
                    }
                } catch (IllegalArgumentException ie) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. Conflicting parameter value specified. " + ie);
                }
            } else {
                throw new UnsupportedPolicyException("Policy is invalid. Unrecognized host '"
                        + host
                        + "'. Expected either 'myhost', 'partnerhost' or '"
                        + partnerHost
                        + "'.");
            }
        }

        WSSpecification.Policy policy = new WSHTTP10.Policy(authInfos);
        Set<WSSpecification.Policy> policySet = new HashSet<WSSpecification.Policy>();
        policySet.add(policy);
        WSHTTP wshttp = new WSHTTP10(policySet);

        // if callback address is not null, then we need to configure a
        // servlet to handle async responses

        Servlet servlet = null;
        Server eoaServer = SystemContext.getContext().getSystem().getServer();

        if (cba != null) {

            // now that we've verified none assertion is mutually exclusive of other
            // auth info parms, if no authentication is required set object to null
            // as required by httpServer api

            if (serverAuthInfo != null && serverAuthInfo.noAuthentication()) {
                serverAuthInfo = null;
            }

            if (!cba.getScheme().equals("http")) {
                throw new HTTPTransportFeatureException("Error creating transport module. Callback address "
                        + cba.toString()
                        + " uses an urecognized scheme. Expected 'http'");
            }

            String callbackPath = cba.getPath();

            if (callbackPath.equals("")) {
                throw new HTTPTransportFeatureException("Failed to configure servlet. Callback uri "
                        + cba.toString()
                        + " has no path.");
            }

            // address used by remote clients should end with a slash to avoid redirects
            // by http server, as required by servlet spec. the trailing slash needs to be
            // removed, however, when used as a path spec for servlet

            if (callbackPath.endsWith("/")) {
                callbackPath = callbackPath.substring(0, callbackPath.length() - 1);
            }

            if (log.isDebugEnabled()) {
                log.debug("creating servlet to handle async responses");
                log.debug("pathSpec: " + callbackPath);
                log.debug("authInfo:");
                log.debug(serverAuthInfo == null ? "non-authenticating" : serverAuthInfo.toString());
            }

            try {
                HTTPServerFeature httpServer = eoaServer.getFeature(HTTPServerFeature.class);
                servlet = new HTTPTransportServlet(epr, Direction.OUT, false);
                httpServer.addServlet(servlet, callbackPath, serverAuthInfo);
            } catch (UniquePathException ue) {
                throw new HTTPTransportFeatureException("Failed to configure servlet. Path "
                        + callbackPath
                        + " defined on callback address "
                        + cba.toString()
                        + " is not unique within context of Server.");
            } catch (Exception ex) {
                throw new HTTPTransportFeatureException("Failed to configure servlet. " + ex);
            }

        } else {
            log.debug("no callback address configured");
            log.debug("no servlet created to handle async responses");
        }

        // add client to handle outbound request. note that default behavior
        // requires that we add a non-authenticating, host agnostic client
        // if no authentication info found

        HTTPClientFeature httpClientFeature = eoaServer.getFeature(HTTPClientFeature.class);
        HTTPClient client = null;
        Map<HostInfo, HTTPClient> clients = new HashMap<HostInfo, HTTPClient>();
        try {

            // now that we've verified none assertion is mutually exclusive of other
            // auth info parms, if no authentication is required set object to null
            // as required by httpClient api

            if (clientAuthInfo != null && clientAuthInfo.noAuthentication()) {
                clientAuthInfo = null;
            }

            HostInfo hostInfo = new HostInfo(partnerHost, partnerPort);

            if (clientAuthInfo == null) {
                if (log.isDebugEnabled()) {
                    log.debug("creating client to handle outbound request.");
                    log.debug("host: " + partnerHost);
                    log.debug("port: " + partnerPort);
                    log.debug("no authentication will be performed.");
                }
                client = httpClientFeature.createHTTPClient(hostInfo, clientAuthInfo);
                clients.put(hostInfo, client);
            } else {
                if (clientAuthInfo.getScheme() == null) {
                    throw new UnsupportedPolicyException("Policy is invalid."
                            + " Error configuring client for host '"
                            + partnerHost
                            + "'. No 'scheme' parameter found.");
                }
                String realm = null;
                if (clientAuthInfo.getRealm() == null) {
                    throw new UnsupportedPolicyException("Policy is invalid."
                            + " Error configuring client for host '"
                            + partnerHost
                            + "'. No 'realm' parameter found.");
                } else {
                    realm = clientAuthInfo.getRealm();
                }
                String user = null;
                if (clientAuthInfo.getUserName() == null) {
                    throw new UnsupportedPolicyException("Policy is invalid"
                            + " Error configuring client for host '"
                            + partnerHost
                            + "'. No 'user' parameter defined.");
                } else {
                    user = clientAuthInfo.getUserName();
                }
                Credential credential = credentialStore.getCredential(hostInfo, realm, user);
                if (credential == null) {
                    credential = credentialStore.getCredential(new HostInfo(partnerHost), realm, user);
                }
                if (credential == null) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. No credential found within credential store for user '"
                                    + user
                                    + "' defined on host '"
                                    + partnerHost
                                    + "' within realm '"
                                    + realm
                                    + "'.");
                }
                String password = null;
                if (!(credential.getPassword() instanceof ClearTextPassword)) {
                    throw new UnsupportedPolicyException(
                            "Policy is invalid. Password retrieved from credential store for user '"
                                    + user
                                    + "' defined on host '"
                                    + partnerHost
                                    + "' within realm '"
                                    + realm
                                    + "' is not recoverable, i.e. password is a one-way hash.");
                } else {
                    password = ((ClearTextPassword)credential.getPassword()).getText();
                    clientAuthInfo.setPassword(password);
                }
                if (log.isDebugEnabled()) {
                    log.debug("creating client to handle outbound request.");
                    log.debug("host: " + partnerHost);
                    log.debug("port: " + partnerPort);
                    log.debug("authInfo:");
                    log.debug(clientAuthInfo.toString());
                }
                client = httpClientFeature.createHTTPClient(hostInfo, clientAuthInfo);
                clients.put(hostInfo, client);
            }
        } catch (HTTPClientFeatureException ex) {
            throw new HTTPTransportFeatureException("Failed to configure client to handle async responses. " + ex);
        }

        Map<String, WSSpecification> supportedVersions = new HashMap<String, WSSpecification>();
        supportedVersions.put(WSHTTP10.NAMESPACE_URI, wshttp);
        HTTPTransport httpTransport = new HTTPTransport(supportedVersions, servlet, null, clients);
        transports.put(epr.getFragmentIdentifier().toString(), httpTransport);

        log.debug("processPartnerEndpoint end");
        return httpTransport;

    }

    private void validatePolicy(DOMSource domSource) throws PolicyValidationException {

        Resource resource = null;
        try {
            resource = deploymentContext.getResource(Constants.POLICY_SCHEMA_LOC);
        } catch (DeploymentException de) {
            throw new PolicyValidationException("Error validating policy. " + de);
        }

        // note that we instantiate schema factory impl directly. a bug in jdk 1.5
        // prevents discovery using jaxp newInstance method. and ... the default
        // jdk 1.5 impl has a bug re: parsing qnames, i.e. a bogus UndeclaredPrefix
        // error

        javax.xml.validation.SchemaFactory schemaFactory = null;
        schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
        if (policySchema == null) {
            try {
                policySchema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
            } catch (Exception ex) {
                throw new PolicyValidationException("Error parsing policy schema. " + ex);
            }
        }

        DOMValidator domValidator = new DOMValidator(policySchema);
        String validationError;
        try {
            validationError = domValidator.validate(domSource);
        } catch (Exception ex) {
            throw new PolicyValidationException("Error validating policy. " + ex);
        }
        if (validationError != null) {
            throw new PolicyValidationException("Policy invalid. " + validationError);
        }

    }

    private void validateConfiguration(DOMSource domSource) throws HTTPTransportFeatureException {

        Resource resource = null;
        try {
            resource = deploymentContext.getResource(Constants.FEATURE_SCHEMA_LOC);
        } catch (DeploymentException de) {
            throw new HTTPTransportFeatureException("Error validating configuration. " + de);
        }

        // note that we instantiate schema factory impl directly. a bug in jdk 1.5
        // prevents discovery using jaxp newInstance method. and ... the default
        // jdk 1.5 impl has a bug re: parsing qnames, i.e. a bogus UndeclaredPrefix
        // error

        javax.xml.validation.SchemaFactory schemaFactory = null;
        schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
        javax.xml.validation.Schema schema = null;
        try {
            schema = schemaFactory.newSchema(new StreamSource(resource.getInputStream()));
        } catch (Exception ex) {
            throw new HTTPTransportFeatureException("Error parsing configuration schema. " + ex);
        }

        DOMValidator domValidator = new DOMValidator(schema);
        String validationError;
        try {
            validationError = domValidator.validate(domSource);
        } catch (Exception ex) {
            throw new HTTPTransportFeatureException("Error validating configuration. " + ex);
        }
        if (validationError != null) {
            throw new HTTPTransportFeatureException("Configuration invalid. " + validationError);
        }

    }

    private PolicyExpression createPublicPolicy(ComponentContext componentContext, String policyType, String host, String scheme, String realm) throws HTTPTransportFeatureException {

        if (scheme == null && realm == null) {
            return null;
        }

        PolicyFactory policyFactory = null;
        policyFactory = SystemContext.getContext().getSystem().getPolicyFactory(policyType);

        if (policyFactory == null) {
            throw new HTTPTransportFeatureException("Unable to generate transport policy. Policy factory "
                    + policyType
                    + " is undefined.");
        }

        // create an empty policy document, which we'll use as a dom factory

        PolicyDocument pd = null;
        try {
            pd = policyFactory.readPolicy(componentContext, null);
        } catch (PolicyException pe) {
            throw new HTTPTransportFeatureException("Error creating implied public policy. " + pe.getMessage());
        }

        // create the new expression using existing language, or default language
        // if no existing expression

        Element policyElement = pd.createElementNS(policyType, "wsp:Policy");
        policyElement.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:wsp", policyType);
        Element authInfoElement = pd.createElementNS(WSHTTP10.Policy.PVN, "dns:authInfo");
        authInfoElement.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:dns", WSHTTP10.Policy.PVN);
        authInfoElement.setAttribute("host", host);
        policyElement.appendChild(authInfoElement);
        Element schemeElement = pd.createElementNS(WSHTTP10.Policy.PVN, "dns:scheme");
        schemeElement.appendChild(pd.createTextNode(scheme));
        authInfoElement.appendChild(schemeElement);
        Element realmElement = pd.createElementNS(WSHTTP10.Policy.PVN, "dns:realm");
        realmElement.appendChild(pd.createTextNode(realm));
        authInfoElement.appendChild(realmElement);

        PolicyExpression policyExpression = null;
        try {
            PolicyDocument tmp = policyFactory.readPolicy(componentContext, new DOMSource(policyElement));
            tmp.deploy(componentContext);
            policyExpression = tmp.getPolicyExpression();
        } catch (Exception ex) {
            throw new HTTPTransportFeatureException("Error creating implied public policy. " + ex.getMessage());
        }

        return policyExpression;

    }

    static class PolicyValidationException extends Exception {

        private static final long serialVersionUID = 1L;

        public PolicyValidationException() {
            super();
        }

        public PolicyValidationException(String message, Throwable cause) {
            super(message, cause);
        }

        public PolicyValidationException(String message) {
            super(message);
        }

        public PolicyValidationException(Throwable cause) {
            super(cause);
        }

    }

}
