/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http.util;

import java.io.ByteArrayOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;

import org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerResponse;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public class HTTPServerResponseImpl implements HTTPServerResponse {

    private Log log = SystemContext.getContext().getSystem().getLog(HTTPServerResponse.class);

    private HttpServletResponse delegate;
    private Map<String, String> headers;
    private int status; // for trace logging
    private OutputStream out;

    public HTTPServerResponseImpl(HttpServletResponse delegate) {
        this.delegate = delegate;
    }

    public void addDateHeader(String name, long date) {
        if (log.isTraceEnabled()) {
            if (headers == null) {
                headers = new HashMap<String, String>();
            }
            headers.put(name, Long.toString(date));
        }
        delegate.addDateHeader(name, date);
    }

    public void addHeader(String name, String value) {
        if (log.isTraceEnabled()) {
            if (headers == null) {
                headers = new HashMap<String, String>();
            }
            headers.put(name, value);
        }
        delegate.addHeader(name, value);
    }

    public void addIntHeader(String name, int value) {
        if (log.isTraceEnabled()) {
            if (headers == null) {
                headers = new HashMap<String, String>();
            }
            headers.put(name, Integer.toString(value));
        }
        delegate.addIntHeader(name, value);
    }

    public boolean containsHeader(String name) {
        return delegate.containsHeader(name);
    }

    public String encodeRedirectURL(String url) {
        return delegate.encodeRedirectURL(url);
    }

    public String encodeURL(String url) {
        return delegate.encodeURL(url);
    }

    public void flushBuffer() throws IOException {
        delegate.flushBuffer();
    }

    public int getBufferSize() {
        return delegate.getBufferSize();
    }

    public String getCharacterEncoding() {
        return delegate.getCharacterEncoding();
    }

    public String getContentType() {
        return delegate.getContentType();
    }

    public Locale getLocale() {
        return delegate.getLocale();
    }

    public OutputStream getOutputStream() throws IOException {
        if (log.isTraceEnabled()) {
            out = new LoggingOutputStream(delegate.getOutputStream());
            return out;
        } else {
            return delegate.getOutputStream();
        }
    }

    public PrintWriter getWriter() throws IOException {
        return delegate.getWriter();
    }

    public boolean isCommitted() {
        return delegate.isCommitted();
    }

    public void reset() {
        delegate.reset();
    }

    public void resetBuffer() {
        delegate.resetBuffer();
    }

    public void sendError(int sc, String msg) throws IOException {
        if (log.isTraceEnabled()) {
            status = sc;
            out = new LoggingOutputStream(msg);
        }
        delegate.sendError(sc, msg);
    }

    public void sendError(int sc) throws IOException {
        status = sc;
        delegate.sendError(sc);
    }

    public void sendRedirect(String location) throws IOException {
        delegate.sendRedirect(location);
    }

    public void setBufferSize(int size) {
        delegate.setBufferSize(size);
    }

    public void setCharacterEncoding(String charset) {
        delegate.setCharacterEncoding(charset);
    }

    public void setContentLength(int len) {
        delegate.setContentLength(len);
    }

    public void setContentType(String type) {
        delegate.setContentType(type);
    }

    public void setDateHeader(String name, long date) {
        if (log.isTraceEnabled()) {
            if (headers == null) {
                headers = new HashMap<String, String>();
            }
            headers.put(name, Long.toString(date));
        }
        delegate.setDateHeader(name, date);
    }

    public void setHeader(String name, String value) {
        if (log.isTraceEnabled()) {
            if (headers == null) {
                headers = new HashMap<String, String>();
            }
            headers.put(name, value);
        }
        delegate.setHeader(name, value);
    }

    public void setIntHeader(String name, int value) {
        if (log.isTraceEnabled()) {
            if (headers == null) {
                headers = new HashMap<String, String>();
            }
            headers.put(name, Integer.toString(value));
        }
        delegate.setIntHeader(name, value);
    }

    public void setLocale(Locale loc) {
        delegate.setLocale(loc);
    }

    public void setStatus(int sc) {
        this.status = sc;
        delegate.setStatus(sc);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.feature.http.server.HTTPServerResponse#trace()
     */
    public void trace() {
        if (log.isTraceEnabled()) {
            log.trace("status: " + status);
            log.trace("headers:");
            log.trace("Content-Type: " + delegate.getContentType());
            if (headers != null) {
                for (Map.Entry<String, String> header : headers.entrySet()) {
                    log.trace(header.getKey() + ": " + header.getValue());
                }
            }
            String content = "";
            if (out != null) {
                content = out.toString();
            }
            log.trace("content:");
            log.trace(content);
        }
    }

    class LoggingOutputStream extends FilterOutputStream {

        private StringBuilder sb = new StringBuilder();

        public LoggingOutputStream(OutputStream out) {
            super(out);
        }

        public LoggingOutputStream(String error) {
            super(new ByteArrayOutputStream());
            sb.append(error);
        }

        @Override
        public void write(int b) throws IOException {
            super.write(b);
            sb.append((char)b);
        }

        @Override
        public String toString() {
            return sb.toString();
        }

    }

}
