/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.transport.http.util;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.bluestemsoftware.specification.eoa.component.binding.rt.BindingRT;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction.Direction;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.rt.HyperTextTransferProtocolRT;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;

public class HTTPTransportServlet extends HttpServlet {

    private static final long serialVersionUID = 1L;

    private static final Log log = SystemContext.getContext().getSystem().getLog(HTTPTransportServlet.class);

    private EndpointReference epr;
    private BindingRT binding;
    private Direction direction;
    private boolean isSelfCallback;
    private String myHost;
    private String myAddress;

    public HTTPTransportServlet(EndpointReference epr, Direction direction, boolean isSelfCallback) {
        this.epr = epr;
        binding = epr.getEndpoint().getBindingReference().getReferencedComponent().getRuntimeProvider();
        this.direction = direction;
        this.isSelfCallback = isSelfCallback;
        if (isSelfCallback) {
            try {
                URI temp = new URI(epr.getEndpoint().getAddress());
                myHost = temp.getHost();
            } catch (Exception fatchance) {
            }
            try {
                myAddress = InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException ue) {
                throw new IllegalStateException(ue.getMessage());
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        doMethod(req, resp);
    }

    /*
     * (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        doMethod(req, resp);
    }

    /*
     * (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doPut(javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doPut(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        doMethod(req, resp);
    }

    /*
     * (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doDelete(javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doDelete(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        doMethod(req, resp);
    }

    private void doMethod(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        if (log.isDebugEnabled()) {
            log.debug("receiving request on behalf of " + epr.getFragmentIdentifier());
        }

        if (isSelfCallback) {

            // we're handling async replies from 'self', i.e. request originated
            // from same engine which is handling response. servlet is therefore
            // non authenticating. as a precaution, however, we insure that the
            // request originated from this server. note that during testing,
            // remote host value is always loopback ip address, so we test for
            // that first

            boolean isLocalHost = true;
            String remoteHost = req.getRemoteHost();
            if (!remoteHost.equals("127.0.0.1")) {
                if (!remoteHost.equals(myAddress)) {
                    if (!remoteHost.equals(myHost)) {
                        if (!remoteHost.equals("localhost")) {
                            isLocalHost = false;
                        }
                    }
                }
            }

            if (isLocalHost == false) {
                String message = "Failed to handle asynchronous response from self."
                    + " Expected remote host to be '127.0.0.1' or '"
                    + myAddress
                    + "' or '"
                    + myHost
                    + "' or 'localhost'.";
                log.error(message);
                throw new ServletException(message);
            }

        }
        
        try {
            ((HyperTextTransferProtocolRT)binding).receiveAction(new HTTPServerRequestImpl(req),
                    new HTTPServerResponseImpl(resp), direction, epr);
        } catch (Throwable th) {
            log.error("caught throwable " + th.toString());
            resp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }

    }

}
