/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.feature.ws.transport.vm;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Set;

import org.bluestemsoftware.specification.eoa.component.binding.Binding;
import org.bluestemsoftware.specification.eoa.component.engine.rt.BindingModule;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference;
import org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule;
import org.bluestemsoftware.specification.eoa.component.engine.rt.TransportModule;
import org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule.WSSpecification;
import org.bluestemsoftware.specification.eoa.component.policy.rt.ActionPolicy;
import org.bluestemsoftware.specification.eoa.component.policy.rt.UnsupportedPolicyException;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.binding.soap.SOAPVMTransportProtocol;
import org.bluestemsoftware.specification.eoa.ext.feature.FeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.addressing.WSA.WSA10;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.http.HTTPTransportFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.vm.VMTransportFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.vm.VMTransportFeatureException;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.vm.VMTransportModule;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.transport.vm.VMTransportProtocol;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.w3c.dom.Element;

public final class VMTransportFeatureImpl implements VMTransportFeature.Provider {

    private static final Log log = SystemContext.getContext().getSystem().getLog(VMTransportFeature.class);

    public static final String IMPL = VMTransportFeatureImpl.class.getName();

    private VMTransportFeature consumer;

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_getFeatureImpl()
     */
    public String spi_getFeatureImpl() {
        return IMPL;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_getPolicyVocabularyNamespaces()
     */
    public Set<String> spi_getPolicyVocabularyNamespaces() {
        Set<String> pvns = new HashSet<String>();
        pvns.add(VMTransportFeature.PVNS_10);
        return pvns;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_setConfiguration(org.w3c.dom.Element)
     */
    public void spi_setConfiguration(Element configuration) {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension.Consumer)
     */
    public void spi_setConsumer(Extension consumer) {
        this.consumer = (VMTransportFeature)consumer;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_init(org.w3c.dom.Element)
     */
    public void spi_init() throws HTTPTransportFeatureException {
        log.debug("init begin");
        log.debug("init end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.Feature$Provider#spi_destroy()
     */
    public void spi_destroy() {
        log.debug("destroy begin");
        log.debug("destroy end");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_createDefaultTransportModule(org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointReference)
     */
    public TransportModule spi_createDefaultTransportModule(EndpointReference epr) throws FeatureException {

        Binding binding = epr.getEndpoint().getBindingReference().getReferencedComponent();

        if (!(binding.getUnderlyingProtocol() instanceof VMTransportProtocol)) {
            throw new VMTransportFeatureException("Binding "
                    + binding.getName()
                    + " employed by referenced endpoint must implement underlying protocol "
                    + VMTransportProtocol.class
                    + ".");
        }

        URI temp = null;
        try {
            temp = new URI(epr.getEndpoint().getAddress());
        } catch (URISyntaxException ue) {
            throw new VMTransportFeatureException("Endpoint address "
                    + epr.getEndpoint().getAddress()
                    + " is malformed. "
                    + ue.getMessage());
        }

        if (!temp.getScheme().equals(SOAPVMTransportProtocol.SCHEME)) {
            throw new VMTransportFeatureException("Endpoint address "
                    + epr.getEndpoint().getAddress()
                    + " uses an unrecognized scheme. Expected '"
                    + SOAPVMTransportProtocol.SCHEME
                    + "'.");
        }

        // note that wsa policy is defined with policy subject endpoint,
        // meaning wsa binding module on all actions defined within
        // endpoint are identical

        EndpointActionReference ear = epr.getEndpointActionReferences().iterator().next();

        if (ear == null) {

            // the asynchronous response will fail at runtime with a
            // transport fault

            log.warn("unable to configure callback address. endpoint reference "
                    + epr.getFragmentIdentifier()
                    + " defines no actions.");

        } else {

            // because we are messaging protocol agnostic, we cannot retrieve
            // ws-addressing binding module by type (which is binding type
            // specific). but we can look for one that implements the core
            // ws-addressing spec, which is not protocol specific

            WSA wsa = null;
            for (BindingModule bindingModule : ear.getBindingModules()) {
                for (WSSpecification s : bindingModule.getSupportedVersions().values()) {
                    if (s instanceof WSA10) {
                        wsa = (WSA)s;
                        break;
                    }
                }
            }

            // currently we support only one version of wsa, but eoa spec
            // allows support for future versions of wsa spec

            if (wsa == null) {
                throw new IllegalStateException("Expected ws-addressing version " + WSA10.NAMESPACE_URI);
            }

            // note that locally hosted engines do not use a fault to epr,
            // i.e. normal and fault responses are returned to same epr

            String replyTo = null;
            WSSpecification.Policy wsaPolicy = wsa.getPolicy().iterator().next();
            if (((WSA10.Policy)wsaPolicy).isAnonymousResponses() == Boolean.FALSE) {
                replyTo = ((WSA10.Policy)wsaPolicy).getReplyToAddress();
            } else {
                if (((WSA10.Policy)wsaPolicy).isAnonymousResponses() == null) {
                    replyTo = ((WSA10.Policy)wsaPolicy).getReplyToAddress();
                }
            }

            if (replyTo == null) {
                throw new VMTransportFeatureException("VM Transport requires non-anonymous responses.");
            }
            
            if (epr.getParent().isMyService()) {

                // map address of referenced endpoint, i.e. my endpoint, to
                // epr which will be used by vm transport modules defined
                // on other engines to send request to me

                VMTransport.addEndpointReference(epr.getEndpoint().getAddress(), epr);
                
                // if a callback address is defined on my endpoint, i.e.
                // callback used to handle async replies in response to
                // invocation of self (engine invokes self), then map
                // replyTo as well
                
                if (replyTo != null) {
                    VMTransport.addEndpointReference(replyTo, epr);
                }

            } else {
                
                // map callback address used by partner service defined
                // on engine other than self

                VMTransport.addEndpointReference(replyTo, epr);

            }

        }

        VMTransportModule.Provider p = new VMTransportModuleImpl();
        return new VMTransportModule(p, consumer);

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_createTransportModule(org.bluestemsoftware.specification.eoa.component.engine.rt.EndpointActionReference,
     *      org.bluestemsoftware.specification.eoa.component.policy.rt.ActionPolicy,
     *      org.bluestemsoftware.specification.eoa.component.policy.rt.ActionPolicy)
     */
    public TransportModule spi_createTransportModule(EndpointActionReference ear, ActionPolicy privatePolicy, ActionPolicy publicPolicy) throws FeatureException, UnsupportedPolicyException {
        // note that we provide no endpoint specific configuration via policy. if
        // user adds policy with with our vocabulary, we silently ignore it and
        // return default module
        return spi_createDefaultTransportModule(ear.getParent().getParent());
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.system.server.TransportFeature$Provider#spi_destroyFeatureModule(org.bluestemsoftware.specification.eoa.component.engine.rt.FeatureModule)
     */
    public void spi_destroyFeatureModule(FeatureModule featureModule) {
    }

}
