/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.wsdl.wsdl11.xerces;

import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.xerces.dom.DOMMessageFormatter;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.ext.wsdl.wsdl11.BindingElement;
import org.bluestemsoftware.specification.eoa.ext.wsdl.wsdl11.BindingOperationElement;
import org.bluestemsoftware.specification.eoa.ext.wsdl.wsdl11.ext.SOAP11BindingElement;
import org.bluestemsoftware.specification.eoa.ext.wsdl.wsdl11.ext.SOAP12BindingElement;
import org.bluestemsoftware.specification.eoa.ext.wsdl.wsdl11.ext.SOAPBindingElement;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

public class BindingElementImpl extends AbstractWSDLElement implements BindingElement {

    private static final long serialVersionUID = 1L;

    private Boolean isValid;
    private SOAPBindingElement soapBindingElement;
    private Map<String, BindingOperationElement> operationElements = new HashMap<String, BindingOperationElement>();
    private boolean isRead;
    
    public BindingElementImpl(WSDL11DocumentImpl owner) {
        super(owner, NAME.getNamespaceURI(), NAME.getPrefix() + ":" + NAME.getLocalPart());
    }

    public BindingElementImpl() {
        super();
    }
    
    public boolean isRead() {
        return isRead;
    }

    public void setIsRead(boolean isRead) {
        this.isRead = isRead;
    }
    
    public void setIsValid(Boolean isValid) {
        this.isValid = isValid;
    }
    
    public Boolean isValid() {
        return isValid;
    }

    public void setName(String name) {
        setAttribute("name", name);
    }

    public String getName() {
        return getAttribute("name").equals("") ? null : getAttribute("name");
    }

    public void setType(ComponentName value) {
        if (value == null) {
            return;
        }
        String prefix = ((WSDL11DocumentImpl)getOwnerDocument()).addNamespace(value.getNamespaceURI());
        setAttribute("type", prefix + ":" + value.getLocalPart());
    }

    public ComponentName getType() {
        String temp = getAttribute("type");
        if (temp.equals("")) {
            return null;
        }
        return ComponentName.valueOf(this, temp);
    }

    public BindingOperationElement addOperationElement(BindingOperationElement operationElement) {
        if (operationElement.getName() == null) {
            throw new IllegalArgumentException("Required attribute 'name' not set.");
        }
        super.insertBefore(operationElement, null);
        operationElements.put(operationElement.getName(), operationElement);
        return operationElement;
    }

    public Map<String, BindingOperationElement> getOperationElements() {
        return operationElements;
    }
    
    public SOAPBindingElement setSOAPBindingElement(SOAPBindingElement soapBindingElement) {
        if (operationElements.size() > 0) {
            super.insertBefore(soapBindingElement, operationElements.values().iterator().next());
        } else {
            super.insertBefore(soapBindingElement, null);
        }
        this.soapBindingElement = soapBindingElement;
        return soapBindingElement;
    }
    
    public SOAPBindingElement getSOAPBindingElement() {
        return soapBindingElement;
    }

    @Override
    public Node appendChild(Node newChild) throws DOMException {
        if (newChild.getNodeType() == Node.ELEMENT_NODE) {
            String namespaceURI = newChild.getNamespaceURI();
            String localPart = newChild.getLocalName();
            QName elementName = new QName(namespaceURI, localPart);
            if (elementName.equals(BindingOperationElement.NAME)) {
                return addOperationElement((BindingOperationElement)newChild);
            }
            if (elementName.equals(SOAP11BindingElement.NAME)) {
                return setSOAPBindingElement((SOAPBindingElement)newChild);
            }
            if (elementName.equals(SOAP12BindingElement.NAME)) {
                return setSOAPBindingElement((SOAPBindingElement)newChild);
            }
        }
        return super.insertBefore(newChild, null);
    }

    @Override
    public Node removeChild(Node oldChild) throws DOMException {
        if (oldChild.getNodeType() == Node.ELEMENT_NODE) {
            String namespaceURI = oldChild.getNamespaceURI();
            String localPart = oldChild.getLocalName();
            QName elementName = new QName(namespaceURI, localPart);
            if (elementName.equals(BindingOperationElement.NAME)) {
                operationElements.remove(((BindingOperationElement)oldChild).getName());
            }
            if (elementName.equals(SOAP11BindingElement.NAME)) {
                soapBindingElement = null;
            }
            if (elementName.equals(SOAP12BindingElement.NAME)) {
                soapBindingElement = null;
            }
        }
        return super.removeChild(oldChild);
    }

    @Override
    public Node replaceChild(Node newChild, Node oldChild) throws DOMException {
        if (oldChild.getNodeType() == Node.ELEMENT_NODE) {
            String namespaceURI = oldChild.getNamespaceURI();
            String localPart = oldChild.getLocalName();
            QName elementName = new QName(namespaceURI, localPart);
            if (elementName.equals(BindingOperationElement.NAME) || elementName.equals(SOAP11BindingElement.NAME)|| elementName.equals(SOAP12BindingElement.NAME)) {
                throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR, DOMMessageFormatter.formatMessage(
                        DOMMessageFormatter.DOM_DOMAIN, "HIERARCHY_REQUEST_ERR", null));
            }
        }
        return super.replaceChild(newChild, oldChild);
    }

    @Override
    public Node insertBefore(Node newChild, Node refChild) throws DOMException {
        if (newChild.getNodeType() == Node.ELEMENT_NODE) {
            String namespaceURI = newChild.getNamespaceURI();
            String localPart = newChild.getLocalName();
            QName elementName = new QName(namespaceURI, localPart);
            if (elementName.equals(BindingOperationElement.NAME) || elementName.equals(SOAP11BindingElement.NAME)|| elementName.equals(SOAP12BindingElement.NAME)) {
                throw new DOMException(DOMException.HIERARCHY_REQUEST_ERR, DOMMessageFormatter.formatMessage(
                        DOMMessageFormatter.DOM_DOMAIN, "HIERARCHY_REQUEST_ERR", null));
            }
        }
        return super.insertBefore(newChild, refChild);
    }

}
