/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.wsdl.wsdl11.xerces.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.wsdl.WSDLException;
import org.bluestemsoftware.specification.eoa.ext.wsdl.wsdl11.WSDL11Document;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.ls.DOMImplementationLS;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Validates WSDL document instances. This class is threadsafe. *
 */
public class WSDLValidator {

    private static WSDLValidator singleton;
    private DOMValidator domValidator;

    public static WSDLValidator getInstance() throws WSDLException {
        if (singleton == null) {
            singleton = new WSDLValidator();
        }
        return singleton;
    }

    private WSDLValidator() throws WSDLException {

        ExtensionFactory ef = null;
        ef = SystemContext.getContext().getSystem().getWSDLFactory(WSDL11Document.TYPE);
        ExtensionFactoryContext factoryContext = ef.getFactoryContext();
        
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        
        DocumentBuilder builder = null;
        try {
            builder = factory.newDocumentBuilder();
            builder.setEntityResolver(factoryContext);
        } catch (ParserConfigurationException pe) {
            throw new WSDLException("Error creating validator. " + pe);
        }
        
        List<String> schemaLocations = new ArrayList<String>();
        schemaLocations.add(Constants.WSDL_SCHEMA_LOCATION);
        schemaLocations.add(Constants.WSOAP11_SCHEMA_LOCATION);
        schemaLocations.add(Constants.WSOAP12_SCHEMA_LOCATION);
        schemaLocations.add(Constants.EOA_WSDL_EXT_SCHEMA_LOCATION);

        Source[] sources = new Source[4];

        for (int i = 0 ; i < schemaLocations.size() ; i++) {
            
            String schemaLocation = schemaLocations.get(i);

            // use a dom source as input to schema factory. when a sax source is
            // used, xerces fails to set our entity resolver on sax parser as of
            // version 2.9.0

            Document document = null;
            try {
                Resource resource = factoryContext.getResource(schemaLocation);
                InputSource inputSource = new InputSource();
                inputSource.setByteStream(resource.getInputStream());
                inputSource.setSystemId(schemaLocation);
                document = builder.parse(inputSource);
                sources[i] = (new DOMSource(document, schemaLocation));
            } catch (Exception ex) {
                throw new WSDLException("Error creating validator. " + ex);
            }
            
        }

        javax.xml.validation.Schema grammar = null;
        try {
            javax.xml.validation.SchemaFactory schemaFactory = null;
            schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
            DOMImplementation di = builder.getDOMImplementation();
            DOMImplementationLS ls = (DOMImplementationLS)di.getFeature("LS", "3.0");
            schemaFactory.setResourceResolver(new LSResourceResolverImpl(ls, factoryContext));
            grammar = schemaFactory.newSchema(sources);
        } catch (Exception ex) {
            throw new WSDLException("Error creating validator. " + ex);
        }

        domValidator = new DOMValidator(grammar);

    }

    public void validate(DOMSource domSource) throws SAXException, IOException {
        String errors = domValidator.validate(domSource);
        if (errors != null) {
            throw new SAXException(errors);
        }
    }

}
