/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.idl.wsdl.wsdl11.xerces;

import java.net.URI;
import java.net.URISyntaxException;

import org.bluestemsoftware.open.eoa.ext.idl.wsdl.wsdl11.xerces.util.DocumentContext;
import org.bluestemsoftware.open.eoa.ext.idl.wsdl.wsdl11.xerces.util.WSDLParser;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature.MetadataType;
import org.bluestemsoftware.specification.eoa.ext.idl.wsdl.WSDLException;
import org.bluestemsoftware.specification.eoa.ext.idl.wsdl.WSDLLocation;
import org.bluestemsoftware.specification.eoa.ext.idl.wsdl.wsdl11.ImportElement;
import org.bluestemsoftware.specification.eoa.ext.idl.wsdl.wsdl11.WSDL11Document;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

public class ImportElementImpl extends AbstractWSDLElement implements ImportElement {

    private static final long serialVersionUID = 1L;
    private WSDL11Document document;
    private WSDLLocation wsdlLocation;

    public ImportElementImpl(WSDL11DocumentImpl owner) {
        super(owner, NAME.getNamespaceURI(), NAME.getPrefix() + ":" + NAME.getLocalPart());
    }

    public ImportElementImpl() {
        super();
    }

    public void setNamespace(String namespace) {
        setAttribute("namespace", namespace);
    }

    public String getNamespace() {
        return getAttribute("namespace").equals("") ? null : getAttribute("namespace");
    }

    public void setLocation(String location) {
        setAttribute("location", location);
    }

    public String getLocation() {
        return getAttribute("location").equals("") ? null : getAttribute("location");
    }

    public WSDLLocation getWsdlLocation() {
        return wsdlLocation;
    }

    public void setWsdlLocation(WSDLLocation wsdlLocation) {
        this.wsdlLocation = wsdlLocation;
    }

    public WSDL11Document getDocument() throws WSDLException {
        
        if (document != null) {
            return document;
        }
        
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        
        try {
            
            DocumentContext documentContext = ((WSDL11DocumentImpl)ownerDocument).getDocumentContext();
            ExtensionFactory factory = documentContext.getFactory();
            thread.setContextClassLoader(factory.getFactoryContext().getClassLoader());
            
            if (getLocation() == null) {
                throw new WSDLException("Error attempting to resolve imported document. Location undefined.");
            }

            String documentBase = ((WSDL11DocumentImpl)ownerDocument).getBaseURI();
            URI expandedLocation;
            try {
                expandedLocation = null;
                if (documentBase == null) {
                    expandedLocation = new URI(getLocation());
                } else {
                    expandedLocation = new URI(documentBase).resolve(getLocation());
                }
            } catch (URISyntaxException ue) {
                throw new WSDLException("Error resolving imported document." + ue);
            }

            EntityResolver er = ((WSDL11DocumentImpl)ownerDocument).getDocumentContext().getEntityResolver();
            
            // definition is defined remotely. if optional feature, ws-mex is enabled, we
            // run the request through it in case policy is resolved via soap request/response.
            // otherwise retrieve deployment's entity resolver

            Server server = SystemContext.getContext().getSystem().getServer();
            WSMexFeature metadataFeature = server.getFeature(WSMexFeature.class);
            
            InputSource inputSource = null;
            if (metadataFeature == null) {
                try {
                    inputSource = er.resolveEntity(documentBase, expandedLocation.toString());
                } catch (Exception ex) {
                    throw new WSDLException("Error attempting to resolve imported document at location '"
                            + expandedLocation
                            + "'. "
                            + ex);
                }
            } else {
                try {
                    inputSource = metadataFeature.getMetadata(MetadataType.WSDL, expandedLocation.toString());
                } catch (Exception ex) {
                    throw new WSDLException("Error attempting to resolve imported document at location '"
                            +  expandedLocation.toString()
                            + "'. "
                            + ex);
                }
            }
            
            try {
                WSDLParser wsdlParser = new WSDLParser(er);
                wsdlParser.parse(inputSource);
                document = (WSDL11DocumentImpl)wsdlParser.getDocument();
            } catch (Exception ex) {
                throw new WSDLException("Error attempting to resolve imported document at location '"
                        + inputSource.getSystemId()
                        + "'. "
                        + ex);
            }

            String expectedNamespace = getNamespace();

            if (!document.getDefinitionsElement().getTargetNamespace().equals(expectedNamespace)) {
                throw new WSDLException("Error attempting to resolve imported document at location '"
                        + inputSource.getSystemId()
                        + "'. Expected targetNamespace '"
                        + expectedNamespace
                        + "'. Found '"
                        + document.getDefinitionsElement().getTargetNamespace()
                        + "'.");
            }

            return document;

        } finally {
            thread.setContextClassLoader(cl);
        }


    }
    
}
