/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.xerces.util;

import java.io.IOException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.dom.DOMSource;

import org.bluestemsoftware.open.eoa.commons.util.DOMValidator;
import org.bluestemsoftware.specification.eoa.Resource;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.XMLSchemaException;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchema;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.ls.DOMImplementationLS;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Validates XML Schema instances. This class is threadsafe. * 
 */
public class SchemaValidator {

    private static SchemaValidator singleton;
    private DOMValidator domValidator;
    
    public static SchemaValidator getInstance() throws XMLSchemaException {
        if (singleton == null) {
            singleton = new SchemaValidator();
        }
        return singleton;
    }
    
    private SchemaValidator() throws XMLSchemaException {
        
        ExtensionFactory ef = null;
        ef = SystemContext.getContext().getSystem().getSchemaFactory(XMLSchema.TYPE);
        ExtensionFactoryContext factoryContext = ef.getFactoryContext();
        
        // use a dom source as input to schema factory. when a sax source is
        // used, xerces fails to set our entity resolver on sax parser as of
        // version 2.9.0

        Document document = null;
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            builder.setEntityResolver(factoryContext);
            Resource resource = factoryContext.getResource(Constants.XML_SCHEMA_SCHEMA_LOCATION);
            InputSource inputSource = new InputSource();
            inputSource.setByteStream(resource.getInputStream());
            inputSource.setSystemId(Constants.XML_SCHEMA_SCHEMA_LOCATION);
            document = builder.parse(inputSource);
        } catch (Exception ex) {
            throw new XMLSchemaException("Error creating validator. " + ex);
        }
        
        javax.xml.validation.Schema grammar = null;
        try {
            javax.xml.validation.SchemaFactory schemaFactory = null;
            schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
            DOMImplementation di = document.getImplementation();
            DOMImplementationLS ls = (DOMImplementationLS)di.getFeature("LS", "3.0");
            schemaFactory.setResourceResolver(new LSResourceResolverImpl(ls, factoryContext));
            DOMSource source = new DOMSource(document);
            source.setSystemId(Constants.XML_SCHEMA_SCHEMA_LOCATION);
            grammar = schemaFactory.newSchema(source);
        } catch (Exception ex) {
            throw new XMLSchemaException("Error creating validator. " + ex);
        }

        domValidator = new DOMValidator(grammar);

    }

    public void validate(DOMSource domSource) throws SAXException, IOException {
        String errors = domValidator.validate(domSource);
        if (errors != null) {
            throw new SAXException(errors);
        }
    }

}
