/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.xerces.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.namespace.QName;
import javax.xml.transform.dom.DOMSource;

import org.apache.xerces.parsers.DOMParser;
import org.apache.xerces.xni.Augmentations;
import org.apache.xerces.xni.NamespaceContext;
import org.apache.xerces.xni.XMLAttributes;
import org.apache.xerces.xni.XMLLocator;
import org.apache.xerces.xni.XNIException;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.XMLSchemaException;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * Parses an input source into a schema document. This class is NOT threadsafe, but is
 * reusable.
 */
public final class SchemaParser extends DOMParser implements ErrorHandler {

    private static Log log = SystemContext.getContext().getSystem().getLog(SchemaParser.class);

    private SchemaValidator validator = null;
    private ArrayList<String> errorMessages = new ArrayList<String>();

    public SchemaParser(EntityResolver entityResolver) throws XMLSchemaException {

        if (entityResolver == null) {
            throw new IllegalArgumentException("entityResolver null");
        }

        setEntityResolver(entityResolver);

        setErrorHandler(this);

        try {
            setFeature(Constants.NAMESPACES_FEATURE, true);
            setProperty(Constants.DOCUMENT_CLASS_NAME_PROPERTY, Constants.SCHEMA_DOCUMENT_CLASS_NAME);
        } catch (Exception ex) {
            throw new XMLSchemaException(ex);
        }

        this.validator = SchemaValidator.getInstance();

    }

    /**
     * Parses inputSource into a Schema document
     * 
     * @param inputSource
     * @throws SAXException
     * @throws IOException
     */
    @Override
    public void parse(InputSource inputSource) throws SAXException, IOException {

        errorMessages.clear();

        super.parse(inputSource);

        // validation is not enabled on super, but we need to check for
        // parsing errors

        if (errorMessages.size() > 0) {
            Iterator<String> messageIterator = errorMessages.iterator();
            StringBuilder errorMessage = new StringBuilder();
            while (messageIterator.hasNext()) {
                errorMessage.append(messageIterator.next() + "  ");
            }
            throw new SAXException(errorMessage.toString());
        }

        // we validate post-parse. if schema validation feature is enabled
        // on super pre-parse, we end up getting psvi on dom, i.e. default
        // values for attributes etc ...

        validator.validate(new DOMSource(getDocument()));

    }

    @Override
    public void startDocument(XMLLocator locator, String encoding, NamespaceContext namespaceContext, Augmentations augs) throws XNIException {
        super.startDocument(locator, encoding, namespaceContext, augs);
    }

    @Override
    public void startElement(org.apache.xerces.xni.QName element, XMLAttributes attributes, Augmentations augs) throws XNIException {
        if (log.isTraceEnabled()) {
            log.trace("start element: " + new QName(element.uri, element.localpart));
        }
        super.startElement(element, attributes, augs);
    }

    @Override
    public void endElement(org.apache.xerces.xni.QName element, Augmentations augs) throws XNIException {
        if (log.isTraceEnabled()) {
            log.trace("end element: " + new QName(element.uri, element.localpart));
        }
        super.endElement(element, augs);
    }

    /** ************************* ErrorHandler Impl ************************************* */

    public void error(SAXParseException spe) throws SAXException {
        errorMessages.add(spe.getMessage());
    }

    public void warning(SAXParseException spe) throws SAXException {
        errorMessages.add(spe.getMessage());
    }

    public void fatalError(SAXParseException spe) throws SAXException {
        errorMessages.add(spe.getMessage());
    }

}
