/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.xerces;

import java.io.ByteArrayInputStream;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamSource;

import org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.xerces.util.DOMSerializer;
import org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.xerces.util.SchemaParser;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier.Scheme;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentType;
import org.bluestemsoftware.specification.eoa.component.types.rt.SchemaDocument;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature;
import org.bluestemsoftware.specification.eoa.ext.feature.ws.mex.WSMexFeature.MetadataType;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.XMLSchemaException;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaFactory;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaReference;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.bluestemsoftware.specification.eoa.system.System.Log;
import org.bluestemsoftware.specification.eoa.system.server.Server;
import org.w3c.dom.Element;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

public final class XMLSchemaFactoryImpl implements XMLSchemaFactory.Provider {
    
    private static final Log log = SystemContext.getContext().getSystem().getLog(XMLSchemaFactory.class);

    public static final String NAME = "xml-schema-10-default";

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.ExtensionFactory$Provider#spi_getName()
     */
    public String spi_getName() {
        return NAME;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.schema.SchemaFactory$Provider#spi_getTargetNamespace(org.w3c.dom.Element)
     */
    public String spi_getTargetNamespace(Element schemaElement) {
        if (schemaElement == null) {
            throw new IllegalArgumentException("schemaElement null");
        }
        return schemaElement.getAttribute("targetNamespace");
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaFactory$Provider#spi_dereferenceLocation(org.xml.sax.EntityResolver, java.lang.String)
     */
    public InputSource spi_dereferenceLocation(EntityResolver entityResolver, String location) throws XMLSchemaException {
        
        // schema expression is defined remotely. if optional feature, ws-mex is
        // enabled, we run the request through it in case schema is resolved via
        // soap request/response. otherwise retrieve using entity resolver

        Server server = SystemContext.getContext().getSystem().getServer();
        WSMexFeature metadataFeature = server.getFeature(WSMexFeature.class);

        InputSource is = null;
        if (metadataFeature == null) {
            try {
                is = entityResolver.resolveEntity(null, location);
            } catch (Exception ex) {
                throw new XMLSchemaException("Error dereferencing schema from URI '" + location + "'. " + ex);
            }
        } else {
            try {
                is = metadataFeature.getMetadata(MetadataType.SCHEMA, location);
            } catch (Exception ex) {
                throw new XMLSchemaException("Error dereferencing schema from URI '" + location + "'. " + ex);
            }
        }

        return is;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaFactory$Provider#spi_readSchemas(org.xml.sax.EntityResolver,
     *      javax.xml.transform.Source)
     */
    public Set<SchemaDocument> spi_readSchemas(EntityResolver entityResolver, Source source) throws XMLSchemaException {
        log.debug("spi_readSchema begin");
        
        if (source == null) {
            XMLSchemaImpl provider = new XMLSchemaImpl();
            Set<SchemaDocument> result = new HashSet<SchemaDocument>();
            result.add(provider);
            return result;
        }

        Map<String, XMLSchemaImpl> workspace = new HashMap<String, XMLSchemaImpl>();

        InputSource is = null;

        if (source instanceof SAXSource) {
            is = ((SAXSource)source).getInputSource();
        } else if (source instanceof StreamSource) {
            is = new InputSource(source.getSystemId());
            is.setByteStream(((StreamSource)source).getInputStream());
        } else if (source instanceof DOMSource) {
            try {
                StringWriter sw = new StringWriter();
                DOMSerializer.serializeNode(((DOMSource)source).getNode(), sw, "UTF-8", false);
                is = new InputSource(source.getSystemId());
                is.setByteStream(new ByteArrayInputStream(sw.toString().getBytes("UTF-8")));
            } catch (Exception ex) {
                throw new XMLSchemaException(ex);
            }
        }

        readSchema(workspace, entityResolver, is);

        log.debug("spi_readSchema end");
        return new HashSet<SchemaDocument>(workspace.values());

    }

    private XMLSchemaImpl readSchema(Map<String, XMLSchemaImpl> workspace, EntityResolver entityResolver, InputSource inputSource) throws XMLSchemaException {

        // check to see if we've already resolved the requested location. if
        // so return it, otherwise make sure the reference isn't circular

        if (workspace.containsKey(inputSource.getSystemId())) {
            if (workspace.get(inputSource.getSystemId()) == null) {
                throw new XMLSchemaException("Circular ref detected while reading"
                        + " referenced schema at location '"
                        + inputSource.getSystemId()
                        + "'.");
            } else {
                return workspace.get(inputSource.getSystemId());
            }
        } else {
            workspace.put(inputSource.getSystemId(), null);
        }

        // parse the input source into a schema document which will serve as the
        // underlying service provider

        XMLSchemaImpl schemaImpl = null;
        try {
            SchemaParser schemaParser = new SchemaParser(entityResolver);
            schemaParser.parse(inputSource);
            schemaImpl = (XMLSchemaImpl)schemaParser.getDocument();
        } catch (Exception ex) {
            throw new XMLSchemaException("Error reading schema. " + ex);
        }

        String targetNamespace = schemaImpl.getTargetNamespace();

        // iterate over schema references and attempt to resolve each which may result in
        // recursive invocations of this method

        for (XMLSchemaReference schemaReference : schemaImpl.getSchemaReferences()) {

            String referencedNamespace = null;

            if (schemaReference.getLocalName().equals("include")) {
                referencedNamespace = targetNamespace;
            }

            if (schemaReference.getLocalName().equals("import")) {
                referencedNamespace = schemaReference.getAttribute("namespace");
            }

            String systemID = schemaReference.getSchemaLocation();

            // if schema location is undefined, then this is a shortcut for
            // importing a schema with tns = namespace and no id. set to
            // pound sign for special treatment below

            if (systemID.equals("")) {
                systemID = "#";
            }
            
            // if location begins with a fragment, this is fragment id of
            // referenced schema, i.e. do not resolve against base uri

            if (systemID.startsWith("#")) {

                // decode fragment to remove any escaped octets before attempting to parse
                // into a fragmentid

                String fragment;
                try {
                    fragment = new URI(systemID).getFragment();
                } catch (URISyntaxException ue) {
                    throw new XMLSchemaException("Error resolving referenced schema at location '"
                            + systemID
                            + "'. "
                            + ue);
                }

                // location is a fragment, i.e. schema was sourced from a deployment and base
                // uri is empty string, in which case value identifies a fragment within the
                // eoa component model. actual reference is resolved during deployment

                FragmentIdentifier fid = null;
                try {
                    // attempt to parse fragment into a fragment id, i.e. if value fully
                    // qualified
                    fid = FragmentIdentifier.valueOf(fragment);
                } catch (IllegalArgumentException ie) {
                    // we assume fragment is just schema id and remainder of fragmentid is
                    // implied
                    fid = new FragmentIdentifier(referencedNamespace, Scheme.EOA, "schema", fragment);
                }

                ((XMLSchemaReferenceImpl)schemaReference).setSchemaFragmentIdentifier(fid);

            } else {
                
                String location = null;
                try {
                    String baseURI = schemaImpl.getDocumentURI();
                    location = baseURI == null ? systemID : new URI(baseURI).resolve(systemID).toString();
                } catch (URISyntaxException ue) {
                    throw new XMLSchemaException("Error resolving referenced schema at location '"
                            + systemID
                            + "' within context of base document location '"
                            + schemaImpl.getDocumentURI()
                            + "'. "
                            + ue.getMessage());
                }

                // schema is defined remotely. if optional feature, ws-mex is enabled,
                // we run the request through it in case document is resolved via soap
                // request/response. otherwise retreive using deployment

                Server server = SystemContext.getContext().getSystem().getServer();
                WSMexFeature metadataFeature = server.getFeature(WSMexFeature.class);

                InputSource is = null;
                if (metadataFeature == null) {
                    try {
                        is = entityResolver.resolveEntity(null, location);
                    } catch (Exception ex) {
                        throw new XMLSchemaException("Error resolving referenced schema at location '"
                                + location
                                + "'. "
                                + ex);
                    }
                } else {
                    try {
                        is = metadataFeature.getMetadata(MetadataType.SCHEMA, location);
                    } catch (Exception ex) {
                        throw new XMLSchemaException("Error resolving referenced schema at location '"
                                + location
                                + "'. "
                                + ex);
                    }
                }

                // using recursion, read the referenced schema and set fragmentid on
                // schema reference element

                XMLSchemaImpl ref = readSchema(workspace, entityResolver, is);
                String tns = ref.getTargetNamespace();
                String id = ref.getID();
                QName refName = new QName(tns, id);
                FragmentIdentifier refFID = FragmentIdentifier.valueOf(ComponentType.SCHEMA, refName);
                ((XMLSchemaReferenceImpl)schemaReference).setSchemaFragmentIdentifier(refFID);

            }

        }

        // update entry in workspace with completed schema, which may be returned
        // when asked to resolve an already resolved location

        workspace.put(inputSource.getSystemId(), schemaImpl);

        return schemaImpl;

    }

}
