/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.xerces;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXResult;
import javax.xml.validation.Validator;
import javax.xml.validation.ValidatorHandler;

import org.apache.xerces.dom.DocumentImpl;
import org.apache.xerces.jaxp.validation.XSGrammarPoolContainer;
import org.apache.xerces.xni.grammars.XMLGrammarPool;
import org.apache.xerces.xni.grammars.XSGrammar;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSModelGroup;
import org.apache.xerces.xs.XSObject;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSParticle;
import org.apache.xerces.xs.XSTerm;
import org.bluestemsoftware.open.eoa.commons.util.QNameUtils;
import org.bluestemsoftware.open.eoa.ext.schema.xs.xs10.xerces.util.LSResourceResolverImpl;
import org.bluestemsoftware.specification.eoa.DeploymentException;
import org.bluestemsoftware.specification.eoa.component.ComponentContext;
import org.bluestemsoftware.specification.eoa.component.ComponentReference;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier;
import org.bluestemsoftware.specification.eoa.component.ComponentDeployment.CurrentReference;
import org.bluestemsoftware.specification.eoa.component.FragmentIdentifier.Scheme;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentName;
import org.bluestemsoftware.specification.eoa.component.RootComponent.ComponentType;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceAction.Direction;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceMessageReference.ContentModel;
import org.bluestemsoftware.specification.eoa.component.intrface.InterfaceOperation.Style;
import org.bluestemsoftware.specification.eoa.component.message.MessagePart;
import org.bluestemsoftware.specification.eoa.component.types.Schema;
import org.bluestemsoftware.specification.eoa.component.types.rt.SchemaException;
import org.bluestemsoftware.specification.eoa.ext.Extension;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactory;
import org.bluestemsoftware.specification.eoa.ext.ExtensionFactoryContext;
import org.bluestemsoftware.specification.eoa.ext.intrface.IRIStyle;
import org.bluestemsoftware.specification.eoa.ext.intrface.MultipartStyle;
import org.bluestemsoftware.specification.eoa.ext.intrface.RPCStyle;
import org.bluestemsoftware.specification.eoa.ext.intrface.IRIStyle.ChildElement;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.XMLSchemaException;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.XMLSchemaValidationException;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchema;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaElement;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaImport;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaInclude;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XMLSchemaReference;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XSComplexTypeDefinition;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XSElementDeclaration;
import org.bluestemsoftware.specification.eoa.ext.schema.xs.xs10.XSSimpleTypeDefinition;
import org.bluestemsoftware.specification.eoa.system.SystemContext;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.ls.DOMImplementationLS;
import org.xml.sax.ContentHandler;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.LexicalHandler;

public final class XMLSchemaImpl extends DocumentImpl implements XMLSchema, XMLSchema.Provider {

    private static final long serialVersionUID = 1L;

    private javax.xml.validation.Schema grammar;
    private FragmentIdentifier fragmentIdentifier;
    private ComponentName componentName;
    private ComponentContext componentContext;
    private Boolean isDeployed = Boolean.FALSE;
    private TransformerFactory tf;

    public XMLSchemaImpl() {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.RootExtension#getExtensionType()
     */
    public final String getExtensionType() {
        return XMLSchema.TYPE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension#getExtensionFactory()
     */
    public final ExtensionFactory getExtensionFactory() {
        return SystemContext.getContext().getSystem().getSchemaFactory(XMLSchema.TYPE);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getName()
     */
    public ComponentName getName() {
        if (componentName == null) {
            String tns = ((XMLSchemaElement)getDocumentElement()).getTargetNamespace();
            String id = ((XMLSchemaElement)getDocumentElement()).getID();
            componentName = new ComponentName(tns, id);
        }
        return componentName;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getComponentType()
     */
    public ComponentType getComponentType() {
        return ComponentType.SCHEMA;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getComponentContext()
     */
    public ComponentContext getComponentContext() {
        return componentContext;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#getRuntimeProvidable()
     */
    public Schema getRuntimeProvidable() {
        return (Schema)getComponentContext().getDeployment().getComponent(ComponentType.SCHEMA, componentName);
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.types.rt.SchemaDocument#getTargetNamespace()
     */
    public String getTargetNamespace() {
        return ((XMLSchemaElement)getDocumentElement()).getTargetNamespace();
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension#getExtensionProvider()
     */
    public org.bluestemsoftware.specification.eoa.ext.Extension.Provider getExtensionProvider() {
        return this;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#deploy(org.bluestemsoftware.specification.eoa.component.ComponentContext)
     */
    public synchronized void deploy(ComponentContext componentContext) throws DeploymentException {
        if (isDeployed == null) {
            throw new DeploymentException("Runtime component "
                    + getFragmentIdentifier()
                    + " is circularly referenced by "
                    + CurrentReference.getCurrentReference());
        }
        if (isDeployed) {
            return;
        }
        isDeployed = null;
        this.componentContext = componentContext;
        // force referenced runtime schema providers to deploy. when runtime message defs are
        // deployed, they requires schemas to validate themselves, all of which must be deployed
        for (XMLSchemaReference xsr : getSchemaReferences()) {
            FragmentIdentifier fid = xsr.getSchemaFragmentIdentifier();
            String tns = fid.getRootComponentNamespace();
            String lname = fid.getPointerPart();
            ComponentName schemaName = new ComponentName(tns, lname);
            ComponentReference<Schema> cr = new SchemaReferenceInternal(schemaName);
            componentContext.findReferencedComponent(cr);
        }
        isDeployed = Boolean.TRUE;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.RootComponentRT#isDeployed()
     */
    public boolean isDeployed() {
        return isDeployed;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.rt.ComponentRT#getFragmentIdentifier()
     */
    public FragmentIdentifier getFragmentIdentifier() {
        if (fragmentIdentifier == null) {
            String tns = ((XMLSchemaElement)getDocumentElement()).getTargetNamespace();
            String id = ((XMLSchemaElement)getDocumentElement()).getID();
            fragmentIdentifier = new FragmentIdentifier(tns, Scheme.EOA, "schemaDocument", id);
        }
        return fragmentIdentifier;
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.ext.Extension$Provider#spi_setConsumer(org.bluestemsoftware.specification.eoa.ext.Extension)
     */
    public void spi_setConsumer(Extension consumer) {
    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.types.rt.SchemaDocument#clone(java.lang.String)
     */
    public synchronized Document clone(String documentURI) throws XMLSchemaException {

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();

        try {

            thread.setContextClassLoader(getExtensionFactory().getFactoryContext().getClassLoader());

            URI base = new URI(documentURI);
            String scheme = base.getScheme();
            String ui = base.getUserInfo();
            String host = base.getHost();
            int port = base.getPort();
            String path = base.getPath();

            synchronized (this) {

                // TODO: this isn't cheap. should we cache the result or can we
                // assume caller is caching the result?

                XMLSchemaImpl clone = (XMLSchemaImpl)this.cloneNode(true);
                clone.setDocumentURI(documentURI);

                // fragment identifier is not cloned, so we must retrieve from our
                // collection using hashcode as key

                XMLSchemaElement schemaElement = (XMLSchemaElement)clone.getDocumentElement();
                for (XMLSchemaReference schemaReference : schemaElement.getSchemaReferences()) {
                    Integer hashcode = new Integer(schemaReference.hashCode());
                    XMLSchemaReference temp = getSchemaReference(hashcode);
                    FragmentIdentifier fid = temp.getSchemaFragmentIdentifier();
                    URI location = new URI(scheme, ui, host, port, path, null, fid.toString());
                    ((XMLSchemaReferenceImpl)schemaReference).setSchemaLocation(location.toString());
                }

                clone.setReadOnly(true, true);

                return clone;

            }

        } catch (Exception ex) {
            throw new XMLSchemaException("Error creating document. " + ex);
        } finally {
            thread.setContextClassLoader(cl);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.types.rt.SchemaDocument#validate(org.w3c.dom.Element)
     */
    public void validate(Element content) throws XMLSchemaException, XMLSchemaValidationException {

        if (content == null) {
            throw new IllegalArgumentException("content null");
        }

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {

            thread.setContextClassLoader(getExtensionFactory().getFactoryContext().getClassLoader());

            final List<String> errorMessages = new ArrayList<String>();

            Validator validator = getGrammar().newValidator();
            validator.setErrorHandler(new ErrorHandler() {

                public void error(SAXParseException spe) throws SAXException {
                    errorMessages.add(spe.getMessage());
                }

                public void warning(SAXParseException spe) throws SAXException {
                    errorMessages.add(spe.getMessage());
                }

                public void fatalError(SAXParseException spe) throws SAXException {
                    errorMessages.add(spe.getMessage());
                }

            });

            try {
                validator.validate(new DOMSource(content));
            } catch (Exception ex) {
                throw new XMLSchemaException("Unable to validate message. " + ex);
            }

            if (errorMessages.size() > 0) {
                Iterator<String> messageIterator = errorMessages.iterator();
                StringBuilder errorMessage = new StringBuilder();
                while (messageIterator.hasNext()) {
                    errorMessage.append(messageIterator.next() + "  ");
                }
                throw new XMLSchemaValidationException(errorMessage.toString());
            }

        } finally {
            thread.setContextClassLoader(cl);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.bluestemsoftware.specification.eoa.component.types.rt.SchemaDocument#validate(org.w3c.dom.Element,
     *      org.xml.sax.ContentHandler, org.xml.sax.ext.LexicalHandler,
     *      org.xml.sax.ErrorHandler)
     */
    public void validate(Element content, ContentHandler contentHandler, LexicalHandler lexicalHandler, ErrorHandler errorHandler) throws SchemaException {

        if (content == null) {
            throw new IllegalArgumentException("content null");
        }

        if (contentHandler == null) {
            throw new IllegalArgumentException("contentHandler null");
        }

        if (lexicalHandler == null) {
            throw new IllegalArgumentException("lexicalHandler null");
        }

        if (errorHandler == null) {
            throw new IllegalArgumentException("errorHandler null");
        }

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {

            thread.setContextClassLoader(getExtensionFactory().getFactoryContext().getClassLoader());

            if (tf == null) {
                tf = TransformerFactory.newInstance();
            }

            // validator handler uses grammar to augment sax events forwarded
            // to caller's content handler with psvi info, e.g. default values
            // ignorable whitespace, etc ...

            ValidatorHandler vh = getGrammar().newValidatorHandler();
            vh.setErrorHandler(errorHandler);
            vh.setContentHandler(contentHandler);
            try {
                Transformer transformer = tf.newTransformer();
                SAXResult result = new SAXResult(vh);
                result.setLexicalHandler(lexicalHandler);
                transformer.transform(new DOMSource(content), result);
            } catch (Exception ex) {
                throw new XMLSchemaException(ex.getMessage());
            }

        } finally {
            thread.setContextClassLoader(cl);
        }

    }

    /*
     * (non-Javadoc) Overrides ancestor such that we can intercept invocations by DOMParser and
     * return a schema element if qname matches an expected name.
     */
    @Override
    public Element createElementNS(String namespaceURI, String qualifiedName, String localpart) throws DOMException {
        QName elementName = new QName(namespaceURI, localpart);
        if (elementName.equals(XMLSchemaElement.NAME)) {
            return new XMLSchemaElementImpl(this);
        }
        if (elementName.equals(XMLSchemaImport.NAME)) {
            return new XMLSchemaImportImpl(this);
        }
        if (elementName.equals(XMLSchemaInclude.NAME)) {
            return new XMLSchemaIncludeImpl(this);
        }
        if (elementName.equals(XSElementDeclaration.NAME)) {
            return new XSElementDeclarationImpl(this);
        }
        if (elementName.equals(XSComplexTypeDefinition.NAME)) {
            return new XSComplexTypeDefinitionImpl(this);
        }
        if (elementName.equals(XSSimpleTypeDefinition.NAME)) {
            return new XSSimpleTypeDefinitionImpl(this);
        }
        return super.createElementNS(namespaceURI, qualifiedName, localpart);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.xerces.dom.CoreDocumentImpl#createElementNS(java.lang.String,
     *      java.lang.String)
     */
    @Override
    public Element createElementNS(String namespaceURI, String qualifiedName) throws DOMException {
        String localPart = QNameUtils.getLocalPart(qualifiedName);
        return createElementNS(namespaceURI, qualifiedName, localPart);
    }

    /*
     * (non-Javadoc)
     * @see org.apache.xerces.dom.DocumentImpl#cloneNode(boolean)
     */
    @Override
    public Node cloneNode(boolean deep) {
        XMLSchemaImpl clone = new XMLSchemaImpl();
        super.cloneNode(clone, true);
        return clone;
    }

    @Override
    public void normalizeDocument() {
        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(getExtensionFactory().getFactoryContext().getClassLoader());
            super.normalizeDocument();
        } finally {
            thread.setContextClassLoader(cl);
        }
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public final boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        XMLSchemaImpl that = null;
        if (!(obj instanceof XMLSchemaImpl)) {
            return false;
        } else {
            that = (XMLSchemaImpl)obj;
        }
        this.normalizeDocument();
        that.normalizeDocument();
        return this.isEqualNode(that);

    }

    String getID() {
        return ((XMLSchemaElementImpl)getDocumentElement()).getID();
    }

    Collection<XMLSchemaReference> getSchemaReferences() {
        return ((XMLSchemaElementImpl)getDocumentElement()).getSchemaReferenceElements();
    }

    private XMLSchemaReference getSchemaReference(Integer hashCode) {
        return ((XMLSchemaElementImpl)getDocumentElement()).schemaReferenceElements.get(hashCode);
    }

    private javax.xml.validation.Schema getGrammar() throws XMLSchemaException {

        if (grammar == null) {

            javax.xml.validation.SchemaFactory schemaFactory = null;
            try {
                schemaFactory = new org.apache.xerces.jaxp.validation.XMLSchemaFactory();
                DOMImplementation di = getImplementation();
                DOMImplementationLS ls = (DOMImplementationLS)di.getFeature("LS", "3.0");
                ExtensionFactoryContext factoryContext = getExtensionFactory().getFactoryContext();
                schemaFactory.setResourceResolver(new LSResourceResolverImpl(ls, factoryContext));
            } catch (IllegalArgumentException ie) {
                // javadoc indicates illegal argument exception is thrown if no provider
                throw new UnsupportedOperationException("Schema validation not supported");
            }

            // set documentURI to empty string, which will be used as base
            // uri in to document method, against which schema fragmentid.
            // is resolved, i.e. resulting uri will be #fragmentid. our
            // resource resolver will resolve all urls which start with #

            try {
                grammar = schemaFactory.newSchema(new DOMSource(clone("")));
            } catch (SAXException se) {
                throw new XMLSchemaException("Error creating javax Schema. " + se);
            }

        }

        return grammar;

    }

    public Map<String, Style> modelStyles(InterfaceMessageReference intefaceMessageReference) throws SchemaException {

        Thread thread = Thread.currentThread();
        ClassLoader cl = thread.getContextClassLoader();
        try {

            thread.setContextClassLoader(getExtensionFactory().getFactoryContext().getClassLoader());

            InterfaceOperation interfaceOperation = intefaceMessageReference.getParent();
            String operationName = interfaceOperation.getName().getLocalPart();
            Set<String> styleIRIS = interfaceOperation.getStyles();
            styleIRIS.addAll(interfaceOperation.getParent().getStyleDefaults());

            XMLGrammarPool p = ((XSGrammarPoolContainer)getGrammar()).getGrammarPool();
            XSModel xsModel = ((XSGrammar)p.retrieveInitialGrammarSet(XMLSchema.TYPE)[0]).toXSModel();

            Map<String, Style> styles = new HashMap<String, Style>();
            for (String iri : styleIRIS) {
                Style style = null;
                if (iri.equals(IRIStyle.STYLE_IRI)) {
                    style = modelIRIStyle(operationName, intefaceMessageReference, xsModel);
                } else if (iri.equals(RPCStyle.STYLE_IRI)) {
                    style = modelRPCStyle(intefaceMessageReference, xsModel);
                } else if (iri.equals(MultipartStyle.STYLE_IRI)) {
                    style = modelMultipartStyle(operationName, intefaceMessageReference, xsModel);
                } else {
                    throw new XMLSchemaException("Style " + iri + " not understood.");
                }
                if (style != null) {
                    styles.put(style.getStyleIRI(), style);
                }
            }

            return styles;
        } finally {
            thread.setContextClassLoader(cl);
        }

    }

    /*
     * Models input message references only as required by WSDL 2.0 spec.
     */
    private Style modelIRIStyle(String operationName, InterfaceMessageReference imr, XSModel xsModel) throws XMLSchemaException {
        if (imr.getDirection() != Direction.IN) {
            return null;
        }
        if (imr.getContentModel() != ContentModel.ELEMENT) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. Style requires content model #element.");
        }
        MessagePart payloadPart = imr.getReferencedComponent().getPart(MessagePart.PAYLOAD);
        String componentName = payloadPart.getSchemaComponentName();
        if (operationName.equals(componentName) == false) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. Style requires payload typed as an element"
                    + " with local name '"
                    + operationName
                    + "'.");
        }
        org.apache.xerces.xs.XSElementDeclaration xse = null;
        xse = xsModel.getElementDeclaration(componentName, getTargetNamespace());
        if (xse == null) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is undefined.");
        }
        org.apache.xerces.xs.XSTypeDefinition xst = xse.getTypeDefinition();
        if (xst == null || xst.getTypeCategory() != org.apache.xerces.xs.XSTypeDefinition.COMPLEX_TYPE) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type.");
        }
        org.apache.xerces.xs.XSComplexTypeDefinition xsc = null;
        xsc = (org.apache.xerces.xs.XSComplexTypeDefinition)xst;
        if (xsc.getAttributeUses().getLength() > 0) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. The complex type that defines the body of the element"
                    + " must not contain any attributes.");
        }
        if (xsc.getContentType() != org.apache.xerces.xs.XSComplexTypeDefinition.CONTENTTYPE_ELEMENT) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. The complex type that defines the body of the element"
                    + " must contain only element declarations.");
        }
        XSParticle xsp = xsc.getParticle();
        if (xsp == null) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence.");
        }
        if (xsp.getMinOccurs() != 1 || xsp.getMaxOccursUnbounded() || xsp.getMaxOccurs() > 1) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence with no occurrence constraints.");
        }
        XSTerm term = xsp.getTerm();
        if (term.getType() != XSConstants.MODEL_GROUP) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence.");
        }
        XSModelGroup xsm = (XSModelGroup)term;
        if (xsm.getCompositor() != XSModelGroup.COMPOSITOR_SEQUENCE) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + IRIStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence.");
        }
        List<ChildElement> childElements = new ArrayList<ChildElement>();
        XSObjectList objectList = xsm.getParticles();
        for (int i = 0; i < objectList.getLength(); i++) {
            XSObject xso = (XSObject)objectList.item(i);
            if (xso.getType() != XSConstants.PARTICLE) {
                throw new XMLSchemaException("InputMessageReference "
                        + imr.getFragmentIdentifier()
                        + " is not consistent with style "
                        + IRIStyle.STYLE_IRI
                        + " defined on parent operation/interface. GED for payload type "
                        + new QName(getTargetNamespace(), componentName)
                        + " is invalid. Content model must be defined using a complex type"
                        + " that contains a sequence which contains only local element decls.");
            }
            xsp = (XSParticle)xso;
            term = xsp.getTerm();
            if (term.getType() != XSConstants.ELEMENT_DECLARATION) {
                throw new XMLSchemaException("InputMessageReference "
                        + imr.getFragmentIdentifier()
                        + " is not consistent with style "
                        + IRIStyle.STYLE_IRI
                        + " defined on parent operation/interface. GED for payload type "
                        + new QName(getTargetNamespace(), componentName)
                        + " is invalid. Content model must be defined using a complex type"
                        + " that contains a sequence which contains only local element decls.");
            }
            org.apache.xerces.xs.XSElementDeclaration childDecl = null;
            childDecl = (org.apache.xerces.xs.XSElementDeclaration)term;
            xst = childDecl.getTypeDefinition();
            if (xst == null || xst.getTypeCategory() != org.apache.xerces.xs.XSTypeDefinition.SIMPLE_TYPE) {
                throw new XMLSchemaException("InputMessageReference "
                        + imr.getFragmentIdentifier()
                        + " is not consistent with style "
                        + IRIStyle.STYLE_IRI
                        + " defined on parent operation/interface. GED for payload type "
                        + new QName(getTargetNamespace(), componentName)
                        + " is invalid. Content model must be defined using a complex type"
                        + " that contains a sequence which contains only local element decls"
                        + " derived from xs:simpleType");
            }
            // note that derivation method is not used, i.e. call is valid for all methods
            boolean invalidExtension = false;
            if (xst.derivedFrom(XMLSchema.TYPE, "QName", XSConstants.DERIVATION_EXTENSION)) {
                invalidExtension = true;
            }
            if (xst.derivedFrom(XMLSchema.TYPE, "NOTATION", XSConstants.DERIVATION_EXTENSION)) {
                invalidExtension = true;
            }
            if (xst.derivedFrom(XMLSchema.TYPE, "base64Binary", XSConstants.DERIVATION_EXTENSION)) {
                invalidExtension = true;
            }
            if (xst.derivedFrom(XMLSchema.TYPE, "hexBinary ", XSConstants.DERIVATION_EXTENSION)) {
                invalidExtension = true;
            }
            if (invalidExtension) {
                throw new XMLSchemaException("InputMessageReference "
                        + imr.getFragmentIdentifier()
                        + " is not consistent with style "
                        + IRIStyle.STYLE_IRI
                        + " defined on parent operation/interface. GED for payload type "
                        + new QName(getTargetNamespace(), componentName)
                        + " is invalid. Content model must be defined using a complex type"
                        + " that contains a sequence which contains only local element decls"
                        + " derived from xs:simpleType not derived from"
                        + " xs:QName, xs:NOTATION, xs:hexBinary or xs:base64Binary.");
            }
            String name = childDecl.getName();
            boolean isNillable = childDecl.getNillable();
            int minOccurs = xsp.getMinOccurs();
            String defaultValue = childDecl.getConstraintValue();
            ChildElement childElement = new ChildElement(name, isNillable, minOccurs, defaultValue);
            childElements.add(childElement);
        }
        return new IRIStyle(new QName(getTargetNamespace(), operationName), childElements);
    }

    private Style modelRPCStyle(InterfaceAction interfaceAction, XSModel xsModel) throws XMLSchemaException {

        // TODO: implement. for now just create an empty model, which binding
        // doesn't use anyway, i.e. application is the only component that cares
        // about structure (i think)

        return new RPCStyle();

    }

    private Style modelMultipartStyle(String operationName, InterfaceMessageReference imr, XSModel xsModel) throws XMLSchemaException {
        if (imr.getDirection() != Direction.IN) {
            return null;
        }
        if (imr.getContentModel() != ContentModel.ELEMENT) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. Style requires content model #element.");
        }
        MessagePart payloadPart = imr.getReferencedComponent().getPart(MessagePart.PAYLOAD);
        String componentName = payloadPart.getSchemaComponentName();
        if (operationName.equals(componentName) == false) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. Style requires payload typed as an element"
                    + " with local name '"
                    + operationName
                    + "'.");
        }
        org.apache.xerces.xs.XSElementDeclaration xse = null;
        xse = xsModel.getElementDeclaration(componentName, getTargetNamespace());
        if (xse == null) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is undefined.");
        }
        org.apache.xerces.xs.XSTypeDefinition xst = xse.getTypeDefinition();
        if (xst == null || xst.getTypeCategory() != org.apache.xerces.xs.XSTypeDefinition.COMPLEX_TYPE) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type.");
        }
        org.apache.xerces.xs.XSComplexTypeDefinition xsc = null;
        xsc = (org.apache.xerces.xs.XSComplexTypeDefinition)xst;
        if (xsc.getAttributeUses().getLength() > 0) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. The complex type that defines the body of the element"
                    + " must not contain any attributes.");
        }
        if (xsc.getContentType() != org.apache.xerces.xs.XSComplexTypeDefinition.CONTENTTYPE_ELEMENT) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. The complex type that defines the body of the element"
                    + " must contain only element declarations.");
        }
        XSParticle xsp = xsc.getParticle();
        if (xsp == null) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence.");
        }
        if (xsp.getMinOccurs() != 1 || xsp.getMaxOccursUnbounded() || xsp.getMaxOccurs() > 1) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence with no occurrence constraints.");
        }
        XSTerm term = xsp.getTerm();
        if (term.getType() != XSConstants.MODEL_GROUP) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence.");
        }
        XSModelGroup xsm = (XSModelGroup)term;
        if (xsm.getCompositor() != XSModelGroup.COMPOSITOR_SEQUENCE) {
            throw new XMLSchemaException("InputMessageReference "
                    + imr.getFragmentIdentifier()
                    + " is not consistent with style "
                    + MultipartStyle.STYLE_IRI
                    + " defined on parent operation/interface. GED for payload type "
                    + new QName(getTargetNamespace(), componentName)
                    + " is invalid. Content model must be defined using a complex type"
                    + " that contains a sequence.");
        }
        List<org.bluestemsoftware.specification.eoa.ext.intrface.MultipartStyle.ChildElement> childElements = null;
        childElements = new ArrayList<org.bluestemsoftware.specification.eoa.ext.intrface.MultipartStyle.ChildElement>();
        XSObjectList objectList = xsm.getParticles();
        for (int i = 0; i < objectList.getLength(); i++) {
            XSObject xso = (XSObject)objectList.item(i);
            if (xso.getType() != XSConstants.PARTICLE) {
                throw new XMLSchemaException("InputMessageReference "
                        + imr.getFragmentIdentifier()
                        + " is not consistent with style "
                        + MultipartStyle.STYLE_IRI
                        + " defined on parent operation/interface. GED for payload type "
                        + new QName(getTargetNamespace(), componentName)
                        + " is invalid. Content model must be defined using a complex type"
                        + " that contains a sequence which contains only local element decls.");
            }
            xsp = (XSParticle)xso;
            term = xsp.getTerm();
            if (term.getType() != XSConstants.ELEMENT_DECLARATION) {
                throw new XMLSchemaException("InputMessageReference "
                        + imr.getFragmentIdentifier()
                        + " is not consistent with style "
                        + MultipartStyle.STYLE_IRI
                        + " defined on parent operation/interface. GED for payload type "
                        + new QName(getTargetNamespace(), componentName)
                        + " is invalid. Content model must be defined using a complex type"
                        + " that contains a sequence which contains only local element decls.");
            }
            org.apache.xerces.xs.XSElementDeclaration childDecl = null;
            childDecl = (org.apache.xerces.xs.XSElementDeclaration)term;
            if (xsp.getMinOccurs() != 1 && (xsp.getMaxOccursUnbounded() || xsp.getMaxOccurs() != 1)) {
                throw new XMLSchemaException("InputMessageReference "
                        + imr.getFragmentIdentifier()
                        + " is not consistent with style "
                        + MultipartStyle.STYLE_IRI
                        + " defined on parent operation/interface. GED for payload type "
                        + new QName(getTargetNamespace(), componentName)
                        + " is invalid. Content model must be defined using a complex type"
                        + " that contains a sequence which contains only local element decls"
                        + " whose 'minOccurs' and 'maxOccurs' values are set to '1'.");
            }
            String name = childDecl.getName();
            boolean in = childDecl.getNillable();
            String dv = childDecl.getConstraintValue();
            org.bluestemsoftware.specification.eoa.ext.intrface.MultipartStyle.ChildElement ce = null;
            ce = new org.bluestemsoftware.specification.eoa.ext.intrface.MultipartStyle.ChildElement(name, in, dv);
            childElements.add(ce);
        }
        return new MultipartStyle(new QName(getTargetNamespace(), operationName), childElements);
    }
    
    private static class SchemaReferenceInternal implements ComponentReference<Schema> {

        private ComponentName schemaName;

        public SchemaReferenceInternal(ComponentName schemaName) {
            this.schemaName = schemaName;
        }

        public FragmentIdentifier getFragmentIdentifier() {
            return new FragmentIdentifier("foo", Scheme.EOA, "bar", "");
        }

        public Schema getReferencedComponent() {
            throw new UnsupportedOperationException();
        }

        public Class<Schema> getReferencedComponentClass() {
            return Schema.class;
        }

        public ComponentName getReferencedComponentName() {
            return schemaName;
        }

        public ComponentType getReferencedComponentType() {
            return ComponentType.SCHEMA;
        }

    }

}
