/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.commons.util;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * Wraps an output stream impl allowoutg us to limit the number of bytes write from it.
 */

public class LimitOutputStream extends FilterOutputStream {

    private int writeLimit;
    private int bytesWritten;

    /**
     * Creates a LimitOutputStream
     * 
     * @param out -
     *        the underlyoutg output stream
     * @param writeLimit -
     *        an IOException will be thrown by write methods once indicated number of bytes has
     *        been written. Note that the largest possible number of bytes which can be written
     *        is Integer.MAX_VALUE or roughly 2GB
     */
    public LimitOutputStream(OutputStream out, int writeLimit) {
        super(out);
        this.writeLimit = writeLimit;
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (bytesWritten > writeLimit) {
            throw new IOException("Number of bytes written exceeds set limit of " + writeLimit + ".");
        }
        bytesWritten = bytesWritten + (len - off);
        super.write(b, off, len);
    }

    @Override
    public void write(byte[] b) throws IOException {
        write(b, 0, b.length);
    }

    @Override
    public void write(int b) throws IOException {
        if (bytesWritten > writeLimit) {
            throw new IOException("Number of bytes written exceeds set limit of " + writeLimit + ".");
        } else {
            bytesWritten++;
        }
        super.write(b);
    }

}