/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.commons.util;

import java.io.File;
import java.net.URI;

public final class URIUtils {
    /**
     * Converts a URI, which must be absolute, into a File. File name is as follows:
     * <p>
     * [scheme.] + [host.] + [port.] + [path] + '[[.]extension]'
     * <p>
     * where all illegal file name chars in [path] fragment are replaced with the '.' character
     * and extension argument is concatenated assuming path does not already end with the given
     * extension. As an example, if URL is:
     * <p>
     * http://bluestemsoftware.org:8090/foo/bar/1.0/
     * <p>
     * And extension is 'xsd', the name of returned file is:
     * <p>
     * http.bluestemsoftware.org.8090.foo.bar.1.0.xsd
     * 
     * @param extension
     *        or <code>null</code> if none, e.g. to create a directory
     * @return File
     */
    public static final File toFile(URI uri, String extension) {

        if (uri == null || !uri.isAbsolute()) {
            throw new IllegalArgumentException("invalid uri");
        }

        String location = uri.getScheme() == null ? "" : uri.getScheme();
        location = uri.getHost() == null ? location : location + '.' + uri.getHost();
        location = uri.getPort() < 0 ? location : location + '.' + uri.getPort();
        location = uri.getPath() == null ? location : location + uri.getPath();
        location = uri.getFragment() == null ? location : location + '.' + uri.getFragment();
        
        location = location.replaceAll("/", ".");
        location = location.replaceAll("\\\\", ".");
        location = location.replaceAll(":", ".");
        location = location.replaceAll("\\*", ".");
        location = location.replaceAll("\\?", ".");
        location = location.replaceAll("\"", ".");
        location = location.replaceAll("<", ".");
        location = location.replaceAll(">", ".");
        location = location.replaceAll("\\|", ".");
        location = location.replaceAll("\\(", ".");
        location = location.replaceAll("\\)", ".");
        
        if (extension != null) {
            location = location.endsWith(".") ? location : location + '.';
            location = location.endsWith(extension) ? location : location + extension;
        } else {
            location = location.endsWith(".") ? location.substring(0, location.length() - 1) : location;
        }

        return new File(location);

    }

}
