/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.commons.util;

import java.util.Enumeration;
import java.util.Stack;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.sax.TransformerHandler;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.NamespaceSupport;

/**
 * Builds a DOM from sax events. Optionally removes ignorable element whitespace and coalesces
 * text nodes.
 * <p>
 * Note that this could have been implemented using {@link TransformerHandler} with a
 * {@link DOMResult}. When using schemas, however, according to XML 1.0 spec, the
 * ignorableWhiteSpace event is not reported, i.e. it is only reported when parsing within
 * context of a DTD. Note also that this behavior, i.e. when used within context of a schema,
 * used to work under jdk 1.5, but was apparently a bug.
 * 
 * see http://www.w3.org/TR/2006/REC-xml-20060816/#sec-white-space see
 * http://bugs.sun.com/bugdatabase/view_bug.do;jsessionid=b5f2856b680cdffffffffe7783f0790b719d?bug_id=6545684
 */
public class DOMHandler extends DefaultHandler implements LexicalHandler {

    private DocumentBuilder documentBuilder;
    private Document document;
    private Stack<Element> elementStack;
    private NamespaceSupport namespaceStack;
    private boolean newContextRequired;
    private boolean insideCDATASection;
    private StringBuilder cdataText;
    private Element currentElement;
    private StringBuilder textBuffer;
    private boolean ignoreElementWhitespace;
    private boolean coalesceTextNodes;

    public DOMHandler(boolean ignoreElementWhitespace, boolean coalesceTextNodes)
            throws ParserConfigurationException {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        documentBuilder = dbf.newDocumentBuilder();
        elementStack = new Stack<Element>();
        namespaceStack = new NamespaceSupport();
        textBuffer = new StringBuilder();
        this.ignoreElementWhitespace = ignoreElementWhitespace;
        this.coalesceTextNodes = coalesceTextNodes;
    }

    /**
     * Gets the completed document.
     * 
     * @return
     */
    public org.w3c.dom.Document getDocument() {
        return document;
    }

    /**
     * If true, handler merges ALL adjacent text events into just one text event, i.e.
     * coalesces into a single text node. Note that cdata is not coalesced.
     * 
     * @return
     */
    public boolean isCoalesceTextNodes() {
        return coalesceTextNodes;
    }

    /**
     * If true, ignores element whitespace. Note that because a schema is not used, it is
     * impossible to know if whitespace is truly ignorable. Consequently the handler just
     * discards all 'empty' text nodes. This feature should therefore be used in conjunction
     * with the coalesce feature.
     * 
     * @return
     */
    public boolean isIgnoreElementWhitespace() {
        return ignoreElementWhitespace;
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#startDocument()
     */
    public void startDocument() {
        document = documentBuilder.newDocument();
        newContextRequired = true;
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#endDocument()
     */
    public void endDocument() {
        elementStack.clear();
        namespaceStack.reset();
        textBuffer = new StringBuilder();
        cdataText = null;
        currentElement = null;
        newContextRequired = false;
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#startPrefixMapping(java.lang.String,
     *      java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri) throws SAXException {
        if (newContextRequired) {
            namespaceStack.pushContext();
            newContextRequired = false;
        }
        namespaceStack.declarePrefix(prefix, uri);
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String,
     *      java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String namespaceURI, String localName, String qualifiedName, Attributes attributes) throws SAXException {
        if (newContextRequired) {
            namespaceStack.pushContext();
        }
        completeCurrentTextNode();
        Element element = document.createElementNS(namespaceURI, qualifiedName);
        addAttributes(element, attributes);
        addDeclaredNamespaces(element);
        if (document.getDocumentElement() == null) {
            document.appendChild(element);
        } else {
            currentElement.appendChild(element);
        }
        elementStack.push(element);
        currentElement = element;
        newContextRequired = true;
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public void endElement(String namespaceURI, String localName, String qName) throws SAXException {
        completeCurrentTextNode();
        elementStack.pop();
        currentElement = elementStack.size() == 0 ? null : elementStack.peek();
        namespaceStack.popContext();
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
     */
    public void characters(char[] ch, int start, int end) throws SAXException {
        if (end == 0) {
            return;
        }
        if (currentElement == null) {
            return;
        }
        if (insideCDATASection) {
            cdataText.append(new String(ch, start, end));
        } else {
            if (coalesceTextNodes) {
                textBuffer.append(ch, start, end);
            } else {
                Text text = document.createTextNode(textBuffer.toString());
                currentElement.appendChild(text);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.ext.LexicalHandler#startDTD(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public void startDTD(String name, String publicId, String systemId) throws SAXException {
        throw new UnsupportedOperationException("startDTD");
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.ext.LexicalHandler#endDTD()
     */
    public void endDTD() throws SAXException {
        throw new UnsupportedOperationException("endDTD");
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.ext.LexicalHandler#startEntity(java.lang.String)
     */
    public void startEntity(String name) throws SAXException {
        throw new UnsupportedOperationException("startEntity");
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.ext.LexicalHandler#endEntity(java.lang.String)
     */
    public void endEntity(String name) throws SAXException {
        throw new UnsupportedOperationException("endEntity");
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.ext.LexicalHandler#startCDATA()
     */
    public void startCDATA() throws SAXException {
        insideCDATASection = true;
        cdataText = new StringBuilder();
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.ext.LexicalHandler#endCDATA()
     */
    public void endCDATA() throws SAXException {
        insideCDATASection = false;
        Node cdata = document.createCDATASection(cdataText.toString());
        currentElement.appendChild(cdata);
    }

    /*
     * (non-Javadoc)
     * @see org.xml.sax.ext.LexicalHandler#comment(char[], int, int)
     */
    public void comment(char[] ch, int start, int end) throws SAXException {
        // ignore
    }

    private void completeCurrentTextNode() {
        if (textBuffer.length() == 0) {
            return;
        }
        if (ignoreElementWhitespace) {
            boolean whitespace = true;
            for (int i = 0, size = textBuffer.length(); i < size; i++) {
                if (!Character.isWhitespace(textBuffer.charAt(i))) {
                    whitespace = false;
                    break;
                }
            }
            if (!whitespace) {
                Text text = document.createTextNode(textBuffer.toString());
                currentElement.appendChild(text);
            }
        } else {
            Text text = document.createTextNode(textBuffer.toString());
            currentElement.appendChild(text);
        }
        textBuffer.setLength(0);
    }

    private void addDeclaredNamespaces(Element element) {
        Enumeration<?> enumeration = namespaceStack.getDeclaredPrefixes();
        while (enumeration.hasMoreElements()) {
            String prefix = (String)enumeration.nextElement();
            String uri = namespaceStack.getURI(prefix);
            if (prefix.equals(XMLConstants.DEFAULT_NS_PREFIX)) {
                element.setAttribute("xmlns", uri);
            } else {
                element.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:" + prefix, uri);
            }
        }
    }

    private void addAttributes(Element element, Attributes attributes) {
        int size = attributes.getLength();
        for (int i = 0; i < size; i++) {
            String attributeQName = attributes.getQName(i);
            if (!attributeQName.startsWith("xmlns")) {
                String namespaceURI = attributes.getURI(i);
                String qualifiedName = attributes.getQName(i);
                String value = attributes.getValue(i);
                if (namespaceURI.equals(XMLConstants.NULL_NS_URI)) {
                    element.setAttribute(qualifiedName, value);
                } else {
                    element.setAttributeNS(namespaceURI, qualifiedName, value);
                }
            }
        }
    }

}