/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.commons.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.transform.dom.DOMSource;
import javax.xml.validation.Schema;
import javax.xml.validation.Validator;

import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

public final class DOMValidator {

    private Schema schema;

    public DOMValidator(Schema schema) {
        this.schema = schema;
    }

    /**
     * Validates source against schema set on constructor. This method is threadsafe.
     * @param source
     * @throws SAXException
     * @throws IOException
     * @return parsing/validation errors or <code>null</code> if schema valid
     */
    public String validate(DOMSource source) throws SAXException, IOException {
        Validator validator = schema.newValidator();
        validator.setErrorHandler(new ErrorHandlerImpl());
        validator.validate(source);
        return ((ErrorHandlerImpl)validator.getErrorHandler()).getErrors();
    }

    static class ErrorHandlerImpl implements ErrorHandler {

        private ArrayList<String> errorMessages = new ArrayList<String>();

        public void error(SAXParseException spe) throws SAXException {
            errorMessages.add(spe.getMessage());
        }

        public void warning(SAXParseException spe) throws SAXException {
            errorMessages.add(spe.getMessage());
        }

        public void fatalError(SAXParseException spe) throws SAXException {
            errorMessages.add(spe.getMessage());
        }

        public String getErrors() {
            if (errorMessages.size() > 0) {
                Iterator<String> messageIterator = errorMessages.iterator();
                StringBuilder errorMessage = new StringBuilder();
                while (messageIterator.hasNext()) {
                    errorMessage.append(messageIterator.next() + "  ");
                }
                return errorMessage.toString();
            }
            return null;
        }

    }

}
