/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.commons.util;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Wraps an input stream impl allowing us to limit the number of bytes
 * read from it.
 */

public class LimitInputStream extends FilterInputStream {
	
	private int readLimit;
	private int bytesRead;
	
	/**
	 * Creates a LimitInputStream
	 * 
	 * @param in - the underlying input stream
	 * @param readLimit - an IOException will be thrown by read methods
	 * once indicated number of bytes has been read. note that the
	 * largest possible message size is Integer.MAX_VALUE or roughly 2GB 	
	 */
	public LimitInputStream(InputStream in, int readLimit) {
		super(in);
		this.readLimit = readLimit;
		
	}
	
	@Override
	public synchronized int read() throws IOException {
		
		if (bytesRead > readLimit) {
			throw new IOException("Number of bytes read exceeds set limit of "
					+ readLimit
					+ ".");
		} else {
			bytesRead++;
		}
		
		return super.read();
		
	}
	
	@Override
	public int read(byte[] b) throws IOException {
		return read(b, 0, b.length);
	}
	
	@Override
	public synchronized int read(byte[] b, int off, int len) throws IOException {

		if (bytesRead > readLimit) {
			throw new IOException("Number of bytes read exceeds set limit of "
					+ readLimit
					+ ".");
		}
		
		int localBytesRead = super.read(b, off, len);
		
		if (localBytesRead > 0) {
			bytesRead = bytesRead + localBytesRead;
		}
		
		return localBytesRead;
	}
	
}