/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.maven.plugin.csum;

import java.io.File;
import java.util.Iterator;
import java.util.List;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.codehaus.plexus.digest.Digester;
import org.codehaus.plexus.util.FileUtils;

/**
 * Checksums the distribution artifacts. Note that we can't use the maven install
 * plugin to do this, because it does not write sums to target dir, i.e. it calcs
 * them to a temp file instead. We ftp directly from the target dir.
 * 
 * @goal checksum
 */
public class ChecksumMojo extends AbstractMojo {

    /**
     * Path to project 'target' dir.
     * 
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private String targetDirectory;

    /**
     * Digester for MD5.
     * 
     * @component role-hint="md5"
     */
    private Digester md5Digester;

    /**
     * Digester for SHA-1.
     * 
     * @component role-hint="sha1"
     */
    private Digester sha1Digester;

    /**
     * @parameter expression="${project.artifact}"
     * @required
     * @readonly
     */
    private Artifact artifact;

    /**
     * @parameter expression="${project.attachedArtifacts}
     * @required
     * @readonly
     */
    private List<?> attachedArtifacts;

    /*
     * (non-Javadoc)
     * @see org.apache.maven.plugin.AbstractMojo#execute()
     */
    public void execute() throws MojoExecutionException {

        File file = artifact.getFile();

        try {
            String sum = md5Digester.calc(file);
            File target = new File(targetDirectory, file.getName() + ".md5");
            FileUtils.fileWrite(target.getAbsolutePath(), sum);
        } catch (Exception ex) {
            throw new MojoExecutionException("Error calculating digest. " + ex);
        }

        try {
            String sum = sha1Digester.calc(file);
            File target = new File(targetDirectory, file.getName() + ".sha1");
            FileUtils.fileWrite(target.getAbsolutePath(), sum);
        } catch (Exception ex) {
            throw new MojoExecutionException("Error calculating digest. " + ex);
        }

        Iterator<?> itr = attachedArtifacts.iterator();

        while (itr.hasNext()) {

            file = ((Artifact)itr.next()).getFile();

            if (!performSum(file.getName())) {
                continue;
            }

            try {
                String sum = md5Digester.calc(file);
                File target = new File(targetDirectory, file.getName() + ".md5");
                FileUtils.fileWrite(target.getAbsolutePath(), sum);
            } catch (Exception ex) {
                throw new MojoExecutionException("Error calculating digest. " + ex);
            }

            try {
                String sum = sha1Digester.calc(file);
                File target = new File(targetDirectory, file.getName() + ".sha1");
                FileUtils.fileWrite(target.getAbsolutePath(), sum);
            } catch (Exception ex) {
                throw new MojoExecutionException("Error calculating digest. " + ex);
            }

        }

    }

    private boolean performSum(String file) {
        return file.endsWith("jar") || file.endsWith("exe") || file.endsWith("gz");
    }
}
