/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.maven.plugin.ftp;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.jibble.simpleftp.InvalidPasswordException;
import org.jibble.simpleftp.SimpleFTP;

/**
 * Ftp's distribution artifacts to remote server(s).
 * 
 * @goal upload
 */
public class FTPMojo extends AbstractMojo {
    
    /**
     * Path to project 'target' dir.
     * 
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private String targetDirectory;

    /**
     * Configuration
     * 
     * @parameter
     */
    private List<Server> servers = new ArrayList<Server>();

    /*
     * (non-Javadoc)
     * @see org.apache.maven.plugin.AbstractMojo#execute()
     */
    public void execute() throws MojoExecutionException {
        
        for (Server server : servers) {

            SimpleFTP ftp = new SimpleFTP(getLog());

            String url = server.getUrl();
            String user = server.getUser();
            String prompt = "Password for " + user + "@" + url + ": ";
            char[] passwd = System.console().readPassword(prompt, (Object[])null);

            try {
                getLog().info("connecting ...");
                try {
                    ftp.connect(url, 21, user, new String(passwd));
                } catch (InvalidPasswordException ie) {
                    try {
                        prompt = "Invalid password. Try again. ";
                        passwd = System.console().readPassword(prompt, (Object[])null);
                        ftp.connect(url, 21, user, new String(passwd));
                    } catch (InvalidPasswordException iee) {
                        prompt = "Invalid password. Last Chance! ";
                        passwd = System.console().readPassword(prompt, (Object[])null);
                        ftp.connect(url, 21, user, new String(passwd));
                    }
                }
                ftp.bin();
                ftp.cwd(server.getTarget());
                getLog().info("working directory is " + ftp.pwd());
                for (String file : server.getFiles()) {
                    if (ftp.exists(file)) {
                        getLog().warn("not overwriting file " + file);
                        continue;
                    }
                    getLog().info("uploading file " + file);
                    if (!ftp.stor(new File(targetDirectory, file))) {
                        throw new MojoExecutionException("Failed transferring file " + file + ".");
                    }
                    if (!ftp.chmod("444", file)) {
                        throw new MojoExecutionException("Failed flagging file " + file + " as readonly.");
                    }
                }
                getLog().info("upload successful");
                ftp.disconnect();
            } catch (IOException ie) {
                throw new MojoExecutionException("Error uploading files. " + ie);
            }

        }

    }

}
