/*
 * $Id: WebPackager.java,v 1.1 2008/05/19 16:28:25 wolfftw Exp $
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 * 
 * http://izpack.org/
 * http://izpack.codehaus.org/
 * 
 * Copyright 2008 Bluestem Software LLC
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.izforge.izpack.compiler;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.jar.JarFile;
import java.util.jar.Pack200;
import java.util.zip.Deflater;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import net.n3.nanoxml.XMLElement;

import com.izforge.izpack.Pack;
import com.izforge.izpack.PackFile;

/**
 * Copied overridden methods from com.izforge.izpack.compiler.Packager and adapted to
 * accommodate pack200 packaging. Removed all non-web related packaging logic.
 * 
 * @author twayne@treleis.org
 * 
 */
public class WebPackager extends Packager {
    
    private Map<String, String> pack200Props = new HashMap<String,String>();

    /**
     * Default constructor. Note that because we use pack200 + gzip, the compression format and
     * level args are not supported / unused.
     * 
     * @throws CompilerException
     */
    public WebPackager() throws CompilerException {
    }
    
    /*
     * (non-Javadoc)
     * @see com.izforge.izpack.compiler.Packager#addConfigurationInformation(net.n3.nanoxml.XMLElement)
     */
    @Override
    public void addConfigurationInformation(XMLElement data) {
        XMLElement pack200Element = data.getFirstChildNamed("pack200");
        if (pack200Element != null) {
            Iterator<?> itr = pack200Element.getChildren().iterator();
            while (itr.hasNext()) {
                XMLElement propertyElement = (XMLElement)itr.next();
                String propertyName = propertyElement.getAttribute("name");
                if (propertyName == null) {
                    throw new RuntimeException("pack200 options property element missing"
                            + " required 'name' attribute.");
                }
                String propertyValue = propertyElement.getAttribute("value");
                if (propertyName == null) {
                    throw new RuntimeException("pack200 options property element missing"
                            + " required 'value' attribute.");
                }
                pack200Props.put(propertyName, propertyValue);
            }
        }
    }    

    /*
     * (non-Javadoc)
     * @see com.izforge.izpack.compiler.IPackager#createInstaller(java.io.File)
     */
    @Override
    public void createInstaller(File primaryFile) throws Exception {

        String baseName = primaryFile.getName();
        if (baseName.endsWith(".jar")) {
            baseName = baseName.substring(0, baseName.length() - 4);
            baseFile = new File(primaryFile.getParentFile(), baseName);
        } else {
            baseFile = primaryFile;
        }

        info.setInstallerBase(baseFile.getName());
        if (info.getWebDirURL() == null) {
            throw new CompilerException("WebPackager requires specification of a 'webdir' URL.");
        } else {
            packJarsSeparate = true;
        }

        // create the installation jar. because this is a web based installer,
        // the pack files are always packed separately

        File file = new File(baseFile.getParentFile(), baseFile.getName() + ".jar");
        sendMsg("Building installer jar: " + file.getAbsolutePath());
        primaryJarStream = new com.izforge.izpack.util.JarOutputStream(file);
        primaryJarStream.setLevel(Deflater.BEST_COMPRESSION);
        primaryJarStream.setPreventClose(true);

        sendStart();

        writeInstaller();

        primaryJarStream.closeAlways();

        sendStop();
    }

    @Override
    protected void writePacks() throws Exception {

        int packNumber = 1;
        Iterator<PackInfo> packIter = packsList.iterator();

        XMLElement root = new XMLElement("packs");

        while (packIter.hasNext()) {

            PackInfo packInfo = packIter.next();
            Pack pack = packInfo.getPack();

            if ((pack.id == null) || (pack.id.length() == 0)) {
                pack.id = pack.name;
            }
            
            boolean isLocalInstallation = info.getWebDirURL().equals("./");

            // create an uncompressed jar specific to this pack

            String jarName = baseFile.getName() + ".pack-" + pack.id + ".jar";
            File jarFile = new File(baseFile.getParentFile(), jarName);
            ByteCountingOutputStream bos = new ByteCountingOutputStream(new FileOutputStream(jarFile));
            ZipOutputStream packStream = new ZipOutputStream(bos);
            packStream.setLevel(Deflater.NO_COMPRESSION);
            sendMsg("Writing pack " + packNumber + ": " + pack.name, PackagerListener.MSG_INFO);

            // Retrieve the correct output stream
            ZipEntry entry = new ZipEntry("metadata");
            packStream.putNextEntry(entry);
            packStream.flush(); // flush before we start counting
            ObjectOutputStream objOut = new ObjectOutputStream(packStream);

            // serialize the pack file metadata, i.e the actual
            // PackFile objects
            objOut.writeInt(packInfo.getPackFiles().size());

            Iterator<?> iter = packInfo.getPackFiles().iterator();
            while (iter.hasNext()) {
                PackFile pf = (PackFile)iter.next();
                objOut.writeObject(pf);
                objOut.flush();
            }

            // Write out information about parsable files
            objOut.writeInt(packInfo.getParsables().size());
            iter = packInfo.getParsables().iterator();
            while (iter.hasNext()) {
                objOut.writeObject(iter.next());
            }

            // Write out information about executable files
            objOut.writeInt(packInfo.getExecutables().size());
            iter = packInfo.getExecutables().iterator();
            while (iter.hasNext()) {
                objOut.writeObject(iter.next());
            }

            // Write out information about updatecheck files
            objOut.writeInt(packInfo.getUpdateChecks().size());
            iter = packInfo.getUpdateChecks().iterator();
            while (iter.hasNext()) {
                objOut.writeObject(iter.next());
            }

            // Cleanup
            objOut.flush();
            packStream.closeEntry();

            // now we write the actual file/dir entry to the jar. note this was
            // re-factored from packager class which included the file
            // within the PackFile object serialization. this is necessary
            // to facilitate packing with pack200

            sendMsg("Writing pack files ... ", PackagerListener.MSG_INFO);
            iter = packInfo.getPackFiles().iterator();
            while (iter.hasNext()) {
                PackFile pf = (PackFile)iter.next();
                File file = packInfo.getFile(pf);
                if (!pf.isDirectory()) {
                    entry = new ZipEntry(pf.getTargetPath());
                    packStream.putNextEntry(entry);
                    PackagerHelper.copyStream(new FileInputStream(file), packStream);
                    packStream.closeEntry();
                }
            }

            packStream.flush();
            objOut.close(); // also closes packStream
            
            if (!isLocalInstallation) {
                // compress the pack file jar using pack200 and gzip
                sendMsg("Compressing pack using pack200 + gzip ... ", PackagerListener.MSG_INFO);
                JarFile source = new JarFile(jarFile);
                File p200File = new File(baseFile.getParentFile(), jarName + ".pack.gz");
                Pack200.Packer packer = Pack200.newPacker();
                packer.properties().putAll(pack200Props);
                FileOutputStream fos = new FileOutputStream(p200File);
                bos = new ByteCountingOutputStream(fos);
                GZIPOutputStream gos = new GZIPOutputStream(bos);
                OutputStream out = new BufferedOutputStream(gos);
                packer.pack(source, out);
                out.close();
                source.close();
            }

            // retrieve number of compressed bytes written by
            // gzipped stream and set as metadata
            pack.nbytes = bos.getByteCount();

            // write pack metadata to file used elsewhere within
            // the compile process
            XMLElement child = new XMLElement("pack");
            child.setAttribute("nbytes", Long.toString(pack.nbytes));
            child.setAttribute("name", pack.name);
            if (pack.id != null)
                child.setAttribute("id", pack.id);
            root.addChild(child);

            packNumber++;

        }

        // Now that we know sizes, write pack metadata to primary jar.
        primaryJarStream.putNextEntry(new org.apache.tools.zip.ZipEntry("packs.info"));
        ObjectOutputStream out = new ObjectOutputStream(primaryJarStream);
        out.writeInt(packsList.size());

        Iterator<PackInfo> i = packsList.iterator();
        while (i.hasNext()) {
            PackInfo pack = i.next();
            out.writeObject(pack.getPack());
        }
        out.flush();
        primaryJarStream.closeEntry();

    }

}
