/*
 * $Id: WebUnpacker.java,v 1.2 2008/05/20 05:03:42 wolfftw Exp $
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 * 
 * http://izpack.org/
 * http://izpack.codehaus.org/
 * 
 * Copyright 2001 Johannes Lehtinen
 * Copyright 2008 Bluestem Software
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.izforge.izpack.installer;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.OutputStream;
import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;

import com.izforge.izpack.ExecutableFile;
import com.izforge.izpack.Pack;
import com.izforge.izpack.PackFile;
import com.izforge.izpack.ParsableFile;
import com.izforge.izpack.UpdateCheck;
import com.izforge.izpack.event.InstallerListener;
import com.izforge.izpack.util.AbstractUIHandler;
import com.izforge.izpack.util.AbstractUIProgressHandler;
import com.izforge.izpack.util.FileExecutor;
import com.izforge.izpack.util.Housekeeper;
import com.izforge.izpack.util.IoHelper;
import com.izforge.izpack.util.OsConstraint;

/**
 * Copied from com.izforge.izpack.installer.Unpacker and modified to accommodate remote package
 * download, verification and installation.
 * 
 * @author twayne@treleis.org
 */
public class WebUnpacker extends UnpackerBase {

    private int progressMeter;

    /**
     * The constructor.
     * 
     * @param idata
     *        The installation data.
     * @param handler
     *        The installation progress handler.
     */
    public WebUnpacker(AutomatedInstallData idata, AbstractUIProgressHandler handler) {
        super(idata, handler);
    }

    /*
     * (non-Javadoc)
     * @see com.izforge.izpack.installer.IUnpacker#run()
     */
    public void run() {

        addToInstances();

        try {

            ArrayList<ParsableFile> parsables = new ArrayList<ParsableFile>();
            ArrayList<ExecutableFile> executables = new ArrayList<ExecutableFile>();
            ArrayList<UpdateCheck> updatechecks = new ArrayList<UpdateCheck>();
            List<?> packs = idata.selectedPacks;
            int npacks = packs.size();
            handler.startAction("Unpacking", npacks);
            udata = UninstallData.getInstance();
            // Custom action listener stuff --- load listeners ----
            List<?>[] customActions = getCustomActions();
            // Custom action listener stuff --- beforePacks ----
            informListeners(customActions, InstallerListener.BEFORE_PACKS, idata, npacks, handler);
            packs = idata.selectedPacks;
            npacks = packs.size();

            // We unpack the selected packs
            for (int i = 0; i < npacks; i++) {

                // We get the pack stream
                // int n = idata.allPacks.indexOf(packs.get(i));
                Pack p = (Pack)packs.get(i);

                // evaluate condition
                if (p.hasCondition()) {
                    if (rules != null) {
                        if (!rules.isConditionTrue(p.getCondition())) {
                            // skip pack, condition is not fullfilled.
                            continue;
                        }
                    } else {
                        // TODO: skip pack, because condition can not be checked
                    }
                }

                // Custom action listener stuff --- beforePack ----
                informListeners(customActions, InstallerListener.BEFORE_PACK, packs.get(i), npacks, handler);

                String tempfile;
                try {
                    tempfile = downloadAndVerify(p);
                    udata.addFile(tempfile, p.uninstall);
                } catch (Exception e) {
                    if ("Cancelled".equals(e.getMessage()))
                        throw new InstallerException("Installation cancelled", e);
                    else
                        throw new InstallerException("Installation failed. " + e);
                }

                // the downloaded file is a jar file. we need to open
                // a stream on the zipped entry which is a serialized
                // Pack object

                JarFile jarFile = new JarFile(tempfile);
                ZipEntry entry = jarFile.getEntry("metadata");
                InputStream in = jarFile.getInputStream(entry);

                // We unpack the metadata
                ObjectInputStream objIn = new ObjectInputStream(in);
                int nfiles = objIn.readInt();

                // We get the internationalized name of the pack
                final Pack pack = ((Pack)packs.get(i));
                String stepname = pack.name;// the message to be passed to the
                // installpanel
                if (langpack != null && !(pack.id == null || "".equals(pack.id))) {

                    final String name = langpack.getString(pack.id);
                    if (name != null && !"".equals(name)) {
                        stepname = name;
                    }
                }

                // each PackFile object is serialized to object stream.
                // iterate over each. check conditions and add writable
                // files (metadata) to list

                List<PackFile> filesToWrite = getFilesToWrite(objIn, nfiles);

                // send metadata used to update step number and used to
                // update the pack progress bar. nfiles is the max
                // number, i.e. we should update progress bar max num
                // of times - 1

                int progressMeterMax = filesToWrite.size();
                progressMeterMax = progressMeterMax + 2;
                handler.nextStep(stepname, i + 1, filesToWrite.size());

                // now we read the rest of the metadata from object input
                // stream begining with information about parsable files

                handler.progress(progressMeter++, "reading metadata ...");

                int numParsables = objIn.readInt();
                for (int k = 0; k < numParsables; k++) {
                    ParsableFile pf = (ParsableFile)objIn.readObject();
                    if (pf.hasCondition() && (rules != null)) {
                        if (!rules.isConditionTrue(pf.getCondition())) {
                            // skip, condition is not fulfilled
                            continue;
                        }
                    }
                    pf.path = IoHelper.translatePath(pf.path, vs);
                    parsables.add(pf);
                }

                // Load information about executable files
                int numExecutables = objIn.readInt();
                for (int k = 0; k < numExecutables; k++) {
                    ExecutableFile ef = (ExecutableFile)objIn.readObject();
                    if (ef.hasCondition() && (rules != null)) {
                        if (!rules.isConditionTrue(ef.getCondition())) {
                            // skip, condition is false
                            continue;
                        }
                    }
                    ef.path = IoHelper.translatePath(ef.path, vs);
                    if (null != ef.argList && !ef.argList.isEmpty()) {
                        String arg = null;
                        for (int j = 0; j < ef.argList.size(); j++) {
                            arg = ef.argList.get(j);
                            arg = IoHelper.translatePath(arg, vs);
                            ef.argList.set(j, arg);
                        }
                    }
                    executables.add(ef);
                    if (ef.executionStage == ExecutableFile.UNINSTALL) {
                        udata.addExecutable(ef);
                    }
                }

                // Custom action listener stuff --- uninstall data ----
                handleAdditionalUninstallData(udata, customActions);

                // Load information about updatechecks
                int numUpdateChecks = objIn.readInt();

                for (int k = 0; k < numUpdateChecks; k++) {
                    UpdateCheck uc = (UpdateCheck)objIn.readObject();
                    updatechecks.add(uc);
                }

                objIn.close();

                // we've finished processing the metadata and we now have a list
                // of entries to unpack to target location

                writeFiles(filesToWrite, jarFile, customActions, pack.uninstall);

                if (performInterrupted()) { // Interrupt was initiated; perform it.
                    return;
                }

                // Custom action listener stuff --- afterPack ----
                handler.progress(progressMeter++, "finalizing ...");
                informListeners(customActions, InstallerListener.AFTER_PACK, packs.get(i), i, handler);

            }

            // We use the scripts parser
            ScriptParser parser = new ScriptParser(parsables, vs);
            parser.parseFiles();
            if (performInterrupted()) { // Interrupt was initiated; perform it.
                return;
            }

            // We use the file executor
            FileExecutor executor = new FileExecutor(executables);
            if (executor.executeFiles(ExecutableFile.POSTINSTALL, handler) != 0) {
                handler.emitError("File execution failed", "The installation was not completed");
                this.result = false;
            }

            if (performInterrupted()) { // Interrupt was initiated; perform it.
                return;
            }

            // We put the uninstaller (it's not yet complete...)
            putUninstaller();

            // Custom action listener stuff --- afterPacks ----
            informListeners(customActions, InstallerListener.AFTER_PACKS, idata, handler, null);

            if (performInterrupted()) { // Interrupt was initiated; perform it.
                return;
            }

            // update checks after uninstaller and any custom action artifacts
            // were put, so we don't delete them
            performUpdateChecks(updatechecks);

            if (performInterrupted()) { // Interrupt was initiated; perform it.
                return;
            }

            // write installation information
            writeInstallationInformation();

            // The end :-)
            handler.stopAction();

        } catch (Exception err) {
            // TODO: finer grained error handling with useful error messages
            handler.stopAction();
            if ("Installation cancelled".equals(err.getMessage())) {
                handler.emitNotification("Installation cancelled");
            } else {
                handler.emitError("An error occured", err.getMessage());
                err.printStackTrace();
            }
            this.result = false;
            Housekeeper.getInstance().shutDown(4);
        } finally {
            removeFromInstances();
        }
    }

    private String downloadAndVerify(Pack p) throws InstallerException {

        boolean isLocalInstallation = false;
        String baseName = idata.info.getInstallerBase();
        String webDirURL = idata.info.getWebDirURL();
        if (webDirURL.equals("./")) {
            isLocalInstallation = true;
        }

        CheckSumObserver cso = null;

        // set up the checksum observer. note that we can't dowload the checksum
        // file until after we download the pack. this is to allow the web
        // accessor to prompt for and set proxy related system props if required

        if (p.getDigestLocation() != null && !isLocalInstallation) {
            String locationPath = p.getDigestLocation().toString();
            int index = locationPath.lastIndexOf(".");
            try {
                cso = new CheckSumObserver(locationPath.substring(index + 1, locationPath.length()));
            } catch (NoSuchAlgorithmException ne) {
                throw new InstallerException("Error processing digest for pack '"
                        + p.name
                        + "'. "
                        + ne.getMessage());
            }
        }

        // download the file using the webaccessor. if we created a checksum observer,
        // we must siphon off the bytes used to create the actual check sum which will
        // be compared to the expected checksum

        File downloadedFile = null;
        try {

            if (isLocalInstallation) {
                
                String packURL = webDirURL + baseName + ".pack-" + p.id + ".jar";
                downloadedFile = new File(packURL);
                if (!downloadedFile.exists()) {
                    throw new FileNotFoundException(packURL);
                }

            } else {
                
                String packURL = webDirURL + "/" + baseName + ".pack-" + p.id + ".jar.pack.gz";
                downloadedFile = File.createTempFile("pack-", ".jar.pack.gz");
                WebAccessor w = new WebAccessor(null);
                InputStream in = w.openInputStream(new URL(packURL));

                if (in == null) {
                    throw new InstallerException("Unable to retrieve pack at location " + packURL);
                }

                if (cso != null) {
                    OutputStream out = null;
                    try {
                        out = new BufferedOutputStream(new FileOutputStream(downloadedFile));
                        byte[] buffer = new byte[4096];
                        for (int length = 0; length >= 0; length = in.read(buffer)) {
                            out.write(buffer, 0, length);
                            cso.update(buffer, length);
                        }
                    } finally {
                        if (in != null) {
                            try {
                                in.close();
                            } catch (IOException ignore) {
                            }
                        }
                        if (out != null) {
                            try {
                                out.close();
                            } catch (IOException ignore) {
                            }
                        }
                    }
                } else {
                    downloadedFile = IoHelper.writeStream(in, downloadedFile);
                }
            }
        } catch (Exception ex) {
            throw new InstallerException("Error downloading pack '" + p.name + "'" + "." + ex);
        }

        // if we're retrieving packs from current dir, we're done here
        
        if (isLocalInstallation) {
            return downloadedFile.getAbsolutePath();
        }
        
        // now we retrieve the digest file and extract the expected
        // checksum value and make the comparison. note that if web
        // dir specifies local directory as pack(s) location, we skip
        // this step

        String expectedChecksum = null;
        if (p.getDigestLocation() != null) {
            int numtries = 3;
            InputStream in = null;
            while (in == null && numtries > 0) {
                try {
                    in = p.getDigestLocation().openStream();
                    expectedChecksum = IoHelper.streamToString(in, null);
                } catch (ConnectException tryAgain) {
                    System.out.println(tryAgain.getMessage());
                    numtries--;
                } catch (Exception ex) {
                    throw new InstallerException("Failed to retrieve digest for pack '" + p.name + "'.", ex);
                }
            }
            if (in == null) {
                throw new InstallerException("Failed to retrieve digest for pack '" + p.name + "'.");
            }
            String actualChecksum = cso.getCheckSum();
            if (!actualChecksum.equals(expectedChecksum)) {
                throw new InstallerException("Message digest comparison failed for pack '"
                        + p.name
                        + "'."
                        + " Found "
                        + actualChecksum
                        + ". Expected "
                        + expectedChecksum);
            }
        }

        // the downloaded file is compressed using pack200 and gzip.
        // we unwind it here and pass path to plain old jar to caller

        // unpack 200 stuff

        File tempFile = null;
        try {
            FileInputStream fin = new FileInputStream(downloadedFile);
            InputStream in = new BufferedInputStream(new GZIPInputStream(fin));
            tempFile = File.createTempFile("pack-", ".jar");
            OutputStream out = new BufferedOutputStream(new FileOutputStream(tempFile));
            JarOutputStream jout = new JarOutputStream(out);
            Pack200.Unpacker unpacker = Pack200.newUnpacker();
            unpacker.unpack(in, jout);
            in.close();
            jout.close();
        } catch (Exception ex) {
            throw new InstallerException("Error unpacking pack200 file for pack '" + p.name + "'.", ex);
        }

        return tempFile.getAbsolutePath();

    }

    private List<PackFile> getFilesToWrite(ObjectInputStream objIn, int nfiles) throws Exception {

        List<PackFile> filesToWrite = new ArrayList<PackFile>();

        for (int j = 0; j < nfiles; j++) {

            // We read the packed file metadata. note that we read
            // metadata from stream even if the file is skipped

            PackFile pf = (PackFile)objIn.readObject();

            // TODO: reaction if condition can not be checked
            if (pf.hasCondition() && (rules != null)) {
                if (!rules.isConditionTrue(pf.getCondition())) {
                    continue;
                }
            }

            if (!OsConstraint.oneMatchesCurrentSystem(pf.osConstraints())) {
                continue;
            }

            // We translate & build the path
            String path = IoHelper.translatePath(pf.getTargetPath(), vs);
            File pathFile = new File(path);

            if (pf.isDirectory()) {
                continue;
            }

            // if this file exists and should not be overwritten,
            // check what to do
            if ((pathFile.exists()) && (pf.override() != PackFile.OVERRIDE_TRUE)) {

                boolean overwritefile = false;

                // don't overwrite file if the user said so
                if (pf.override() != PackFile.OVERRIDE_FALSE) {
                    if (pf.override() == PackFile.OVERRIDE_TRUE) {
                        overwritefile = true;
                    } else if (pf.override() == PackFile.OVERRIDE_UPDATE) {
                        // check mtime of involved files
                        // (this is not 100% perfect, because the
                        // already existing file might
                        // still be modified but the new installed
                        // is just a bit newer; we would
                        // need the creation time of the existing
                        // file or record with which mtime
                        // it was installed...)
                        overwritefile = (pathFile.lastModified() < pf.lastModified());
                    } else {
                        int def_choice = -1;

                        if (pf.override() == PackFile.OVERRIDE_ASK_FALSE)
                            def_choice = AbstractUIHandler.ANSWER_NO;
                        if (pf.override() == PackFile.OVERRIDE_ASK_TRUE)
                            def_choice = AbstractUIHandler.ANSWER_YES;

                        int answer = handler.askQuestion(idata.langpack.getString("InstallPanel.overwrite.title")
                                + " - "
                                + pathFile.getName(), idata.langpack.getString("InstallPanel.overwrite.question")
                                + pathFile.getAbsolutePath(), AbstractUIHandler.CHOICES_YES_NO, def_choice);

                        overwritefile = (answer == AbstractUIHandler.ANSWER_YES);
                    }

                }

                if (!overwritefile) {
                    continue;
                }

            }

            filesToWrite.add(pf);

        }

        return filesToWrite;

    }

    private void writeFiles(List<PackFile> filesToWrite, JarFile jarFile, List<?>[] customActions, boolean uninstall) throws Exception {

        for (PackFile pf : filesToWrite) {

            String path = IoHelper.translatePath(pf.getTargetPath(), vs);
            File pathFile = new File(path);

            File dir = pathFile.getParentFile();

            // If there are custom actions which would be called
            // at creating a directory, create each recursively.
            // else if no notification is required, we create all
            // the directories in one shot

            if (!dir.exists()) {
                List<?> fileListeners = customActions[customActions.length - 1];
                if (fileListeners != null && fileListeners.size() > 0) {
                    mkDirsWithEnhancement(dir, pf, customActions);
                } else {
                    if (!dir.mkdirs()) {
                        handler.emitError("Error creating directories", "Could not create directory "
                                + dir.getPath());
                        handler.stopAction();
                        this.result = false;
                        return;
                    }
                }
            }

            // Custom action listener stuff --- beforeFile ----
            informListeners(customActions, InstallerListener.BEFORE_FILE, pathFile, pf, null);
            udata.addFile(path, uninstall);
            handler.progress(progressMeter++, path);

            // We write the entry to target location
            ZipEntry entry = jarFile.getEntry(pf.getTargetPath());
            IoHelper.writeStream(jarFile.getInputStream(entry), pathFile);

            // Set file modification time if specified
            if (pf.lastModified() >= 0)
                pathFile.setLastModified(pf.lastModified());
            // Custom action listener stuff --- afterFile ----
            informListeners(customActions, InstallerListener.AFTER_FILE, pathFile, pf, null);

        }

    }

    /**
     * Inner class which Encapsulates digest manipulations.
     * 
     * @author twayne@treleis.org
     * 
     */
    private static class CheckSumObserver {

        private MessageDigest messageDigest = null;
        private static final int BYTE_MASK = 0xFF;

        public CheckSumObserver(String algorithm) throws NoSuchAlgorithmException {
            messageDigest = MessageDigest.getInstance(algorithm);
        }

        /**
         * Resets the digester. Any data passed via {@link #update(byte[], int)} is effectively
         * discarded.
         */
        public void reset() {
            messageDigest.reset();
        }

        /**
         * Adds data to be used to compute the checksum.
         * 
         * @param buffer
         * @param length
         */
        public void update(byte[] buffer, int length) {
            messageDigest.update(buffer, 0, length);
        }

        /**
         * Uses added data to compute a checksum using given algorithm. Automatically performs
         * a reset following the computation.
         * 
         * @return Returns the checksum as an ASCII encoded hex string to facilitate
         *         comparison.
         */
        public String getCheckSum() {

            // convert the byte array into a string. each byte is represented
            // as two hex characters

            byte[] digest = messageDigest.digest();
            StringBuilder result = new StringBuilder();
            for (int i = 0; i < digest.length; i++) {
                String temp = Integer.toHexString(digest[i] & BYTE_MASK);
                if (temp.length() == 1) {
                    result.append("0" + temp);
                } else {
                    result.append(temp);
                }
            }
            return result.toString();

        }

    }
}
