/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.system.plugin.izpack;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.ArtifactUtils;
import org.apache.maven.artifact.DefaultArtifact;
import org.apache.maven.artifact.resolver.filter.ScopeArtifactFilter;
import org.apache.maven.model.Build;
import org.apache.maven.model.Model;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;
import org.apache.maven.project.path.DefaultPathTranslator;
import org.apache.maven.project.path.PathTranslator;
import org.bluestemsoftware.open.eoa.system.plugin.izpack.util.FileUtils;
import org.bluestemsoftware.open.eoa.system.plugin.izpack.util.InterpolationHelper;

import com.izforge.izpack.compiler.CompilerConfig;

/**
 * Using path to an izpack installation document invokes the izpack compiler after maven
 * substitution vars have been replaced w/ actual values retrieved from project model.
 * 
 * @goal izpack
 */
public class IzPackMojo extends AbstractMojo {

    /**
     * The maven project.
     * 
     * @parameter expression="${project}"
     * @required
     * @readonly
     * @description "the maven project to use"
     */
    private MavenProject project;

    /**
     * Directory that resources are copied to during the build.
     * 
     * @parameter expression="${project.build.directory}/${project.build.finalName}"
     * @required
     */
    private String workDirectory;

    /**
     * The directory for the generated source attachment.
     * 
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private String outputDirectory;

    /**
     * @parameter expression="${project.build.finalName}"
     * @required
     */
    private String finalName;

    /**
     * @parameter expression="${plugin.artifacts}"
     */
    private List<DefaultArtifact> pluginArtifacts;

    /**
     * 
     * Project helper injected by maven.
     * 
     * @component
     */
    private MavenProjectHelper projectHelper;

    /**
     * The izpack installation type, i.e. 'standard' or 'web'.
     * 
     * @parameter
     * @required
     */
    private String installationType;

    /**
     * Path to the izpack installation file.
     * 
     * @parameter
     * @required
     */
    private String installationFile;

    /**
     * A list of izpack 'panels', i.e. which are to be included within the install artifact.
     * Each 'panel' must be defined as a project dependency with scope 'runtime' and type
     * 'jar'. The dependencies are copied by the plugin to the 'bin/panels' sub-directory of
     * the izpack distro and each will be named {@link Panel#getName()} (note that this is also
     * the value used for the 'classname' attribute on the izpack 'panel' element within your
     * installation file.
     * 
     * @parameter
     */
    private List<Panel> panels = new ArrayList<Panel>();

    /**
     * A list of izpack 'custom actions', i.e. which are to be included within the install
     * artifact. Each 'custom action' must be defined as a project dependency with scope
     * 'runtime' and type 'jar'. The dependencies are copied by the plugin to the
     * 'bin/customActions' sub-directory of the izpack distro and each will be named
     * {@link CustomAction#getName()} (note that this is also the value used for the 'compiler'
     * and/or 'installer' and/or 'uninstaller' attribute on the izpack 'listener' element
     * within your installation file.
     * 
     * @parameter
     */
    private List<CustomAction> customActions = new ArrayList<CustomAction>();

    /**
     * A list of izpack 'resource' files which are to be included by plugin within the 'res'
     * subdirectory of the installation jar. Each 'resource' must be defined as a project
     * dependency with scope 'runtime.'
     * 
     * @parameter
     */
    private List<Resource> resources = new ArrayList<Resource>();

    /**
     * A list of izpack 'jars', i.e. which are to be included within the install artifact. Each
     * 'jar' must be defined as a project dependency with scope 'runtime' and type 'jar'. The
     * jars are copied by the plugin to the following path, i.e. this is the path you will use
     * within the corresponding 'jar' tag defined within the izpack installation file. The
     * example path assumes the dependency artifactId is 'foo'.
     * <p>
     * Note, these jars are not added to the COMPILATION classpath, i.e. they are only visible
     * at INSTALL time. If you have implemented e.g. a compiler listener or custom packager,
     * these should be defined as a project dependency with scope 'compile' to make them
     * visible at compile time.
     * 
     * ${project.build.directory}/izpack/lib/foo.jar
     * 
     * @parameter
     */
    private List<InstallationJar> installationJars = new ArrayList<InstallationJar>();

    /**
     * Extension type used to identify generated pack files to attach to project for install
     * phase. If for example you use the WebPackager to package your pack files, you'd specify
     * the gzip extension type 'gz' to attach the gzipped pack200 pack files to the project. If
     * you use the standard packager, you'd specificy 'jar'. The pack files will then be
     * installed along with project into repository by the install plugin (along with the
     * optional checksum files.)
     * 
     * @parameter
     * @required
     */
    private String packFileExtensionType;

    /*
     * (non-Javadoc)
     * @see org.apache.maven.plugin.AbstractMojo#execute()
     */
    public void execute() throws MojoExecutionException {

        getLog().info(" ======= IzPackMojo settings =======");
        getLog().info("workDirectory[" + workDirectory + "]");
        getLog().info("outputDirectory[" + outputDirectory + "]");
        getLog().info("installationFile[" + installationFile + "]");

        String input = null;
        try {
            input = interpolateFile(installationFile, project.getModel()).getAbsolutePath();
        } catch (Exception ex) {
            throw new MojoExecutionException("Error replacing maven vars. " + ex);
        }

        // the izpack distribution is defined as a plugin dependency.
        // unzip it to the projects build directory where it will be
        // used by the compiler

        File izpackDistro = null;
        for (DefaultArtifact defaultArtifact : pluginArtifacts) {
            String artifactID = defaultArtifact.getArtifactId();
            if (artifactID.equals("izpack-distro")) {
                izpackDistro = defaultArtifact.getFile();
                break;
            }
        }

        File izpackDirectory = new File(outputDirectory, "izpack");
        if (izpackDistro == null) {
            throw new MojoExecutionException("Missing plugin dependency 'izpack-distro'");
        } else {
            if (izpackDirectory.exists()) {
                getLog().warn("izpack distro already upacked to build dir? skipping.");
            } else {
                getLog().info("unpacking izpack distribution to build directory ...");
                FileUtils.unpackArchive(izpackDistro, izpackDirectory);
            }
        }

        CompilerConfig.setIzpackHome(izpackDirectory.getAbsolutePath());

        // copy the custom action dependencies
        for (CustomAction customAction : customActions) {
            String name = customAction.getName();
            String versionlessKey = customAction.getVersionlessKey();
            String classifer = customAction.getClassifier();
            if (!name.endsWith(".jar")) {
                name = name + ".jar";
            }
            File targetFile = new File(izpackDirectory, "bin/customActions/" + name);
            copyIzPackDependency(versionlessKey, classifer, "jar", targetFile);
        }

        // copy the custom action dependencies
        for (Resource resource : resources) {
            String name = resource.getName();
            String versionlessKey = resource.getVersionlessKey();
            String classifier = resource.getClassifier();
            String type = resource.getType();
            File targetFile = new File(izpackDirectory, "res/" + name);
            copyIzPackDependency(versionlessKey, classifier, type, targetFile);
        }

        // copy the panel dependencies
        for (Panel panel : panels) {
            String name = panel.getName();
            String versionlessKey = panel.getVersionlessKey();
            String classifier = panel.getClassifier();
            if (!name.endsWith(".jar")) {
                name = name + ".jar";
            }
            File targetFile = new File(izpackDirectory, "bin/panels/" + name);
            copyIzPackDependency(versionlessKey, classifier, "jar", targetFile);
        }

        // copy the jar dependencies
        for (InstallationJar jar : installationJars) {
            String name = jar.getName();
            String versionlessKey = jar.getVersionlessKey();
            String classifier = jar.getClassifier();
            if (!name.endsWith(".jar")) {
                name = name + ".jar";
            }
            File targetFile = new File(izpackDirectory, "lib/" + name);
            copyIzPackDependency(versionlessKey, classifier, "jar", targetFile);
        }

        // now we kick off the compilation

        File targetFile = new File(outputDirectory, finalName + "-install.jar");

        CompilerConfig compilerConfig = null;
        try {
            getLog().info("compiling izpack installation ...");
            String targetPath = targetFile.getAbsolutePath();
            compilerConfig = new CompilerConfig(input, outputDirectory, installationType, targetPath);
            compilerConfig.executeCompiler();
        } catch (Exception ex) {
            throw new MojoExecutionException("Error compiling izpack installation file. " + ex);
        }
        
        // add the generated install jar as a project attachment so that it can
        // be seen by other plugins, e.g. the install plugin and the gpg plugin,
        // etc ...
        
        projectHelper.attachArtifact(project, "jar", "install", targetFile);
        
        // add the generated pack files, if any, to project as attachments as
        // well
        
        String classifier = null;
        File outputDir = new File(outputDirectory);
        File[] list = outputDir.listFiles();
        for (File file : list) {
            String name = file.getName();
            int begin = name.indexOf("install.pack");
            int end = name.indexOf("." + packFileExtensionType);
            if (begin > -1 && end > -1) {
                classifier = name.substring(begin, end);
                projectHelper.attachArtifact(project, packFileExtensionType, classifier, file);
            }
        }

    }

    /*
     * replaces maven variables which may be defined within installation file with values
     * retrieved from project model
     */
    @SuppressWarnings("unchecked")
    private File interpolateFile(String input, Model model) throws Exception {

        if (new File(input).exists() == false) {
            throw new FileNotFoundException(new File(input).getAbsolutePath());
        }

        Map buildContext = new HashMap(System.getProperties());
        File projectDir = project.getBasedir();
        buildContext.put("basedir", project.getBasedir().getAbsolutePath());
        PathTranslator pathTranslator = new DefaultPathTranslator();
        Build build = model.getBuild();

        // MNG-1927, MNG-2124, MNG-3355:
        // If the build section is present and the project directory is non-null, we should
        // make sure interpolation of the directories below uses translated paths
        buildContext.put("build.directory", pathTranslator.alignToBaseDirectory(build.getDirectory(), projectDir));
        buildContext.put("build.outputDirectory", pathTranslator.alignToBaseDirectory(build.getOutputDirectory(),
                projectDir));
        buildContext.put("build.testOutputDirectory", pathTranslator.alignToBaseDirectory(build
                .getTestOutputDirectory(), projectDir));
        buildContext.put("build.sourceDirectory", pathTranslator.alignToBaseDirectory(build.getSourceDirectory(),
                projectDir));
        buildContext.put("build.testSourceDirectory", pathTranslator.alignToBaseDirectory(build
                .getTestSourceDirectory(), projectDir));

        String config = IOUtils.toString(new FileInputStream(input), "UTF-8");
        InterpolationHelper helper = new InterpolationHelper(getLog(), buildContext);
        config = helper.interpolate(config, model);
        File temp = File.createTempFile("izpack", ".xml");
        temp.deleteOnExit();
        FileOutputStream out = new FileOutputStream(temp);
        ByteArrayInputStream in = new ByteArrayInputStream(config.getBytes("UTF-8"));
        IOUtils.copy(in, out);
        out.flush();
        out.close();
        return temp;

    }

    private void copyIzPackDependency(String versionlessKey, String classifier, String type, File targetFile) throws MojoExecutionException {

        ScopeArtifactFilter runtimeFilter = new ScopeArtifactFilter(Artifact.SCOPE_RUNTIME);

        Iterator<?> itr = project.getArtifacts().iterator();
        File artifact = null;
        String artifactID = null;
        while (itr.hasNext()) {
            Artifact dependency = (Artifact)itr.next();
            if (!runtimeFilter.include(dependency)) {
                continue;
            }
            if (!dependency.getType().equals(type)) {
                continue;
            }
            if (classifier != null) {
                if (dependency.getClassifier() == null || !dependency.getClassifier().equals(classifier)) {
                    continue;
                }
            }
            String groupID = dependency.getGroupId();
            artifactID = dependency.getArtifactId();
            String key = ArtifactUtils.versionlessKey(groupID, artifactID);
            if (!key.equals(versionlessKey)) {
                continue;
            }
            artifact = dependency.getFile();
            break;
        }
        if (artifact == null) {
            getLog().warn(
                    "izpack resource "
                            + versionlessKey
                            + ":"
                            + classifier
                            + ":"
                            + type
                            + " not defined as a 'runtime' scoped project dependency.");
        } else {
            FileUtils.copyFile(artifact, targetFile);
        }

    }

}
