/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package org.bluestemsoftware.open.eoa.system.plugin.izpack.util;

import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.maven.model.Model;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.interpolation.ModelInterpolationException;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.introspection.ReflectionValueExtractor;

/*
 * Adapted from org.apache.maven.project.interpolation.RegexBasedModelInterpolator which was
 * released under the following license:
 * 
 * <p> Copyright 2001-2005 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

public class InterpolationHelper {

    private static final Pattern EXPRESSION_PATTERN = Pattern
            .compile("\\$\\{(pom\\.|project\\.|env\\.)?([^}]+)\\}");

    private Properties envars;
    private Map<String, String> buildContext;
    private Log log;

    public InterpolationHelper(Log log, Map<String, String> buildContext) {
        this.log = log;
        this.buildContext = buildContext;
        this.envars = System.getProperties();
    }

    public String interpolate(String src, Model model) throws ModelInterpolationException {

        boolean isSnapshotModel = (model.getVersion() == null ? model.getParent() == null ? "" : model.getParent()
                .getVersion() : model.getVersion()).indexOf("SNAPSHOT") >= 0;

        String result = src;
        Matcher matcher = EXPRESSION_PATTERN.matcher(result);
        while (matcher.find()) {
            String wholeExpr = matcher.group(0);
            String prefix = matcher.group(1);
            String realExpr = matcher.group(2);

            prefix = prefix == null ? "" : prefix;

            Object value = null;

            boolean isPomExpression = "pom.".equals(prefix) || "project.".equals(prefix);

            // Check for special expressions that should NOT be interpolated.
            // See DefaultProjectBuilder and MNG-2124/MNG-1927.
            if ((buildContext.get(realExpr) == null) && buildContext.containsKey(realExpr)) {
                continue;
            }

            if (isPomExpression) {
                value = getValueFromModel(realExpr, model, wholeExpr, log);
            }

            if (value == null) {
                value = buildContext.get(realExpr);
            }

            if (value == null) {
                value = model.getProperties().getProperty(realExpr);
            }

            if (value == null) {
                value = envars.getProperty(realExpr);
            }

            // Any expression, not just artifactId, version etc., but also scm.repository
            // were evaluated against the model, even if there is no prefix.
            // If the 2.1 strategy fails, try the legacy approach. If it yields a result, warn
            // for it.
            if ((value == null) && (prefix.length() == 0)) {
                value = getValueFromModel(realExpr, model, wholeExpr, log);

                if (isSnapshotModel && (value != null) && (log != null)) {
                    // TODO: Bring this back into the open once we're ready to deal with it.
                    log.debug("Expression: "
                            + wholeExpr
                            + " is missing its prefix. Instead, use ${pom."
                            + realExpr
                            + "} (model: "
                            + model.getId()
                            + ")");
                }
            }

            if (value != null) {
                // if the expression refers to itself, skip it.
                // replace project. expressions with pom. expressions to circumvent
                // self-referencing expressions using
                // the 2 different model expressions.
                if (StringUtils.replace(value.toString(), "${project.", "${pom.").indexOf(
                        StringUtils.replace(wholeExpr, "${project.", "${pom.")) > -1) {
                    throw new ModelInterpolationException(wholeExpr, "Expression value '"
                            + value
                            + "' references itself in '"
                            + model.getId()
                            + "'.");
                }

                result = StringUtils.replace(result, wholeExpr, value.toString());
                // could use:
                // result = matcher.replaceFirst( stringValue );
                // but this could result in multiple lookups of stringValue, and replaceAll is
                // not correct behaviour
                matcher.reset(result);
            }

        }

        return result;
    }

    private static Object getValueFromModel(String realExpr, Model model, String wholeExpr, Log log) {
        try {
            // NOTE: We've already trimmed off any leading expression parts like 'project.'
            // or 'pom.', and now we have to ensure that the ReflectionValueExtractor
            // doesn't try to do it again.
            return ReflectionValueExtractor.evaluate(realExpr, model, false);
        } catch (Exception e) {
            log.debug("POM interpolation cannot proceed with expression: " + wholeExpr + ". Skipping...", e);
            return null;
        }
    }

}
