/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.system.plugin.release;

import java.util.Iterator;
import java.util.List;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.bluestemsoftware.open.eoa.system.plugin.release.util.SCM;

/**
 * Iterates over projects in reactor and commits changes.
 * 
 * @aggregator
 * @goal checkin
 */
public class ReleaseCheckInMojo extends AbstractMojo {

    /**
     * @parameter expression="${reactorProjects}"
     * @required
     * @readonly
     */
    private List<?> reactorProjects;

    /*
     * (non-Javadoc)
     * @see org.apache.maven.plugin.AbstractMojo#execute()
     */
    public void execute() throws MojoExecutionException {
        for (Iterator<?> i = reactorProjects.iterator(); i.hasNext();) {
            processProject((MavenProject)i.next());
        }
    }

    /*
     * as of version 1.5 svn does not recurse into directories defined via the svn:externals
     * property on working copy root when performing a commit (status and updates do recurse.)
     * we must therefore mimic that behavior here. by convention, artifactId of poms defined on
     * root of each trunk end with '-trunk' (or -trunks which name of pom within the root of
     * working copy.) we issue the commit from there. note this operation is not atomic
     */
    private void processProject(MavenProject project) throws MojoExecutionException {

        boolean issueCommand = false;

        if (project.getPackaging().equals("pom")) {

            if (project.getArtifactId().equals("eoa-parent")) {
                issueCommand = true;
            }

            if (project.getArtifactId().endsWith("-trunk")) {
                issueCommand = true;
            }

            if (project.getArtifactId().endsWith("-trunks")) {
                issueCommand = true;
            }

        }

        if (issueCommand) {
            String message = System.getProperty("message");
            if (message == null) {
                throw new MojoExecutionException("Commit 'message' must be passed as system property.");
            }
            SCM.commit(project, getLog(), message);
        }

    }

}
