/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package org.bluestemsoftware.open.eoa.system.plugin.release.util;

import java.io.File;
import java.io.FileWriter;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.xerces.jaxp.DocumentBuilderFactoryImpl;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

public class SystemPathUpdater {

    public static void updateProject(MavenProject project, String fromVersion, String toVersion) throws MojoExecutionException {

        // we force the xerces impl. the default impl returned by new instance
        // method performed a four space indent rather than a two space indent

        Document pom = null;
        try {
            DocumentBuilderFactory dbf = new DocumentBuilderFactoryImpl();
            dbf.setNamespaceAware(true);
            DocumentBuilder builder = dbf.newDocumentBuilder();
            pom = builder.parse(project.getFile());
        } catch (Exception pe) {
            throw new MojoExecutionException("Error parsing pom.", pe);
        }

        boolean modified = false;

        Element projectElement = pom.getDocumentElement();

        Element dependenciesElement = DOMUtils.getChildElement(projectElement, new QName(Constants.POM_TNS, "dependencies"));
        if (dependenciesElement != null) {
            List<Element> kids = DOMUtils.getChildElements(dependenciesElement, new QName(Constants.POM_TNS, "dependency"));
            for (Element dependencyElement : kids) {
                Element systemPathElement = DOMUtils.getChildElement(dependencyElement, new QName(Constants.POM_TNS,
                        "systemPath"));
                if (systemPathElement != null) {
                    Text textNode = (Text)systemPathElement.getFirstChild();
                    String current = textNode.getData();
                    int begin = current.indexOf("/specification-eoa-api-") + 23;
                    int end = current.indexOf(".jar");
                    if (begin > -1 && end > -1) {
                        systemPathElement.removeChild(textNode);
                        String target = current.substring(begin, end);
                        if (target.equals(fromVersion)) {
                            String replacement = current.replace(target, toVersion);
                            systemPathElement.appendChild(pom.createTextNode(replacement));
                            modified = true;
                        }
                    }
                }
            }
        }

        if (modified) {
            try {
                FileWriter fileWriter = new FileWriter(project.getFile());
                fileWriter.write(DOMSerializer.serializeNode(pom));
                fileWriter.flush();
                fileWriter.close();
            } catch (Exception ie) {
                throw new MojoExecutionException("Error updating pom.", ie);
            }
        }

        // if we're processing an actual project, i.e. not a temp project
        // created within methods below, process the integration test poms

        if (!project.getArtifactId().equals(MavenProject.EMPTY_PROJECT_ARTIFACT_ID)) {
            updateTestPOMs(project, fromVersion, toVersion);
        }
        
        // if we're processing an archetype project, we need to update
        // tags within poms embedded within archetype

        if (project.getArtifactId().endsWith("archetype")) {
            updateArchetypePOMs(project, fromVersion, toVersion);
        }

        // if we're processing the assembly directory, we need to update
        // tags within system instance poms

        if (project.getArtifactId().equals("alakai-standalone-assembly")) {
            updateSystemPOMs(project, fromVersion, toVersion);
        }

    }

    private static void updateTestPOMs(MavenProject project, String fromVersion, String toVersion) throws MojoExecutionException {

        File baseDir = project.getFile().getParentFile();
        File testDir = new File(baseDir, "src/it/");
        if (!testDir.exists()) {
            return;
        }

        File[] suiteDirs = testDir.listFiles(new FileFilterImpl());
        for (File suiteDir : suiteDirs) {
            if (suiteDir.isDirectory()) {
                File pomFile = new File(suiteDir, "system/resources/alakai/etc/pom.xml");
                if (!pomFile.exists()) {
                    throw new MojoExecutionException("Project "
                            + project.getArtifactId()
                            + " missing test POM "
                            + pomFile.getAbsolutePath());
                }
                MavenProject temp = new MavenProject();
                temp.setFile(pomFile);
                updateProject(temp, fromVersion, toVersion);
            }
        }

    }

    private static void updateSystemPOMs(MavenProject project, String fromVersion, String toVersion) throws MojoExecutionException {

        File baseDir = project.getFile().getParentFile();
        File systemDir = new File(baseDir, "src/main/assembly/system/");
        if (!systemDir.exists()) {
            throw new MojoExecutionException("Project alakai-standalone-assembly missing subdirectory "
                    + systemDir.getAbsolutePath());
        }

        File[] systemInstanceDirs = systemDir.listFiles(new FileFilterImpl());
        for (File systemInstanceDir : systemInstanceDirs) {
            if (systemInstanceDir.isDirectory()) {
                recurseDirectory(systemInstanceDir, fromVersion, toVersion);
            }
        }

    }
    
    private static void updateArchetypePOMs(MavenProject project, String fromVersion, String toVersion) throws MojoExecutionException {

        File baseDir = project.getFile().getParentFile();
        File pomFile = new File(baseDir, "src/main/resources/pom.xml");
        if (!pomFile.exists()) {
            throw new MojoExecutionException("Archetype project missing file " + pomFile.getAbsolutePath());
        } else {
            MavenProject temp = new MavenProject();
            temp.setFile(pomFile);
            updateProject(temp, fromVersion, toVersion);
        }
        
        pomFile = new File(baseDir, "src/main/resources/src/it/suite1/system/resources/alakai/etc/pom.xml");
        if (!pomFile.exists()) {
            throw new MojoExecutionException("Archetype project missing file " + pomFile.getAbsolutePath());
        } else {
            MavenProject temp = new MavenProject();
            temp.setFile(pomFile);
            updateProject(temp, fromVersion, toVersion);
        }
        
    }
    
    private static void recurseDirectory(File systemInstanceDir, String fromVersion, String toVersion) throws MojoExecutionException {
        for (String name : systemInstanceDir.list()) {
            File f = new File(systemInstanceDir, name);
            if (f.isDirectory()) {
                if (name.equals("etc")) {
                    File pomFile = new File(systemInstanceDir, "etc/pom.xml");
                    if (!pomFile.exists()) {
                        throw new MojoExecutionException("Project alakai-standalone-assembly missing system POM "
                                + pomFile.getAbsolutePath());
                    }
                    MavenProject temp = new MavenProject();
                    temp.setFile(pomFile);
                    updateProject(temp, fromVersion, toVersion);
                } else {
                    recurseDirectory(f, fromVersion, toVersion);
                }
            }
        }
    }

}
