/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.plugin;

/**
 * Adapted from org.apache.maven.surefire.booter.SurefireBooter which was released under the
 * following license:
 * <p>
 * Copyright 2001-2005 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.maven.surefire.booter.SurefireExecutionException;
import org.apache.maven.surefire.booter.output.FileOutputConsumerProxy;
import org.apache.maven.surefire.booter.output.ForkingStreamConsumer;
import org.apache.maven.surefire.booter.output.StandardOutputConsumer;
import org.apache.maven.surefire.report.ConsoleReporter;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.cli.CommandLineUtils;
import org.codehaus.plexus.util.cli.Commandline;
import org.codehaus.plexus.util.cli.StreamConsumer;

/**
 * Forks jvm and executes surefire. If we're executing an integration test and client tests are
 * defined, forks a new jvm as a sub-process of system test jvm and executes client tests
 * against alakai, i.e. the same system instance against which system tests were run.
 */
public class SurefireBooter {

    static final String LINE_BREAK = System.getProperty("line.separator");

    private String groupID;
    private String artifactID;
    private String packaging;
    private String version;
    private List<Object[]> testSuites;
    private List<Object[]> reports;
    private Set<String> parentClasspath;
    private Set<String> sharedClasspath;
    private Set<String> deploymentClasspath;
    private Set<String> clientClasspath;
    private Set<String> surefireBooterClasspath;
    private Set<String> surefireClasspath;
    private File reportsDirectory;
    private String basedir;
    private String testClassesDirectory;
    private String workingDirectory;
    private boolean redirectTestOutputToFile;
    private boolean isIntegrationTest;
    private boolean isClientTest;

    private static String DEBUG = "-Xdebug -Xnoagent -Djava.compiler=NONE -Xrunjdwp:transport=dt_socket,server=y,suspend=y,address=5005";
    private static String SYSTEM_PROPERTY_PREFIX = "_system.";

    public SurefireBooter() {
        reports = new ArrayList<Object[]>();
        testSuites = new ArrayList<Object[]>();
    }

    public void addReport(String report, Object[] constructorParams) {
        reports.add(new Object[] { report, constructorParams });
    }

    public void addTestSuite(String suiteClassName, Object[] constructorParams) {
        testSuites.add(new Object[] { suiteClassName, constructorParams });
    }

    public void setDeploymentClasspath(Set<String> deploymentClasspath) {
        this.deploymentClasspath = deploymentClasspath;
    }

    public void setClientClasspath(Set<String> clientClasspath) {
        this.clientClasspath = clientClasspath;
    }

    public void setParentClasspath(Set<String> parentClasspath) {
        this.parentClasspath = parentClasspath;
    }

    public void setSharedClasspath(Set<String> sharedClasspath) {
        this.sharedClasspath = sharedClasspath;
    }

    public void setSurefireBooterClasspath(Set<String> surefireBooterClasspath) {
        this.surefireBooterClasspath = surefireBooterClasspath;
    }

    public void setSurefireClasspath(Set<String> surefireClasspath) {
        this.surefireClasspath = surefireClasspath;
    }

    public void setRedirectTestOutputToFile(boolean redirectTestOutputToFile) {
        this.redirectTestOutputToFile = redirectTestOutputToFile;
    }

    public void setReportsDirectory(File reportsDirectory) {
        this.reportsDirectory = reportsDirectory;
    }

    public void setWorkingDirectory(String workingDirectory) {
        this.workingDirectory = workingDirectory;
    }

    public void setArtifactID(String artifactID) {
        this.artifactID = artifactID;
    }

    public void setIntegrationTest(boolean isIntegrationTest) {
        this.isIntegrationTest = isIntegrationTest;
    }

    public void setGroupID(String groupID) {
        this.groupID = groupID;
    }

    public void setPackaging(String packaging) {
        this.packaging = packaging;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public void setTestClassesDirectory(String testClassesDirectory) {
        this.testClassesDirectory = testClassesDirectory;
    }

    public void setBasedir(String basedir) {
        this.basedir = basedir;
    }

    public void setClientTest(boolean isClientTest) {
        this.isClientTest = isClientTest;
    }

    public boolean run() throws Exception {

        Commandline cli = new Commandline();
        cli.setExecutable(System.getProperty("java.home") + File.separator + "bin" + File.separator + "java");

        // if we are running within context of an integration test, and debug system
        // property is set, we need to check to see if name is client debug property
        // or system debug so that we wait for jwdp attachment on correct jvm instance

        if (isIntegrationTest) {
            if (isClientTest) {
                if (System.getProperty("maven.surefire.it.client.debug") != null) {
                    cli.createArg().setLine(DEBUG);
                }
            } else {
                if (System.getProperty("maven.surefire.it.debug") != null) {
                    cli.createArg().setLine(DEBUG);
                }
            }
        } else {
            if (System.getProperty("maven.surefire.debug") != null) {
                cli.createArg().setLine(DEBUG);
            }
        }

        cli.createArg().setValue("-classpath");
        cli.createArg().setValue(StringUtils.join(surefireBooterClasspath.iterator(), File.pathSeparator));
        cli.createArg().setValue(SurefireBooter.class.getName());
        cli.setWorkingDirectory(workingDirectory);
        
        cli.createArg().setFile(writePropertiesFile());

        StreamConsumer out = getForkingStreamConsumer();
        StreamConsumer err = getForkingStreamConsumer();

        int exitCode = CommandLineUtils.executeCommandLine(cli, out, err);
        
        return exitCode == 0 ? true : false;

    }

    /*
     * executed by our main method within forked jvm.
     */
    @SuppressWarnings("unchecked")
    boolean runSurefire() throws SurefireExecutionException {

        boolean success = true;

        // define classloaders. note that the tests classloader is the deployment
        // classloader unless we are executing within context of a client test in
        // which case we set tests classloader to client classloader

        ClassLoader testsClassloader = null;
        ClassLoader surefireClassLoader = null;
        try {
            ClassLoader pcl = getParentClassLoader();
            if (isClientTest) {
                testsClassloader = getClientClassLoader();
            } else {
                ClassLoader scl = getSharedClassLoader(pcl);
                testsClassloader = getDeploymentClassLoader(pcl, scl);
            }
        } catch (Exception ex) {
            throw new SurefireExecutionException(ex.getMessage(), ex);
        }

        // write instance vars required by abstract configuration objects and by
        // tests which identify 'deployment under test' as system properties

        System.setProperty("eoa.system.dut.groupid", groupID);
        System.setProperty("eoa.system.dut.artifactid", artifactID);
        System.setProperty("eoa.system.dut.packaging", packaging);
        System.setProperty("eoa.system.dut.version", version);
        System.setProperty("basedir", basedir);
        System.setProperty("testClassesDirectory", testClassesDirectory);

        // if integration test, execute common configuration. check to see if user has
        // specified a custom configuration. if not, load configuration class provided
        // by test artifact, i.e. either system-test or client-test as appropriate

        ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();

        Object configuration = null;

        if (isIntegrationTest) {

            if (isClientTest) {
                System.out.println(LINE_BREAK + "running client configuration ...");
            } else {
                System.out.println(LINE_BREAK + "running system configuration ...");
            }
            Class<?> configurationClass = null;
            try {
                configuration = getConfiguration(testsClassloader);
                configurationClass = configuration.getClass();
                Method setUp = configurationClass.getMethod("setUp", new Class[] {});
                Thread.currentThread().setContextClassLoader(testsClassloader);
                success = (Boolean)setUp.invoke(configuration, new Object[] {});
            } catch (Exception ex) {
                if (ex.getMessage() == null) {
                    throw new SurefireExecutionException("Error executing configuration.", ex);
                } else {
                    throw new SurefireExecutionException("Error executing configuration. " + ex.getMessage(), ex);
                }
            } finally {
                Thread.currentThread().setContextClassLoader(currentClassLoader);
            }

            if (isClientTest == false) {

                // if configuration is used by an application test, set mock
                // provider collection as a system property such that it
                // is available to mock bindings. take care not to load any
                // system test classes with our definining classloader

                Class<?> tmp = null;
                try {
                    tmp = testsClassloader
                            .loadClass("org.bluestemsoftware.open.eoa.test.system.cfg.ApplicationConfiguration");
                } catch (ClassNotFoundException ce) {
                    throw new SurefireExecutionException("Error executing configuration. " + ce.getMessage(), ce);
                }

                if (tmp.isAssignableFrom(configurationClass)) {
                    Object proles;
                    try {
                        Method method = tmp.getMethod("getMockPartnerRoles", new Class[] {});
                        Thread.currentThread().setContextClassLoader(testsClassloader);
                        proles = method.invoke(configuration, new Object[] {});
                    } catch (Exception ex) {
                        if (ex.getMessage() == null) {
                            throw new SurefireExecutionException("Error executing configuration.", ex);
                        } else {
                            throw new SurefireExecutionException("Error executing configuration. "
                                    + ex.getMessage(), ex);
                        }
                    }
                    if (proles == null) {
                        proles = new HashMap<Object, Object>();
                    }
                    Map<String, Object> testProperties = (Map<String, Object>)System.getProperties().get(
                            "eoa.system.test.properties");
                    if (testProperties == null) {
                        throw new SurefireExecutionException(
                                "Error executing configuration. ApplicationConfiguration instance "
                                        + " failed to define test properties map as system property", null);
                    }
                    testProperties.put("mock.partner.roles", proles);
                }

            }

        }

        // if common configuration failed for any reason, we cannot continue
        // the test

        if (success == false) {
            System.out.println(LINE_BREAK + "error setting up configuration. see alakai.log for details");
            return success;
        }

        // execute the tests. if integration test flag is false, we
        // are executing unit tests. if client test flag is true we
        // are executing client tests, otherwise they're system tests

        if (isIntegrationTest) {
            if (isClientTest) {
                System.out.println(LINE_BREAK + "executing client tests");
            } else {
                System.out.println(LINE_BREAK + "executing system tests");
            }
        } else {
            System.out.println(LINE_BREAK + "executing unit tests");
        }

        try {
            String className = "org.apache.maven.surefire.Surefire";
            surefireClassLoader = getSurefireClassLoader(testsClassloader);
            Class<?> surefireClass = surefireClassLoader.loadClass(className);
            Class<?>[] signature = new Class[] { List.class, List.class, ClassLoader.class, ClassLoader.class };
            Method run = surefireClass.getMethod("run", signature);
            Thread.currentThread().setContextClassLoader(testsClassloader);
            Object[] args = new Object[] { reports, testSuites, surefireClassLoader, testsClassloader };
            success = (Boolean)run.invoke(surefireClass.newInstance(), args);
        } catch (Exception ex) {
            if (ex.getMessage() == null) {
                throw new SurefireExecutionException("Error executing tests.", ex);
            } else {
                throw new SurefireExecutionException("Error executing tests. " + ex.getMessage(), ex);
            }
        } finally {
            Thread.currentThread().setContextClassLoader(currentClassLoader);
        }

        // if we're unit testing or if we're integration testing and the
        // test failed, no sense in continuing

        if (isIntegrationTest == false || success == false) {
            return success;
        }

        // retrieve client test classes directory, which if not null
        // implies that we are within context of an integration test,
        // ie. system tests have run and system is still running. we
        // need to fork the jvm again for client tests, i.e. as a sub
        // process of this jvm

        String clientTestClassesDirectory = getClientTestClassesDirectory();

        if (clientTestClassesDirectory != null) {
            SurefireBooter surefireBooter = new SurefireBooter();
            surefireBooter.setClientTest(true);
            surefireBooter.setIntegrationTest(true);
            surefireBooter.setGroupID(groupID);
            surefireBooter.setArtifactID(artifactID);
            surefireBooter.setPackaging(packaging);
            surefireBooter.setVersion(version);
            surefireBooter.setTestClassesDirectory(clientTestClassesDirectory);
            surefireBooter.setRedirectTestOutputToFile(redirectTestOutputToFile);
            surefireBooter.setReportsDirectory(getClientTestReportsDirectory());
            surefireBooter.setParentClasspath(parentClasspath);
            surefireBooter.setSharedClasspath(sharedClasspath);
            surefireBooter.setDeploymentClasspath(deploymentClasspath);
            surefireBooter.setClientClasspath(clientClasspath);
            surefireBooter.setSurefireBooterClasspath(surefireBooterClasspath);
            surefireBooter.setSurefireClasspath(surefireClasspath);
            surefireBooter.setBasedir(basedir);
            surefireBooter.setWorkingDirectory(getClientWorkingDirectory());
            addClientTestSuiteDefinitions(surefireBooter);
            addClientReportDefinitions(surefireBooter);
            boolean temp;
            try {
                temp = surefireBooter.run();
            } catch (Exception ex) {
                if (ex.getMessage() == null) {
                    throw new SurefireExecutionException("Error executing client tests.", ex);
                } else {
                    throw new SurefireExecutionException("Error executing client tests. " + ex.getMessage(), ex);
                }
            }
            success = success && temp;
        } else {
            if (isIntegrationTest && isClientTest == false) {
                System.out.println(LINE_BREAK + "no client tests to execute" + LINE_BREAK);
            }
        }

        // if we're running within context of an integration test, destroy config
        // which could be system configuration or client configuration

        if (configuration != null) {
            if (isClientTest) {
                System.out.println(LINE_BREAK + "cleaning up client configuration");
            } else {
                System.out.println(LINE_BREAK + "cleaning up system configuration");
            }
            boolean temp = false;
            try {
                Class<?> configurationClass = configuration.getClass();
                Method destroy = configurationClass.getMethod("destroy", new Class[] {});
                Thread.currentThread().setContextClassLoader(testsClassloader);
                temp = (Boolean)destroy.invoke(configuration, new Object[] {});
            } catch (Exception ex) {
                if (ex.getMessage() == null) {
                    throw new SurefireExecutionException("Error destroying configuration.", ex);
                } else {
                    throw new SurefireExecutionException("Error destroying configuration. " + ex.getMessage(), ex);
                }
            } finally {
                Thread.currentThread().setContextClassLoader(currentClassLoader);
            }
            if (temp == false) {
                System.out.println(LINE_BREAK + "error cleaning up configuration. see log for details");
            }
            success = temp && success;
        }

        return success;

    }

    /*
     * writes instance variables to properties file the location of which is passed as a
     * command line argument to forked jvm
     */
    private File writePropertiesFile() throws Exception {

        Properties properties = new Properties();

        for (String key : System.getProperties().stringPropertyNames()) {
            properties.put(SYSTEM_PROPERTY_PREFIX + key, System.getProperty(key));
        }

        properties.setProperty("groupID", groupID);
        properties.setProperty("artifactID", artifactID);
        properties.setProperty("packaging", packaging);
        properties.setProperty("version", version);
        properties.setProperty("basedir", basedir);
        properties.setProperty("testClassesDirectory", testClassesDirectory);
        properties.setProperty("redirectTestOutputToFile", Boolean.toString(redirectTestOutputToFile));
        properties.setProperty("reportsDirectory", reportsDirectory.getAbsolutePath());
        properties.setProperty("workingDirectory", workingDirectory);
        properties.setProperty("isIntegrationTest", Boolean.toString(isIntegrationTest));
        properties.setProperty("isClientTest", Boolean.toString(isClientTest));

        String debugProperty = getDebugProperty();
        if (debugProperty != null) {
            properties.setProperty("debugProperty", debugProperty);
        }

        addPropertiesForTypeHolder(reports, properties, "report.");
        addPropertiesForTypeHolder(testSuites, properties, "testSuite.");

        int a = 0;
        for (String url : parentClasspath) {
            properties.setProperty("parentClasspathURL." + a++, url);
        }

        int b = 0;
        for (String url : sharedClasspath) {
            properties.setProperty("sharedClasspathURL." + b++, url);
        }

        int c = 0;
        for (String url : deploymentClasspath) {
            properties.setProperty("deploymentClasspathURL." + c++, url);
        }

        int d = 0;
        for (String url : clientClasspath) {
            properties.setProperty("clientClasspathURL." + d++, url);
        }

        int e = 0;
        for (String url : surefireClasspath) {
            properties.setProperty("surefireClasspathURL." + e++, url);
        }

        int f = 0;
        for (String url : surefireBooterClasspath) {
            properties.setProperty("surefireBooterClasspathURL." + f++, url);
        }

        File propertiesFile = File.createTempFile("surefire-", ".tmp");
        propertiesFile.deleteOnExit();
        FileOutputStream out = new FileOutputStream(propertiesFile);
        try {
            properties.store(out, "surefire");
        } finally {
            IOUtil.close(out);
        }

        return propertiesFile;
    }

    private void addPropertiesForTypeHolder(List<Object[]> typeHolderList, Properties properties, String propertyPrefix) {
        for (int i = 0; i < typeHolderList.size(); i++) {
            Object[] definition = typeHolderList.get(i);
            String className = (String)definition[0];
            Object[] params = (Object[])definition[1];
            properties.setProperty(propertyPrefix + i, className);
            if (params != null) {
                String paramProperty = params[0].toString();
                String typeProperty = params[0].getClass().getName();
                for (int j = 1; j < params.length; j++) {
                    paramProperty += "|";
                    typeProperty += "|";
                    if (params[j] != null) {
                        paramProperty += params[j].toString();
                        typeProperty += params[j].getClass().getName();
                    }
                }
                properties.setProperty(propertyPrefix + i + ".params", paramProperty);
                properties.setProperty(propertyPrefix + i + ".types", typeProperty);
            }
        }
    }

    private StreamConsumer getForkingStreamConsumer() {
        if (redirectTestOutputToFile) {
            FileOutputConsumerProxy proxy = null;
            proxy = new FileOutputConsumerProxy(new StandardOutputConsumer(), reportsDirectory);
            return new ForkingStreamConsumer(proxy);
        } else {
            return new ForkingStreamConsumer(new StandardOutputConsumer());
        }
    }

    /**
     * Main method invoked by sub-process, i.e. main method of a forked JVM instance.
     * 
     * @param args
     * @throws Throwable
     */
    public static void main(String[] args) {

        try {

            // retrieve properties file written by run method and passed as a
            // jvm argument

            Properties p = loadProperties(new File(args[0]));

            // instantiate booter instance and reconstitute instance variables
            // by reading from properties file

            SurefireBooter sb = new SurefireBooter();
            sb.groupID = p.getProperty("groupID");
            sb.artifactID = p.getProperty("artifactID");
            sb.packaging = p.getProperty("packaging");
            sb.version = p.getProperty("version");
            sb.basedir = p.getProperty("basedir");
            sb.testClassesDirectory = p.getProperty("testClassesDirectory");
            sb.redirectTestOutputToFile = Boolean.parseBoolean(p.getProperty("redirectTestOutputToFile"));
            sb.reportsDirectory = new File(p.getProperty("reportsDirectory"));
            sb.workingDirectory = p.getProperty("workingDirectory");
            sb.isIntegrationTest = Boolean.parseBoolean(p.getProperty("isIntegrationTest"));
            sb.isClientTest = Boolean.parseBoolean(p.getProperty("isClientTest"));

            sb.parentClasspath = new HashSet<String>();
            for (Enumeration<?> e = p.propertyNames(); e.hasMoreElements();) {
                String name = (String)e.nextElement();
                if (name.startsWith("parentClasspathURL.")) {
                    sb.parentClasspath.add(p.getProperty(name));
                }
            }

            sb.sharedClasspath = new HashSet<String>();
            for (Enumeration<?> e = p.propertyNames(); e.hasMoreElements();) {
                String name = (String)e.nextElement();
                if (name.startsWith("sharedClasspathURL.")) {
                    sb.sharedClasspath.add(p.getProperty(name));
                }
            }

            sb.surefireClasspath = new HashSet<String>();
            for (Enumeration<?> e = p.propertyNames(); e.hasMoreElements();) {
                String name = (String)e.nextElement();
                if (name.startsWith("surefireClasspathURL.")) {
                    sb.surefireClasspath.add(p.getProperty(name));
                }
            }

            sb.surefireBooterClasspath = new HashSet<String>();
            for (Enumeration<?> e = p.propertyNames(); e.hasMoreElements();) {
                String name = (String)e.nextElement();
                if (name.startsWith("surefireBooterClasspathURL.")) {
                    sb.surefireBooterClasspath.add(p.getProperty(name));
                }
            }

            sb.deploymentClasspath = new HashSet<String>();
            for (Enumeration<?> e = p.propertyNames(); e.hasMoreElements();) {
                String name = (String)e.nextElement();
                if (name.startsWith("deploymentClasspathURL.")) {
                    sb.deploymentClasspath.add(p.getProperty(name));
                }
            }

            sb.clientClasspath = new HashSet<String>();
            for (Enumeration<?> e = p.propertyNames(); e.hasMoreElements();) {
                String name = (String)e.nextElement();
                if (name.startsWith("clientClasspathURL.")) {
                    sb.clientClasspath.add(p.getProperty(name));
                }
            }

            for (Enumeration<?> e = p.propertyNames(); e.hasMoreElements();) {
                String name = (String)e.nextElement();
                if (name.startsWith("report.") && !name.endsWith(".params") && !name.endsWith(".types")) {
                    String className = p.getProperty(name);
                    String params = p.getProperty(name + ".params");
                    String types = p.getProperty(name + ".types");
                    sb.addReport(className, constructParamObjects(params, types));
                } else if (name.startsWith("testSuite.") && !name.endsWith(".params") && !name.endsWith(".types")) {
                    String className = p.getProperty(name);
                    String params = p.getProperty(name + ".params");
                    String types = p.getProperty(name + ".types");
                    sb.addTestSuite(className, constructParamObjects(params, types));
                }
            }

            String debugProperty = p.getProperty("debugProperty");
            if (debugProperty != null) {
                System.setProperty(debugProperty, "");
            }

            boolean success = sb.runSurefire();

            System.exit(success ? 0 : 1);

        } catch (Throwable t) {
            t.printStackTrace(System.err);
            System.exit(-1);
        }

    }

    private static Properties loadProperties(File file) throws IOException {
        
        Properties p = new Properties();
        if (file != null && file.exists()) {
            FileInputStream inStream = new FileInputStream(file);
            try {
                p.load(inStream);
            } finally {
                IOUtil.close(inStream);
            }
        }
        
        // restore system properties which existed prior to forking the jvm.
        // as a precaution, we do not overwrite any of the jvm generated
        // props
        
        for (String key : p.stringPropertyNames()) {
            if (key.startsWith(SYSTEM_PROPERTY_PREFIX)) {
                String value = p.getProperty(key);
                p.remove(key);
                key = key.substring(SYSTEM_PROPERTY_PREFIX.length());
                if (System.getProperty(key) == null) {
                    System.setProperty(key, value);
                }                
            }
        }
        
        return p;
    }

    private static Object[] constructParamObjects(String paramProperty, String typeProperty) {
        Object[] paramObjects = null;
        if (paramProperty != null) {

            // bit of a glitch that it needs to be done twice to do an odd number of vertical
            // bars (eg |||, |||||).
            String[] params = StringUtils.split(StringUtils.replace(StringUtils
                    .replace(paramProperty, "||", "| |"), "||", "| |"), "|");
            String[] types = StringUtils.split(StringUtils.replace(StringUtils.replace(typeProperty, "||", "| |"),
                    "||", "| |"), "|");

            paramObjects = new Object[params.length];

            for (int i = 0; i < types.length; i++) {
                if (types[i].trim().length() == 0) {
                    params[i] = null;
                } else if (types[i].equals(String.class.getName())) {
                    paramObjects[i] = params[i];
                } else if (types[i].equals(File.class.getName())) {
                    paramObjects[i] = new File(params[i]);
                } else if (types[i].equals(ArrayList.class.getName())) {
                    paramObjects[i] = processStringList(params[i]);
                } else if (types[i].equals(Boolean.class.getName())) {
                    paramObjects[i] = Boolean.valueOf(params[i]);
                } else if (types[i].equals(Integer.class.getName())) {
                    paramObjects[i] = Integer.valueOf(params[i]);
                } else {
                    // TODO: could attempt to construct with a String constructor if needed
                    throw new IllegalArgumentException("Unknown parameter type: " + types[i]);
                }
            }
        }
        return paramObjects;
    }

    private static List<String> processStringList(String stringList) {
        String sl = stringList;

        if (sl.startsWith("[") && sl.endsWith("]")) {
            sl = sl.substring(1, sl.length() - 1);
        }

        List<String> list = new ArrayList<String>();

        String[] stringArray = StringUtils.split(sl, ",");

        for (int i = 0; i < stringArray.length; i++) {
            list.add(stringArray[i].trim());
        }
        return list;
    }

    private ClassLoader getParentClassLoader() throws Exception {

        List<URL> constituents = new ArrayList<URL>();
        for (String constituent : parentClasspath) {
            constituents.add(new URL(constituent));
        }

        ClassLoader parentClassLoader = new URLClassLoader(toArray(constituents));
        System.getProperties().put("eoa.system.parent.classloader", parentClassLoader);
        return parentClassLoader;

    }

    private ClassLoader getSharedClassLoader(ClassLoader parent) throws Exception {

        List<URL> constituents = new ArrayList<URL>();
        for (String constituent : sharedClasspath) {
            constituents.add(new URL(constituent));
        }

        ClassLoader scl = null;
        try {
            String className = "org.bluestemsoftware.specification.eoa.SharedClassLoader";
            Class<?> clazz = parent.loadClass(className);
            Constructor<?> ctr = clazz.getConstructor(new Class[] { URL[].class, ClassLoader.class });
            scl = (ClassLoader)ctr.newInstance(new Object[] { toArray(constituents), parent });
        } catch (Exception ex) {
            if (ex.getMessage() == null) {
                throw new SurefireExecutionException("Error creating shared classloader.", ex);
            } else {
                throw new SurefireExecutionException("Error creating shared classloader. " + ex.getMessage(), ex);
            }
        }

        scl.setDefaultAssertionStatus(true);
        System.getProperties().put("eoa.system.shared.classloader", scl);
        return scl;

    }

    private ClassLoader getDeploymentClassLoader(ClassLoader parent, ClassLoader scl) throws Exception {

        List<URL> constituents = new ArrayList<URL>();
        for (String constituent : deploymentClasspath) {
            constituents.add(new URL(constituent));
        }

        ClassLoader dcl = null;
        try {
            String className = "org.bluestemsoftware.specification.eoa.DeploymentClassLoader";
            Class<?> clazz = parent.loadClass(className);
            Constructor<?> ctr = clazz.getConstructor(new Class[] { String.class, URL[].class, ClassLoader.class,
                    scl.getClass() });
            String id = null;
            if (packaging.equals("eoa-component")) {
                id = groupID + "/" + artifactID + "/" + version + "/" + packaging;
            } else {
                id = groupID + "/" + artifactID + "/" + packaging;
            }
            dcl = (ClassLoader)ctr.newInstance(new Object[] { id, toArray(constituents), parent, scl });
        } catch (Exception ex) {
            if (ex.getMessage() == null) {
                throw new SurefireExecutionException("Error creating deployment classloader.", ex);
            } else {
                throw new SurefireExecutionException("Error creating deployment classloader. " + ex.getMessage(),
                        ex);
            }
        }

        dcl.setDefaultAssertionStatus(true);
        System.getProperties().put("eoa.system.deployment.classloader", dcl);
        return dcl;

    }

    private ClassLoader getClientClassLoader() throws Exception {
        List<URL> constituents = new ArrayList<URL>();
        for (String constituent : clientClasspath) {
            constituents.add(new URL(constituent));
        }
        return new URLClassLoader(toArray(constituents));
    }

    private ClassLoader getSurefireClassLoader(ClassLoader parent) throws Exception {
        List<URL> constituents = new ArrayList<URL>();
        for (String constituent : surefireClasspath) {
            constituents.add(new URL(constituent));
        }
        ClassLoader surefireClassLoader = new URLClassLoader(toArray(constituents), parent);
        surefireClassLoader.setDefaultAssertionStatus(true);
        return surefireClassLoader;
    }

    private URL[] toArray(List<URL> constituents) {
        URL[] array = new URL[constituents.size()];
        Iterator<URL> itr = constituents.iterator();
        for (int i = 0; itr.hasNext(); i++) {
            array[i] = itr.next();
        }
        return array;
    }

    private Object getConfiguration(ClassLoader cl) throws Exception {

        // retreive provider config file formatted according to java's service
        // provider loading mechanism. the path used depends upon whether we
        // are executing a system test or a client test

        String pcp = null;
        if (isClientTest) {
            pcp = "META-INF/services/org.bluestemsoftware.open.eoa.test.client.cfg.ClientConfiguration";
        } else {
            pcp = "META-INF/services/org.bluestemsoftware.open.eoa.test.system.cfg.SystemConfiguration";
        }

        Class<?> clazz = null;

        URL resource = cl.getResource(pcp);
        if (resource != null) {
            List<String> classNames = new ArrayList<String>();
            BufferedReader reader = null;
            try {
                reader = new BufferedReader(new InputStreamReader(resource.openStream(), "UTF-8"));
                for (String line = reader.readLine(); line != null; line = reader.readLine()) {
                    if (line.indexOf('#') >= 0) {
                        line = line.substring(0, line.indexOf('#')).trim();
                    }
                    if (line.length() > 0) {
                        classNames.add(line);
                    }
                }
            } finally {
                if (reader != null) {
                    try {
                        reader.close();
                    } catch (IOException ignore) {
                    }
                }
            }
            try {
                clazz = cl.loadClass(classNames.get(0));
            } catch (RuntimeException re) {
                throw new SurefireExecutionException("Error loading service provider implementation of "
                        + pcp
                        + ". ", re);
            }
        }

        // if no custom configuration is provided by user, load the default configuration
        // object. the instance used depends upon whether we're running within client jvm
        // or system jvm

        if (clazz == null) {
            String sc = null;
            if (isClientTest) {
                sc = "org.bluestemsoftware.open.eoa.test.client.cfg.AbstractClientConfiguration";
            } else {
                sc = "org.bluestemsoftware.open.eoa.test.system.cfg.AbstractSystemConfiguration";
            }
            clazz = cl.loadClass(sc);
        }

        return clazz.newInstance();

    }

    private String getClientTestClassesDirectory() {
        if (!isIntegrationTest) {
            // we're executing unit tests. client
            // tests are not applicable
            return null;
        }
        if (isClientTest) {
            // we've already forked the jvm and are
            // in process of running client tests
            return null;
        }
        File system = new File(testClassesDirectory).getParentFile();
        File suite = system.getParentFile();
        File client = new File(suite, "client/");
        if (!client.exists()) {
            return null;
        } else {
            return new File(client, "test-classes/").getAbsolutePath();
        }
    }

    private File getClientTestReportsDirectory() {
        File suite = reportsDirectory.getParentFile();
        return new File(suite, "client/");
    }

    private String getClientWorkingDirectory() {
        File system = new File(workingDirectory);
        File suite = system.getParentFile();
        return new File(suite, "client/").getAbsolutePath();
    }

    private void addClientTestSuiteDefinitions(SurefireBooter surefireBooter) {
        List<String> includes = new ArrayList<String>(Arrays.asList(new String[] { "**/Test*.java",
                "**/*Test.java", "**/*TestCase.java" }));
        List<String> excludes = new ArrayList<String>(Arrays.asList(new String[] { "**/Abstract*Test.java",
                "**/Abstract*TestCase.java", "**/*$*" }));
        String className = "org.apache.maven.surefire.junit4.JUnit4DirectoryTestSuite";
        File clientTestClassesDirectory = new File(getClientTestClassesDirectory());
        Object[] parms = new Object[] { clientTestClassesDirectory, includes, excludes };
        surefireBooter.addTestSuite(className, parms);
    }

    private void addClientReportDefinitions(SurefireBooter surefireBooter) {
        for (Object[] report : reports) {
            String name = (String)report[0];
            Object[] parms = (Object[])report[1];
            if (name.equals(ConsoleReporter.class.getName())) {
                Boolean trimStackTrace = (Boolean)parms[0];
                surefireBooter.addReport(name, new Object[] { trimStackTrace });
            } else {
                Boolean trimStackTrace = (Boolean)parms[1];
                File testReportsDirectory = getClientTestReportsDirectory();
                surefireBooter.addReport(name, new Object[] { testReportsDirectory, trimStackTrace });
            }
        }
    }

    private String getDebugProperty() {
        if (System.getProperty("maven.surefire.debug") != null) {
            return "maven.surefire.debug";
        }
        if (System.getProperty("maven.surefire.it.debug") != null) {
            return "maven.surefire.it.debug";
        }
        if (System.getProperty("maven.surefire.it.client.debug") != null) {
            return "maven.surefire.it.client.debug";
        }
        return null;
    }

}
