/**
 *
 *  Copyright 2004 Jeremy Boynes
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.tranql.connector.derby;

import java.sql.SQLException;
import java.util.Properties;

import org.apache.derby.jdbc.ClientBaseDataSource;
import org.apache.derby.jdbc.ClientDataSource;
import org.tranql.connector.NoExceptionsAreFatalSorter;
import org.tranql.connector.jdbc.AbstractLocalDataSourceMCF;

/**
 * ManagedConnectionFactory that wraps a Derby ClientDataSource for use with a remote server.
 *
 * @version $Revision: 227 $ $Date: 2005-08-07 17:38:59 -0500 (Sun, 07 Aug 2005) $
 */
public class ClientLocalMCF extends AbstractLocalDataSourceMCF {
    private final ClientDataSource ds;

    /**
     * Default constructor for a Derby Client Local DataSource.
     */
    public ClientLocalMCF() {
        super(new ClientDataSource(), new NoExceptionsAreFatalSorter(), false);
        ds = (ClientDataSource) super.dataSource;
    }

    /**
     * Return the user name used to establish the connection.
     *
     * @return the user name used to establish the connection
     */
    public String getUserName() {
        return ds.getUser();
    }

    /**
     * Set the user name used establish the connection.
     * This value is used if no connection information is supplied by the application
     * when attempting to create a connection.
     *
     * @param user the user name used to establish the connection; may be null
     */
    public void setUserName(String user) {
        ds.setUser(user);
    }

    /**
     * Return the password credential used to establish the connection.
     *
     * @return the password credential used to establish the connection
     */
    public String getPassword() {
        try {
            return ClientBaseDataSource.getPassword(ds.getProperties());
        } catch (SQLException e) {
            throw new AssertionError();
        }
    }

    /**
     * Set the user password credential establish the connection.
     * This value is used if no connection information is supplied by the application
     * when attempting to create a connection.
     *
     * @param password the password credential used to establish the connection; may be null
     */
    public void setPassword(String password) {
        ds.setPassword(password);
    }

    /**
     * Return the name of the server to connect to.
     *
     * @return the name of the server to connect to
     */
    public String getServerName() {
        return ds.getServerName();
    }

    /**
     * Set the name of the server to connect to.
     *
     * @param serverName the name of the server to connect to
     */
    public void setServerName(String serverName) {
        ds.setServerName(serverName);
    }

    /**
     * Return the port number to connect to.
     *
     * @return the port number to connect to
     */
    public Integer getPortNumber() {
        return new Integer(ds.getPortNumber());
    }

    /**
     * Set the port number to connect to.
     *
     * @param port the port number to connect to
     */
    public void setPortNumber(Integer port) {
        ds.setPortNumber(port.intValue());
    }

    /**
     * Return the database name.
     *
     * @return the database name
     */
    public String getDatabaseName() {
        return ds.getDatabaseName();
    }

    /**
     * Set the database name.
     *
     * @param name the database name
     */
    public void setDatabaseName(String name) {
        ds.setDatabaseName(name);
    }

    /**
     * Return whether the database should be created if it does not already exist.
     *
     * @return true if the database should be created
     */
    public Boolean getCreateDatabase() {
        try {
            return Boolean.valueOf(ds.getProperties().containsKey("create"));
        } catch (SQLException e) {
            throw new AssertionError();
        }
    }

    /**
     * Set whether the database should be created if it does not already exist.
     *
     * @param create TRUE if the database should be created
     */
    public void setCreateDatabase(Boolean create) {
        try {
            Properties props = ds.getProperties();
            if (create.booleanValue()) {
                props.setProperty("create", "create");
            } else {
                props.remove("create");
            }
        } catch (SQLException e) {
            throw new AssertionError();
        }
    }

    /**
     * Return whether the client should retrieve the text of messages from the server.
     *
     * @return true if message text should be retrieved from the server
     */
    public Boolean getRetrieveMessageText() {
        return Boolean.valueOf(ds.getRetrieveMessageText());
    }

    /**
     * Set whether the client should retrieve the text of messages from the server.
     *
     * @param b whether the client should retrieve the text of messages from the server
     */
    public void setRetrieveMessageText(Boolean b) {
        ds.setRetrieveMessageText(b.booleanValue());
    }
}
