/**
 *
 *  Copyright 2004 Jeremy Boynes
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.tranql.connector.derby;

import org.apache.derby.jdbc.EmbeddedDataSource;
import org.tranql.connector.NoExceptionsAreFatalSorter;
import org.tranql.connector.jdbc.AbstractLocalDataSourceMCF;

/**
 * ManagedConnectionFactory that wraps a Derby EmbeddedDataSource for use in-VM.
 *
 * @version $Revision: 297 $ $Date: 2005-12-06 23:52:14 -0600 (Tue, 06 Dec 2005) $
 */
public class EmbeddedLocalMCF extends AbstractLocalDataSourceMCF {
    private final EmbeddedDataSource ds;

    /**
     * Default constructor for a Derby Embedded Local DataSource.
     */
    public EmbeddedLocalMCF() {
        super(new EmbeddedDataSource(), new NoExceptionsAreFatalSorter(), false);
        ds = (EmbeddedDataSource) super.dataSource;
    }

    /**
     * Return the user name used to establish the connection.
     *
     * @return the user name used to establish the connection
     */
    public String getUserName() {
        return ds.getUser();
    }

    /**
     * Set the user name used establish the connection.
     * This value is used if no connection information is supplied by the application
     * when attempting to create a connection.
     *
     * @param user the user name used to establish the connection; may be null
     */
    public void setUserName(String user) {
        ds.setUser(user);
    }

    /**
     * Return the password credential used to establish the connection.
     *
     * @return the password credential used to establish the connection
     */
    public String getPassword() {
        return ds.getPassword();
    }

    /**
     * Set the user password credential establish the connection.
     * This value is used if no connection information is supplied by the application
     * when attempting to create a connection.
     *
     * @param password the password credential used to establish the connection; may be null
     */
    public void setPassword(String password) {
        ds.setPassword(password);
    }

    /**
     * Return the database name.
     *
     * @return the database name
     */
    public String getDatabaseName() {
        return ds.getDatabaseName();
    }

    /**
     * Set the database name.
     *
     * @param name the database name
     */
    public void setDatabaseName(String name) {
        ds.setDatabaseName(name);
    }

    /**
     * Return whether the database should be created if it does not already exist.
     *
     * @return true if the database should be created
     */
    public Boolean getCreateDatabase() {
        return Boolean.valueOf("create".equals(ds.getCreateDatabase()));
    }

    /**
     * Set whether the database should be created if it does not already exist.
     *
     * @param create TRUE if the database should be created
     */
    public void setCreateDatabase(Boolean create) {
        ds.setCreateDatabase(create.booleanValue() ? "create" : null);
    }

    /**
     *     Set this property if one wishes to shutdown the database identified by databaseName.
     *     
     * @param shutdown - if set to the string "shutdown", this data source will shutdown the 
     *                   database if it is running.
     */
    public void setShutdownDatabase(String shutDown) {
        ds.setShutdownDatabase(shutDown);
    }

    /**
     *   @return String "shutdown" if shutdown is set, or null if not
     */
    public String getShutdownDatabase() {
        return ds.getShutdownDatabase();
    }

}
