
package org.blufin.core.auth.core;
import lombok.Getter;
import org.blufin.base.interfaces.Initializable;
import org.blufin.core.auth.client.ClientDataContainer;
import org.blufin.core.auth.data.RequestHeaders;
import org.blufin.core.auth.enums.AuthMethod;
import org.blufin.core.auth.exceptions.AuthenticationException;
import org.blufin.core.auth.handlers.*;
import org.blufin.core.server.config.ApiConfig;
import org.blufin.sdk.embedded.base.EmbeddedSDK;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.text.MessageFormat;
import java.util.Optional;
@Getter
@Service
public class AuthMethodInvoker extends Initializable<Boolean> {
    @Autowired
    EmbeddedSDK embeddedSDK;
    private PublicHandler publicHandler;
    private InternalHandler internalHandler;
    private InternalAccountHandler internalAccountHandler;
    private InternalAccountUserHandler internalAccountUserHandler;
    private OAuthAccountHandler oAuthAccountHandler;
    private OAuthAccountUserHandler oAuthAccountUserHandler;
    private ClientOriginatingAccountHandler clientOriginatingAccountHandler;
    private ClientOriginatingAccountUserHandler clientOriginatingAccountUserHandler;
    public final Optional<ClientDataContainer> invoke(AuthMethod authMethod, RequestHeaders requestHeaders) {
        try {
            ClientDataContainer clientDataContainer;
            switch (authMethod) {
                case NONE:
                    clientDataContainer = publicHandler.authenticate(requestHeaders);
                    break;
                case INTERNAL:
                    clientDataContainer = internalHandler.authenticate(requestHeaders);
                    break;
                case INTERNAL_ACCOUNT:
                    clientDataContainer = internalAccountHandler.authenticate(requestHeaders);
                    break;
                case INTERNAL_ACCOUNT_USER:
                    clientDataContainer = internalAccountUserHandler.authenticate(requestHeaders);
                    break;
                case CLIENT_ORIGINATING_ACCOUNT:
                    clientDataContainer = clientOriginatingAccountHandler.authenticate(requestHeaders);
                    break;
                case CLIENT_ORIGINATING_ACCOUNT_USER:
                    clientDataContainer = clientOriginatingAccountUserHandler.authenticate(requestHeaders);
                    break;
                case OAUTH_ACCOUNT:
                    clientDataContainer = oAuthAccountHandler.authenticate(requestHeaders);
                    break;
                case OAUTH_ACCOUNT_USER:
                    clientDataContainer = oAuthAccountUserHandler.authenticate(requestHeaders);
                    break;
                default:
                    throw new AuthenticationException(MessageFormat.format("Unhandled AuthMethod: {0}", authMethod.toString()));
            }
            return (clientDataContainer == null) ? Optional.empty() : Optional.of(clientDataContainer);
        } catch (AuthenticationException e) {
            return Optional.empty();
        }
    }
    public void init(ApiConfig apiConfig) {
        initializeClass(true);
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.NONE)) {
            this.publicHandler = new PublicHandler();
        }
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.INTERNAL)) {
            this.internalHandler = new InternalHandler(
                    apiConfig.getAuthorizationHeader()
            );
        }
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.INTERNAL_ACCOUNT)) {
            this.internalAccountHandler = new InternalAccountHandler(
                    apiConfig.getAuthorizationHeader(),
                    embeddedSDK.getEmbeddedAccountService()
            );
        }
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.INTERNAL_ACCOUNT_USER)) {
            this.internalAccountUserHandler = new InternalAccountUserHandler(
                    apiConfig.getAuthorizationHeader(),
                    embeddedSDK.getEmbeddedAccountService(),
                    embeddedSDK.getEmbeddedUserService()
            );
        }
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.OAUTH_ACCOUNT)) {
            this.oAuthAccountHandler = new OAuthAccountHandler(
                    embeddedSDK.getEmbeddedThirdPartyDeveloperService(),
                    embeddedSDK.getEmbeddedThirdPartyApplicationService(),
                    embeddedSDK.getEmbeddedAccountService()
            );
        }
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.OAUTH_ACCOUNT_USER)) {
            this.oAuthAccountUserHandler = new OAuthAccountUserHandler(
                    embeddedSDK.getEmbeddedThirdPartyDeveloperService(),
                    embeddedSDK.getEmbeddedThirdPartyApplicationService(),
                    embeddedSDK.getEmbeddedAccountService(),
                    embeddedSDK.getEmbeddedUserService()
            );
        }
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.CLIENT_ORIGINATING_ACCOUNT)) {
            this.clientOriginatingAccountHandler = new ClientOriginatingAccountHandler(
                    apiConfig.getAuthorizationHeader(),
                    embeddedSDK.getEmbeddedAccountService()
            );
        }
        if (apiConfig.getAuthorizationLevel().getAuthMethods().contains(AuthMethod.CLIENT_ORIGINATING_ACCOUNT_USER)) {
            this.clientOriginatingAccountUserHandler = new ClientOriginatingAccountUserHandler(
                    apiConfig.getAuthorizationHeader(),
                    embeddedSDK.getEmbeddedAccountService(),
                    embeddedSDK.getEmbeddedUserService()
            );
        }
    }
}
