
package org.blufin.core.auth.core;
import org.blufin.core.auth.data.ControllerAnnotations;
import org.blufin.core.auth.data.RequestHeaders;
import org.blufin.core.auth.enums.AuthMethod;
import org.blufin.core.auth.exceptions.UnmetHeaderRequisiteException;
import java.text.MessageFormat;
public class AuthMethodResolver {
    public static final AuthMethod resolve(RequestHeaders requestHeaders, ControllerAnnotations controllerAnnotations) throws UnmetHeaderRequisiteException {
        if (!controllerAnnotations.isPubliclyAccessible() && controllerAnnotations.isSecured()) {
            if (controllerAnnotations.isOAuth()) {
                if (controllerAnnotations.isAccountAware() && !controllerAnnotations.isUserAware()) {
                    if (hasOAuthAccountPrerequisites(requestHeaders)) {
                        return AuthMethod.OAUTH_ACCOUNT;
                    }
                    throw new UnmetHeaderRequisiteException();
                } else if (controllerAnnotations.isAccountAware() && controllerAnnotations.isUserAware()) {
                    if (hasOAuthAccountUserPrerequisites(requestHeaders)) {
                        return AuthMethod.OAUTH_ACCOUNT_USER;
                    }
                    throw new UnmetHeaderRequisiteException();
                }
            } else {
                if (!controllerAnnotations.isAccountAware() && !controllerAnnotations.isUserAware()) {
                    if (hasInternalPrerequisites(requestHeaders)) {
                        return AuthMethod.INTERNAL;
                    }
                    throw new UnmetHeaderRequisiteException();
                } else if (controllerAnnotations.isAccountAware() && !controllerAnnotations.isUserAware()) {
                    if (hasInternalAccountPrerequisites(requestHeaders)) {
                        return AuthMethod.INTERNAL_ACCOUNT;
                    } else if (hasClientOriginatingAccountPrerequisites(requestHeaders)) {
                        return AuthMethod.CLIENT_ORIGINATING_ACCOUNT;
                    }
                    throw new UnmetHeaderRequisiteException();
                } else if (controllerAnnotations.isAccountAware() && controllerAnnotations.isUserAware()) {
                    if (hasInternalAccountUserPrerequisites(requestHeaders)) {
                        return AuthMethod.INTERNAL_ACCOUNT_USER;
                    } else if (hasClientOriginatingAccountUserPrerequisites(requestHeaders)) {
                        return AuthMethod.CLIENT_ORIGINATING_ACCOUNT_USER;
                    }
                    throw new UnmetHeaderRequisiteException();
                }
            }
        } else if (controllerAnnotations.isPubliclyAccessible() && !controllerAnnotations.isSecured()) {
            if (!controllerAnnotations.isAccountAware() && !controllerAnnotations.isUserAware() && !controllerAnnotations.isOAuth()) {
                return AuthMethod.NONE;
            }
        }
        throw new RuntimeException(MessageFormat.format("Resource has invalid Annotations: {0}", controllerAnnotations.toString()));
    }
    private static final boolean hasInternalPrerequisites(RequestHeaders requestHeaders) {
        return requestHeaders.getAuthInternal() != null;
    }
    private static final boolean hasInternalAccountPrerequisites(RequestHeaders requestHeaders) {
        return requestHeaders.getAuthInternal() != null && requestHeaders.getAuthAccountName() != null;
    }
    private static final boolean hasInternalAccountUserPrerequisites(RequestHeaders requestHeaders) {
        return requestHeaders.getAuthInternal() != null && requestHeaders.getAuthAccountName() != null && requestHeaders.getAuthAccountUser() != null;
    }
    private static final boolean hasClientOriginatingAccountPrerequisites(RequestHeaders requestHeaders) {
        return requestHeaders.getAuthIdentifier() != null && requestHeaders.getAuthAccountName() != null && requestHeaders.getAuthToken() != null;
    }
    private static final boolean hasClientOriginatingAccountUserPrerequisites(RequestHeaders requestHeaders) {
        return requestHeaders.getAuthIdentifier() != null && requestHeaders.getAuthAccountName() != null && requestHeaders.getAuthAccountUser() != null && requestHeaders.getAuthToken() != null;
    }
    private static final boolean hasOAuthAccountPrerequisites(RequestHeaders requestHeaders) {
        return requestHeaders.getOAuthDev() != null && requestHeaders.getOAuthApp() != null && requestHeaders.getOAuthAccountToken() != null;
    }
    private static final boolean hasOAuthAccountUserPrerequisites(RequestHeaders requestHeaders) {
        return requestHeaders.getOAuthDev() != null && requestHeaders.getOAuthApp() != null && requestHeaders.getOAuthAccountToken() != null && requestHeaders.getOAuthAccountUser() != null;
    }
}
