
package org.blufin.core.auth.handlers;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.blufin.base.exceptions.BlufinNotImplementedException;
import org.blufin.core.auth.client.ClientDataContainer;
import org.blufin.core.auth.client.ClientDataFactory;
import org.blufin.core.auth.data.RequestHeaders;
import org.blufin.core.auth.enums.AuthMethod;
import org.blufin.core.auth.exceptions.AuthenticationException;
import org.blufin.sdk.embedded.dto.app.EmbeddedUser;
import org.blufin.sdk.embedded.dto.common.EmbeddedAccount;
import org.blufin.sdk.embedded.dto.common.EmbeddedThirdPartyApplication;
import org.blufin.sdk.embedded.dto.common.EmbeddedThirdPartyDeveloper;
import org.blufin.sdk.embedded.service.EmbeddedAccountService;
import org.blufin.sdk.embedded.service.EmbeddedThirdPartyApplicationService;
import org.blufin.sdk.embedded.service.EmbeddedThirdPartyDeveloperService;
import org.blufin.sdk.embedded.service.EmbeddedUserService;
import org.blufin.sdk.enums.AuthHeader;
import org.blufin.sdk.response.ApiResponse;
import java.text.MessageFormat;
@Setter(AccessLevel.PROTECTED)
public abstract class AbstractAuthenticationHandler {
    @Getter(AccessLevel.PROTECTED)
    private String authorizationHeader;
    private EmbeddedThirdPartyDeveloperService thirdPartyDeveloperService;
    private EmbeddedThirdPartyApplicationService thirdPartyApplicationService;
    private EmbeddedAccountService accountService;
    private EmbeddedUserService userService;
    public abstract ClientDataContainer authenticate(RequestHeaders headers) throws Exception;
    protected EmbeddedThirdPartyApplication getThirdPartyApplication(String developerHash, String applicationHash) throws AuthenticationException {
        try {
            ApiResponse<EmbeddedThirdPartyApplication> applicationApiResponse = thirdPartyApplicationService.get()
                    .where().hash().is(applicationHash)
                    .entries(1)
                    .execute();
            if (applicationApiResponse.getPagination().getTotalEntries() != 1) {
                throw new AuthenticationException();
            }
            ApiResponse<EmbeddedThirdPartyDeveloper> developerApiResponse = thirdPartyDeveloperService.get()
                    .where().id().is(applicationApiResponse.getBody().get(0).getThirdPartyDeveloperId())
                    .where().hash().is(developerHash)
                    .entries(1)
                    .execute();
            if (developerApiResponse.getPagination().getTotalEntries() == 0) {
                throw new AuthenticationException();
            }
            return applicationApiResponse.getBody().get(0);
        } catch (Exception e) {
            throw new RuntimeException(MessageFormat.format("Couldn''t resolve valid 3rd-Party application for header(s) {0}: {1} and {2}: {3}", AuthHeader.X_OAUTH_APP.toString(), applicationHash, AuthHeader.X_OAUTH_DEV.toString(), developerHash));
        }
    }
    protected EmbeddedAccount getAccount(String accountNameOrToken, AuthMethod authMethod) throws AuthenticationException {
        try {
            String accountName = getAccountName(accountNameOrToken, authMethod);
            try {
                ApiResponse<EmbeddedAccount> apiResponse = accountService.get()
                        .where().accountName().is(accountName)
                        .entries(1)
                        .execute();
                ClientDataFactory.setAccount(apiResponse.getBody().get(0).getAccountName());
                return apiResponse.getBody().get(0);
            } catch (Exception e) {
                throw new AuthenticationException(MessageFormat.format("Couldn''t resolve Account: {0}", accountName));
            }
        } catch (Exception e) {
            throw new AuthenticationException(e.getMessage());
        }
    }
    protected EmbeddedUser getUser(String accountNameOrToken, int userId, AuthMethod authMethod) throws AuthenticationException {
        try {
            String accountName = getAccountName(accountNameOrToken, authMethod);
            try {
                ApiResponse<EmbeddedUser> apiResponse = userService.get()
                        .as(accountName)
                        .where().id().is(userId)
                        .entries(1)
                        .execute();
                return apiResponse.getBody().get(0);
            } catch (Exception e) {
                throw new RuntimeException(MessageFormat.format("Couldn''t resolve valid User for Account: {0} and User ID: {1}", accountName, userId));
            }
        } catch (Exception e) {
            throw new AuthenticationException(e.getMessage());
        }
    }
    private String getAccountName(String accountNameOrToken, AuthMethod authMethod) throws AuthenticationException {
        switch (authMethod) {
            case INTERNAL_ACCOUNT:
            case INTERNAL_ACCOUNT_USER:
            case CLIENT_ORIGINATING_ACCOUNT:
            case CLIENT_ORIGINATING_ACCOUNT_USER:
                return accountNameOrToken;
            case OAUTH_ACCOUNT:
            case OAUTH_ACCOUNT_USER:
                try {
                    throw new BlufinNotImplementedException();
                } catch (Exception e) {
                    throw new AuthenticationException(MessageFormat.format("Couldn''t resolve valid Account for header {0}: {1}", AuthHeader.X_OAUTH_ACCOUNT_TOKEN.toString(), accountNameOrToken));
                }
            default:
                throw new RuntimeException(MessageFormat.format("AuthMethod → {0} should never require an Account.", authMethod.toString()));
        }
    }
}
