
package org.blufin.core.cloud;
import com.amazonaws.auth.profile.ProfileCredentialsProvider;
import com.amazonaws.services.sqs.AmazonSQSAsync;
import com.amazonaws.services.sqs.AmazonSQSAsyncClient;
import com.amazonaws.services.sqs.model.*;
import org.blufin.base.utils.UtilsLogger;
import org.blufin.core.cloud.queue.MessageClientInterface;
import org.blufin.core.cloud.queue.MessageDeliveryHandler;
import org.blufin.core.cloud.queue.MessageIncoming;
import org.blufin.core.cloud.queue.MessageOutgoing;
import org.blufin.core.cloud.queue.MessageQueueConfig;
import org.blufin.jackson.Jackson;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
public class AwsSQSClient implements MessageClientInterface {
    private static final int MAXIMUM_MESSAGES = 3;
    private static final int MAXIMUM_POLL_TIME = 20;
    private static final int SECONDS_TO_SLEEP_AFTER_RETRIEVAL = 2;
    private final AmazonSQSAsync amazonSQS;
    private final String queueURL;
    private final ReceiveMessageRequest messageRequest;
    public AwsSQSClient(MessageQueueConfig config) throws Exception {
        ReceiveMessageRequest messageRequest = new ReceiveMessageRequest();
        messageRequest.setMaxNumberOfMessages(MAXIMUM_MESSAGES);
        messageRequest.setWaitTimeSeconds(AwsSQSClient.MAXIMUM_POLL_TIME);
        messageRequest.setQueueUrl(config.getHost());
        this.amazonSQS = new AmazonSQSAsyncClient(new ProfileCredentialsProvider().getCredentials());
        this.queueURL = config.getHost();
        this.messageRequest = messageRequest;
    }
    @Override
    public void produce(MessageOutgoing message) throws Exception {
        String messageBody = Jackson.getObjectMapper().writeValueAsString(message);
        amazonSQS.sendMessage(new SendMessageRequest(queueURL, messageBody));
    }
    @Override
    @SuppressWarnings("InfiniteLoopStatement")
    public void consume(MessageDeliveryHandler messageDeliveryHandler) {
        Thread messagePoller = new Thread() {
            @Override
            public void run() {
                while (true) {
                    try {
                        List<MessageIncoming> messages = getMessages();
                        if (messages.size() > 0) {
                            List<MessageIncoming> successfulMessages = messageDeliveryHandler.handleDelivery(messages);
                            if (successfulMessages.size() > 0) {
                                Map<String, String> messagesToDelete = new HashMap<>();
                                for (MessageIncoming successfulMessage : successfulMessages) {
                                    messagesToDelete.put(successfulMessage.getId(), successfulMessage.getReceiptHandle());
                                }
                                deleteMessages(messagesToDelete);
                            }
                            if (sleepTime() > 0) {
                                Thread.sleep(sleepTime());
                            }
                        }
                    } catch (InterruptedException e) {
                        UtilsLogger.alertDeveloper("The thread which polls for Amazon SQS messages was interrupted.", e);
                    } catch (Exception e) {
                        UtilsLogger.alertDeveloper("Something went wrong whilst polling for Amazon SQS messages.", e);
                    }
                }
            }
        };
        messagePoller.start();
    }
    private List<MessageIncoming> getMessages() {
        List<MessageIncoming> sqsMessages = new ArrayList<>();
        try {
            List<Message> messages = amazonSQS.receiveMessage(messageRequest).getMessages();
            if (messages.size() > 0) {
                for (Message message : messages) {
                    MessageOutgoing sqsMessageContent = Jackson.getObjectMapper().readValue(message.getBody(), MessageOutgoing.class);
                    MessageIncoming sqsMessage = new MessageIncoming(
                            sqsMessageContent.getType(),
                            sqsMessageContent.getBody(),
                            sqsMessageContent.getIp(),
                            message.getMessageId(),
                            message.getReceiptHandle()
                    );
                    sqsMessages.add(sqsMessage);
                }
            }
        } catch (Exception e) {
            UtilsLogger.alertDeveloper("Failed to retrieve message(s) from Amazon SQS.", e);
        }
        return sqsMessages;
    }
    private void deleteMessages(Map<String, String> messageIds) {
        if (messageIds.size() == 0) {
            UtilsLogger.alertDeveloper("An empty HashMap (of messages) was passed to " + AwsSQSClient.class.getSimpleName() + ".deleteMessages().");
            return;
        }
        DeleteMessageBatchResult deleteMessageBatchResult = amazonSQS.deleteMessageBatch(deleteMessageBatchRequest(messageIds));
        if (deleteMessageBatchResult.getFailed().size() > 1) {
            List<String> errorEntries = new ArrayList<>();
            deleteMessageBatchResult.getFailed().forEach(errorEntry -> errorEntries.add(errorEntry.getMessage()));
            UtilsLogger.alertDeveloper("Failed to delete message(s) from Amazon SQS. Received the following error(s):", errorEntries.toString());
        }
    }
    private DeleteMessageBatchRequest deleteMessageBatchRequest(Map<String, String> messages) {
        List<DeleteMessageBatchRequestEntry> deleteMessageBatchRequestEntries = new ArrayList<>();
        if (messages.size() > 0) {
            messages.forEach((id, receiptHandle) -> {
                DeleteMessageBatchRequestEntry deleteMessageBatchRequestEntry = new DeleteMessageBatchRequestEntry();
                deleteMessageBatchRequestEntry.setId(id);
                deleteMessageBatchRequestEntry.setReceiptHandle(receiptHandle);
                deleteMessageBatchRequestEntries.add(deleteMessageBatchRequestEntry);
            });
        }
        DeleteMessageBatchRequest deleteMessageBatchRequest = new DeleteMessageBatchRequest();
        deleteMessageBatchRequest.setEntries(deleteMessageBatchRequestEntries);
        deleteMessageBatchRequest.setQueueUrl(queueURL);
        return deleteMessageBatchRequest;
    }
    private int sleepTime() {
        return SECONDS_TO_SLEEP_AFTER_RETRIEVAL * 1000;
    }
}
