
package org.blufin.core.server.db;
import com.zaxxer.hikari.HikariDataSource;
import org.blufin.base.exceptions.BlufinServerException;
import org.blufin.core.server.db.base.AbstractDataSource;
import org.blufin.core.server.db.data.DataSourceConfiguration;
import org.blufin.core.server.helper.DataSourceConfigurationTransformer;
import org.blufin.core.utils.UtilsThread;
import org.blufin.sdk.embedded.dto.common.EmbeddedAccount;
import org.blufin.sdk.embedded.dto.config.EmbeddedDb;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.sql.Connection;
import java.text.MessageFormat;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
@Service
public class AccountAwareDataSource extends AbstractDataSource {
    @Autowired
    DataSourceConfigurationTransformer dataSourceConfigurationTransformer;
    private volatile ConcurrentHashMap<Integer, HikariDataSource> dataSourceMap = new ConcurrentHashMap<>();
    private volatile ConcurrentHashMap<String, Integer> accountToDataSourceMap = new ConcurrentHashMap<>();
    public boolean refresh(List<? extends EmbeddedDb> databaseList, List<? extends EmbeddedAccount> accountList) throws BlufinServerException {
        try {
            refreshInProgress = true;
            UtilsThread.sleep(2000);
            ConcurrentHashMap<Integer, HikariDataSource> newDataSourceMap = new ConcurrentHashMap<>();
            ConcurrentHashMap<String, Integer> newAccountToDataSourceMap = new ConcurrentHashMap<>();
            if (dataSourceMap != null) {
                for (Map.Entry<Integer, HikariDataSource> entry : dataSourceMap.entrySet()) {
                    entry.getValue().close();
                }
            }
            if (databaseList != null && databaseList.size() > 0) {
                for (EmbeddedDb database : databaseList) {
                    DataSourceConfiguration configuration = dataSourceConfigurationTransformer.transform(database);
                    newDataSourceMap.put(database.getId(), build(configuration));
                }
            }
            accountList.stream().filter(account -> !newAccountToDataSourceMap.containsKey(account.getAccountName())).forEach(account -> newAccountToDataSourceMap.put(account.getAccountName(), account.getDbId()));
            dataSourceMap = newDataSourceMap;
            accountToDataSourceMap = newAccountToDataSourceMap;
            return true;
        } catch (Exception e) {
            throw new BlufinServerException(MessageFormat.format("Failed to refresh account aware data-source(s): {0}", e.getMessage()));
        } finally {
            refreshInProgress = false;
        }
    }
    public Connection getConnection(String accountName) throws BlufinServerException {
        try {
            waitWhileRefreshing();
            Connection connection = dataSourceMap.get(accountToDataSourceMap.get(accountName)).getConnection();
            connection.setCatalog(accountName);
            connection.setAutoCommit(false);
            return connection;
        } catch (Exception e) {
            throw new BlufinServerException(MessageFormat.format("AccountAwareDataSource.getConnection() encountered an Exception with message: {0}", e.getMessage()));
        }
    }
}
