
package org.blufin.core.server.deserialization;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeType;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.blufin.base.constants.Constants;
import org.blufin.base.enums.DataType;
import org.blufin.base.exceptions.BlufinAlertDeveloperException;
import org.blufin.base.exceptions.BlufinClientException;
import org.blufin.base.utils.UtilsDate;
import org.blufin.base.utils.UtilsLogger;
import org.blufin.base.utils.UtilsString;
import org.blufin.core.auth.client.ClientData;
import org.blufin.jackson.Jackson;
import org.blufin.sdk.base.AbstractMetaData;
import org.blufin.sdk.base.MetaDataException;
import org.blufin.sdk.normalization.DataNormalizationException;
import org.blufin.sdk.normalization.DataNormalizer;
import org.blufin.sdk.normalization.TerminologyNormalizer;
import org.blufin.sdk.response.AckError;
import org.blufin.sdk.response.AckResolver;
import org.blufin.sdk.response.AckWarning;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
public class JsonMappingValidator {
    public static final int LIST_LENGTH = "_list".length();
    public static final String ERROR_PAYLOAD_EMPTY = "Received empty payload. Expected → {0}";
    public static final String ERROR_PAYLOAD_EMPTY_LIST = "Received empty List. Expected → List of Objects ({0})";
    public static final String ERROR_PAYLOAD_INVALID_TYPE = "Invalid payload. Expected: {0} . Got → {1}";
    public static final String ERROR_PAYLOAD_INVALID_TYPE_WITH_VALUE = "Invalid payload. Expected: {0} . Got → {1} ({2})";
    public static final String ERROR_TRANSIENT_NULL_SUFFIX = " (not even as NULL)";
    public static void validate(JsonNode incomingNode, JsonNode outgoingNode, AckResolver ackResolver, AbstractMetaData metaData, TerminologyNormalizer.Type expectedNode) {
        validate(incomingNode, outgoingNode, ackResolver, metaData, expectedNode, Constants.EMPTY_STRING);
    }
    private static void validate(JsonNode incomingNode, JsonNode outgoingNode, AckResolver ackResolver, AbstractMetaData metaData, TerminologyNormalizer.Type expectedNode, String tree) {
        if (incomingNode == null) {
            throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_EMPTY, expectedNode)));
        } else {
            validateNodeType(incomingNode, expectedNode);
        }
        if (incomingNode.isArray()) {
            if (incomingNode.size() == 0 && !metaData.isNestedObjectList()) {
                throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_EMPTY_LIST, UtilsString.underscoreToUpperCamel(metaData.getTable()))));
            }
            for (int i = 0; i < incomingNode.size(); i++) {
                JsonNode nestedNode = Jackson.getObjectMapper().createObjectNode();
                validate(incomingNode.get(i), nestedNode, ackResolver, metaData, tree);
                ((ArrayNode) outgoingNode).add(nestedNode);
            }
        } else {
            validate(incomingNode, outgoingNode, ackResolver, metaData, tree);
        }
    }
    private static void validate(JsonNode incomingNode, JsonNode outgoingNode, AckResolver ackResolver, AbstractMetaData metaData, String tree) throws BlufinClientException {
        validateNodeType(incomingNode, TerminologyNormalizer.Type.OBJECT);
        Iterator<Map.Entry<String, JsonNode>> fields = incomingNode.fields();
        List<String> unrecognizedProperties = new ArrayList<>();
        List<String> requiredFields = metaData.getRequiredFieldsIncludingContainers();
        while (fields.hasNext()) {
            Map.Entry<String, JsonNode> entry = fields.next();
            String fieldName = entry.getKey();
            String fieldNameUnderScore = UtilsString.camelToUnderscore(entry.getKey());
            String fieldNameWithTree = tree.equals(Constants.EMPTY_STRING) ? fieldName : tree + Constants.PERIOD + fieldName;
            JsonNode value = entry.getValue();
            try {
                if (!metaData.containsField(fieldNameUnderScore)) {
                    throw new MetaDataException();
                }
                if (requiredFields.contains(fieldNameUnderScore)) {
                    requiredFields.remove(fieldNameUnderScore);
                }
                if (metaData.isTransient(fieldNameUnderScore)) {
                    ackResolver.addWarning(AckWarning.TRANSIENT_PROPERTY_IGNORED, fieldNameWithTree, ERROR_TRANSIENT_NULL_SUFFIX);
                    continue;
                }
                if (value.isNull()) {
                    if (metaData.getType(fieldNameUnderScore).equals(DataType.OBJECT_LIST)) {
                        ackResolver.addError(AckError.JSON_NULL_NOT_ALLOWED_FOR_LIST, fieldNameWithTree);
                    }
                    continue;
                }
                if ((AbstractMetaData.CONTAINER_TYPES.contains(metaData.getType(fieldNameUnderScore)) && !value.isContainerNode()) ||
                        (!AbstractMetaData.CONTAINER_TYPES.contains(metaData.getType(fieldNameUnderScore)) && value.isContainerNode())) {
                    ackResolver.addError(AckError.JSON_TYPE_MISMATCH, fieldNameWithTree, metaData.getType(fieldNameUnderScore).toString(), value.getNodeType().toString());
                    continue;
                }
                switch (metaData.getType(fieldNameUnderScore)) {
                    case BOOLEAN:
                        ((ObjectNode) outgoingNode).put(fieldName, DataNormalizer.normalizeBoolean(value.asText(), fieldNameWithTree, ackResolver));
                        if (!value.isBoolean()) {
                            ackResolver.addWarning(AckWarning.JSON_TYPE_MISMATCH, fieldNameWithTree, JsonNodeType.BOOLEAN.toString(), value.getNodeType().toString(), value.asText());
                        }
                        break;
                    case DATE:
                        ((ObjectNode) outgoingNode).put(fieldName, UtilsDate.convertLocalDateToString(DataNormalizer.normalizeDate(value.asText(), fieldNameWithTree, ackResolver)));
                        break;
                    case DATETIME:
                        ((ObjectNode) outgoingNode).put(fieldName, String.valueOf(DataNormalizer.normalizeDateTime(value.asText(), fieldNameWithTree, ackResolver, ClientData.getTimeZone())));
                        break;
                    case DATETIME_INSERT:
                        throw new BlufinAlertDeveloperException(MessageFormat.format("{0} should always be transient and this statement should never be reached.", DataType.DATETIME_INSERT.toString()));
                    case DATETIME_UPDATE:
                        throw new BlufinAlertDeveloperException(MessageFormat.format("{0} should always be transient and this statement should never be reached.", DataType.DATETIME_UPDATE.toString()));
                    case DECIMAL:
                        String valueAsString;
                        if (value.isBigDecimal()) {
                            valueAsString = value.decimalValue().toPlainString();
                        } else {
                            valueAsString = value.asText();
                        }
                        ((ObjectNode) outgoingNode).put(fieldName, DataNormalizer.normalizeDecimal(valueAsString, fieldNameWithTree, ackResolver, metaData.getDecimalDistribution(fieldNameUnderScore)).toPlainString());
                        if (!value.isNumber()) {
                            ackResolver.addWarning(AckWarning.JSON_TYPE_MISMATCH, fieldNameWithTree, JsonNodeType.NUMBER.toString(), value.getNodeType().toString(), value.asText());
                        }
                        break;
                    case ENUM:
                    case ENUM_CUSTOM:
                    case ENUM_SYSTEM:
                        ((ObjectNode) outgoingNode).put(fieldName, DataNormalizer.normalizeStringEnum(value.asText(), fieldNameWithTree, ackResolver, metaData.getEnumValues(fieldNameUnderScore)));
                        break;
                    case INT:
                    case INT_AUTO:
                        ((ObjectNode) outgoingNode).put(fieldName, DataNormalizer.normalizeInt(value.asText(), fieldNameWithTree, ackResolver));
                        if (!value.isNumber()) {
                            ackResolver.addWarning(AckWarning.JSON_TYPE_MISMATCH, fieldNameWithTree, JsonNodeType.NUMBER.toString(), value.getNodeType().toString(), value.asText());
                        }
                        break;
                    case INT_TINY:
                        ((ObjectNode) outgoingNode).put(fieldName, DataNormalizer.normalizeIntTiny(value.asText(), fieldNameWithTree, ackResolver));
                        if (!value.isNumber()) {
                            ackResolver.addWarning(AckWarning.JSON_TYPE_MISMATCH, fieldNameWithTree, JsonNodeType.NUMBER.toString(), value.getNodeType().toString(), value.asText());
                        }
                        break;
                    case INT_SMALL:
                        ((ObjectNode) outgoingNode).put(fieldName, DataNormalizer.normalizeIntSmall(value.asText(), fieldNameWithTree, ackResolver));
                        if (!value.isNumber()) {
                            ackResolver.addWarning(AckWarning.JSON_TYPE_MISMATCH, fieldNameWithTree, JsonNodeType.NUMBER.toString(), value.getNodeType().toString(), value.asText());
                        }
                        break;
                    case INT_BIG:
                        ((ObjectNode) outgoingNode).put(fieldName, DataNormalizer.normalizeIntBig(value.asText(), fieldNameWithTree, ackResolver));
                        if (!value.isNumber()) {
                            ackResolver.addWarning(AckWarning.JSON_TYPE_MISMATCH, fieldNameWithTree, JsonNodeType.NUMBER.toString(), value.getNodeType().toString(), value.asText());
                        }
                        break;
                    case TEXT:
                    case TEXT_LONG:
                    case VARCHAR:
                        ((ObjectNode) outgoingNode).put(fieldName, value.asText());
                        if (!value.isTextual()) {
                            ackResolver.addWarning(AckWarning.JSON_TYPE_MISMATCH, fieldNameWithTree, JsonNodeType.STRING.toString(), value.getNodeType().toString(), "\"" + value.asText() + "\"");
                        }
                        break;
                    case OBJECT:
                        if (!value.isObject()) {
                            ackResolver.addError(AckError.JSON_TYPE_MISMATCH, fieldNameWithTree, DataType.OBJECT.toString(), value.getNodeType().toString());
                            continue;
                        }
                        JsonNode nestedOutgoingObjectNode = Jackson.getObjectMapper().createObjectNode();
                        validate(value, nestedOutgoingObjectNode, ackResolver, metaData.getNestedMetaData(fieldNameUnderScore), TerminologyNormalizer.Type.OBJECT, fieldNameWithTree);
                        ((ObjectNode) outgoingNode).set(fieldName, nestedOutgoingObjectNode);
                        break;
                    case OBJECT_LIST:
                        if (!value.isArray()) {
                            ackResolver.addError(AckError.JSON_TYPE_MISMATCH, fieldNameWithTree, DataType.OBJECT_LIST.toString(), value.getNodeType().toString());
                            continue;
                        }
                        JsonNode nestedArrayNode = Jackson.getObjectMapper().createArrayNode();
                        for (int i = 0; i < value.size(); i++) {
                            JsonNode nestedNode = Jackson.getObjectMapper().createObjectNode();
                            validate(value.get(i), nestedNode, ackResolver, metaData.getNestedMetaData(fieldNameUnderScore.substring(0, (fieldNameUnderScore.length() - LIST_LENGTH))), fieldNameWithTree);
                            ((ArrayNode) nestedArrayNode).add(nestedNode);
                        }
                        ((ObjectNode) outgoingNode).set(fieldName, nestedArrayNode);
                        break;
                    case OBJECT_LINK:
                        continue;
                    default:
                        throw new BlufinAlertDeveloperException(MessageFormat.format("Unhandled DataType: {0}", metaData.getType(fieldNameUnderScore)));
                }
            } catch (MetaDataException e) {
                unrecognizedProperties.add(fieldNameWithTree);
            } catch (DataNormalizationException e) {
            }
        }
        if (unrecognizedProperties.size() > 0) {
            ackResolver.addError(AckError.JSON_UNRECOGNIZED_PROPERTIES, String.join(", ", unrecognizedProperties));
        }
        if (requiredFields.size() > 0) {
            List<String> requiredFieldsCamelCased = new ArrayList<>();
            requiredFields.forEach(requiredField -> requiredFieldsCamelCased.add(tree.equals(Constants.EMPTY_STRING) ? UtilsString.underscoreToLowerCamel(requiredField) : tree + Constants.PERIOD + UtilsString.underscoreToLowerCamel(requiredField)));
            ackResolver.addError(AckError.JSON_MISSING_PROPERTIES, String.join(", ", requiredFieldsCamelCased));
        }
    }
    private static void validateNodeType(JsonNode rootNode, TerminologyNormalizer.Type expectedNode) throws BlufinClientException {
        switch (rootNode.getNodeType()) {
            case ARRAY:
                if (expectedNode != TerminologyNormalizer.Type.LIST) {
                    throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_INVALID_TYPE, expectedNode, TerminologyNormalizer.type(rootNode.getNodeType().toString()))));
                }
                break;
            case OBJECT:
                if (expectedNode != TerminologyNormalizer.Type.OBJECT) {
                    throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_INVALID_TYPE, expectedNode, TerminologyNormalizer.type(rootNode.getNodeType().toString()))));
                }
                break;
            case NUMBER:
                throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_INVALID_TYPE_WITH_VALUE, expectedNode, TerminologyNormalizer.type(rootNode.getNodeType().toString()), rootNode.toString().trim().replaceAll("^\\s*\"", "").replaceAll("\"\\s*$", ""))));
            case STRING:
                throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_INVALID_TYPE_WITH_VALUE, expectedNode, TerminologyNormalizer.type(rootNode.getNodeType().toString()), "'" + rootNode.toString().trim().replaceAll("^\\s*\"", "").replaceAll("\"\\s*$", "") + "'")));
            case BINARY:
            case BOOLEAN:
            case MISSING:
            case NULL:
            case POJO:
                throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_INVALID_TYPE, expectedNode, TerminologyNormalizer.type(rootNode.getNodeType().toString()))));
            default:
                UtilsLogger.alertDeveloper(MessageFormat.format("Un-handled JSON NodeType: {0}", rootNode.getNodeType()));
                throw new BlufinClientException(AckError.JSON_ERROR.toString(MessageFormat.format(ERROR_PAYLOAD_INVALID_TYPE, expectedNode, TerminologyNormalizer.type(rootNode.getNodeType().toString()))));
        }
    }
}
