
package org.blufin.core.server.helper;
import lombok.Setter;
import org.blufin.base.enums.SchemaType;
import org.blufin.base.exceptions.BlufinAlertDeveloperException;
import org.blufin.base.exceptions.BlufinServerException;
import org.blufin.base.helper.Pair;
import org.blufin.core.auth.core.AuthInitializationLevel;
import org.blufin.core.server.db.AccountAwareDataSource;
import org.blufin.sdk.embedded.base.EmbeddedSDK;
import org.blufin.sdk.embedded.dto.common.EmbeddedAccount;
import org.blufin.sdk.embedded.dto.config.EmbeddedDb;
import org.blufin.sdk.embedded.dto.config.EmbeddedProject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.List;
@Service
public final class AccountAwareDataSourceRefresher {
    @Autowired
    EmbeddedSDK embeddedSDK;
    @Autowired
    AccountAwareDataSource accountAwareDataSource;
    @Setter
    private String projectName;
    @Setter
    private AuthInitializationLevel authInitializationLevel;
    public Pair<List<? extends EmbeddedDb>, List<? extends EmbeddedAccount>> refresh() throws BlufinServerException {
        if (Arrays.asList(
                AuthInitializationLevel.INTERNAL_ACCOUNT_USER,
                AuthInitializationLevel.OAUTH_ACCOUNT_USER).contains(authInitializationLevel)) {
            List<EmbeddedDb> databaseList = embeddedSDK.getEmbeddedDbService().get()
                    .where().projectId().is(getProjectId())
                    .where().schemaType().is(SchemaType.APP)
                    .execute()
                    .getBody();
            List<EmbeddedAccount> accountList = embeddedSDK.getEmbeddedAccountService().get()
                    .execute()
                    .getBody();
            accountAwareDataSource.refresh(databaseList, accountList);
            return Pair.of(databaseList, accountList);
        } else {
            throw new BlufinAlertDeveloperException(MessageFormat.format("This code should never be reached. Authentication Level does not support this operation: {0}", authInitializationLevel.toString()));
        }
    }
    private int getProjectId() {
        List<EmbeddedProject> projectList = embeddedSDK.getEmbeddedProjectService().get()
                .where().projectName().is(projectName)
                .execute()
                .getBody();
        if (projectList.size() != 1) {
            throw new BlufinAlertDeveloperException(MessageFormat.format("{0} expected exactly one project named ''{1}'' to exist in the database, instead got → {2}", AccountAwareDataSourceRefresher.class.getSimpleName(), projectName, projectList.size()));
        }
        return projectList.get(0).getId();
    }
}
