package org.blufin.core.server.rest;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.blufin.base.enums.Environment;
import org.blufin.base.exceptions.BlufinClientException;
import org.blufin.base.utils.UtilsEnvironment;
import org.blufin.core.server.deserialization.JsonMappingValidator;
import org.blufin.core.server.helper.EndPointHelper;
import org.blufin.core.server.startup.ApiApplication;
import org.blufin.jackson.Jackson;
import org.blufin.sdk.base.AbstractMetaData;
import org.blufin.sdk.enums.HttpMethod;
import org.blufin.sdk.exceptions.ResourceNotFoundException;
import org.blufin.sdk.normalization.TerminologyNormalizer;
import org.blufin.sdk.response.AckError;
import org.blufin.sdk.response.AckResolver;
import org.blufin.sdk.response.AckResolverThreadLocal;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.IOException;
import java.text.MessageFormat;

public class ModifiedRequest extends HttpServletRequestWrapper {

    public static final String ERROR_INVALID_JSON = "Invalid JSON. Please check your syntax and try again.";

    private final static ObjectMapper mapper = Jackson.getObjectMapper();

    private AbstractMetaData metaData;
    private ServletInputStream modifiedInputStream;

    /**
     * Wraps the request so we can get the body and modify it.
     * See: https://stackoverflow.com/questions/50932518/how-to-modify-request-body-before-reaching-controller-in-spring-boot
     */
    public ModifiedRequest(HttpServletRequest servletRequest) throws IOException {

        super(servletRequest);

        modifiedInputStream = servletRequest.getInputStream();

        try {

            String endPoint = EndPointHelper.normalizeEndPoint(servletRequest.getRequestURI(), HttpMethod.valueOf(servletRequest.getMethod().toUpperCase()));

            this.metaData = ApiApplication.getResourceData().getMetaData(endPoint);

        } catch (ResourceNotFoundException e) {

            /**
             * Means that no MetaData is present and this is probably a non-registered end-point.
             * For this we simply return.
             */
            return;
        }
    }

    public void validate(TerminologyNormalizer.Type expectedType) {

        if (metaData != null) {

            try {

                AckResolver ackResolver = new AckResolver();

                JsonNode incomingNode = mapper.readTree(this.modifiedInputStream);
                JsonNode outgoingNode = expectedType.equals(TerminologyNormalizer.Type.OBJECT) ? mapper.createObjectNode() : mapper.createArrayNode();

                JsonMappingValidator.validate(incomingNode, outgoingNode, ackResolver, metaData, expectedType);

                if (ackResolver.getErrors().size() > 0) {
                    throw new BlufinClientException(ackResolver.getErrors(), ackResolver.getWarnings());
                }

                AckResolverThreadLocal.set(ackResolver);

                byte[] byteArray = mapper.writeValueAsBytes(outgoingNode);

                /** See: https://stackoverflow.com/questions/30484388/inputstream-to-servletinputstream */
                this.modifiedInputStream = new ModifiedServletInputStream(byteArray);

            } catch (JsonProcessingException e) {

                String errorMessage = (UtilsEnvironment.isLocal() || UtilsEnvironment.isDevelopment()) ? MessageFormat.format("{0}. This message is only shown in {1}/{2}, in {3}/{4}/{5} you would see: {6}", e.getMessage(), Environment.LOCAL, Environment.DEVELOPMENT, Environment.SANDBOX, Environment.STAGING, Environment.PRODUCTION, ERROR_INVALID_JSON) : ERROR_INVALID_JSON;

                throw new BlufinClientException(AckError.JSON_ERROR.toString(errorMessage));

            } catch (IOException e) {

                /** TODO NOW - Implement {@link org.blufin.base.exceptions.BlufinServerException}. */

                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public ServletInputStream getInputStream() {

        return this.modifiedInputStream;
    }
}