
package org.blufin.core.server.startup;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.blufin.base.exceptions.BlufinServerException;
import org.blufin.base.helper.Pair;
import org.blufin.core.auth.core.AuthInitializationLevel;
import org.blufin.core.auth.core.AuthMethodInvoker;
import org.blufin.core.security.Encryptor;
import org.blufin.core.server.config.ApiConfig;
import org.blufin.core.server.db.CommonDataSource;
import org.blufin.core.server.db.ConfigDataSource;
import org.blufin.core.server.helper.AccountAwareDataSourceRefresher;
import org.blufin.core.server.rest.ModifiedRequest;
import org.blufin.core.server.rest.RestInterceptor;
import org.blufin.sdk.base.ResourceData;
import org.blufin.sdk.embedded.base.EmbeddedSDK;
import org.blufin.sdk.enums.HttpMethod;
import org.blufin.sdk.service.BlufinApiService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Arrays;
@ComponentScan({
        "org.blufin.core.auth.core",
        "org.blufin.core.server",
        "org.blufin.sdk.embedded.base"
})
public abstract class ApiApplication<T extends BlufinApiService, R extends ResourceData> extends AbstractApplication implements
        ApplicationListener<ContextRefreshedEvent>,
        WebMvcConfigurer,
        Filter {
    @Autowired
    AuthMethodInvoker authMethodInvoker;
    @Autowired
    AccountAwareDataSourceRefresher accountAwareDataSourceRefresher;
    @Autowired
    ApiConfig apiConfig;
    @Autowired
    EmbeddedSDK embeddedSDK;
    @Autowired
    T autowiredSdk;
    @Autowired
    R autowiredResourceData;
    @Getter
    @Setter(AccessLevel.PRIVATE)
    private static ResourceData resourceData;
    @Override
    public void onApplicationEvent(ContextRefreshedEvent context) {
        try {
            init();
            ApiApplication.setResourceData(autowiredResourceData);
            Encryptor.initialize(Pair.of(apiConfig.getEncryptionKey(), apiConfig.getEncryptionIv()));
            ConfigDataSource.refresh(apiConfig.getConfigDatabase());
            CommonDataSource.refresh(apiConfig.getCommonDatabase());
            autowiredSdk.initialize(autowiredResourceData);
            embeddedSDK.initialize(autowiredResourceData);
            if (Arrays.asList(
                    AuthInitializationLevel.INTERNAL_ACCOUNT_USER,
                    AuthInitializationLevel.OAUTH_ACCOUNT_USER).contains(apiConfig.getAuthorizationLevel())) {
                accountAwareDataSourceRefresher.setProjectName(baseConfig.getProjectName());
                accountAwareDataSourceRefresher.setAuthInitializationLevel(apiConfig.getAuthorizationLevel());
                accountAwareDataSourceRefresher.refresh();
            }
            authMethodInvoker.init(apiConfig);
        } catch (Exception e) {
            throw new RuntimeException(MessageFormat.format("Application failed to start: {0}", e.getMessage()));
        }
    }
    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        HttpMethod httpMethod = HttpMethod.valueOf(((HttpServletRequest) request).getMethod());
        switch (httpMethod) {
            case POST:
            case PUT:
            case PATCH:
                request = new ModifiedRequest((HttpServletRequest) request);
                break;
            case GET:
            case DELETE:
                break;
            default:
                throw new BlufinServerException(MessageFormat.format("Unsupported {0}: {1}", HttpMethod.class.getSimpleName(), httpMethod));
        }
        chain.doFilter(request, response);
    }
    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(new RestInterceptor());
    }
}
