
package org.blufin.sdk.base;
import lombok.Getter;
import org.blufin.base.constants.Constants;
import org.blufin.base.enums.DataType;
import org.blufin.base.enums.SchemaType;
import org.blufin.base.exceptions.BlufinAlertDeveloperException;
import org.blufin.base.helper.Pair;
import java.text.MessageFormat;
import java.util.*;
public abstract class AbstractMetaData {
    public static final String DECIMAL_DISTRIBUTION = "decimal_distribution";
    public static final String DESCRIPTION = "description";
    public static final String ENCRYPTED = "encrypted";
    public static final String ENUM_NAME = "enum_name";
    public static final String ENUM_VALUES = "enum_values";
    public static final String FKEY = "fkey";
    public static final String FLAG = "flag";
    public static final String ID = "id";
    public static final String LINK = "link";
    public static final String MAX_LENGTH = "max_length";
    public static final String CHILD_OF = "child_of";
    public static final String REQUIRED = "required";
    public static final String REQUIRED_IF = "required_if";
    public static final String TYPE = "type";
    public static final String TRANSIENT = "transient";
    public static final String FLAG_AUTO_INCREMENT = "AUTO_INCREMENT";
    public static final String FLAG_INDEX = "INDEX";
    public static final String FLAG_NULLABLE = "NULLABLE";
    public static final String FLAG_PRIMARY_KEY = "PRIMARY_KEY";
    public static final String FLAG_UNIQUE = "UNIQUE";
    public static final List<DataType> CONTAINER_TYPES = Collections.unmodifiableList(Arrays.asList(DataType.OBJECT, DataType.OBJECT_LIST, DataType.OBJECT_LINK));
    protected final LinkedHashMap<String, HashMap<String, Object>> metaData = new LinkedHashMap<>();
    @Getter
    private final SchemaType schema;
    @Getter
    private final String table;
    @Getter
    private final String parentTable;
    @Getter
    private final List<String> tableHierarchy;
    @Getter
    private final DataType childType;
    protected AbstractMetaData(SchemaType schema, String table, String parentTable, List<String> tableHierarchy, DataType childType) {
        this.schema = schema;
        this.table = table;
        this.parentTable = parentTable;
        this.tableHierarchy = tableHierarchy;
        this.childType = childType;
    }
    public abstract AbstractMetaData getNestedMetaData(String nestedTable);
    public boolean containsFieldExcludingObjects(String fieldName) {
        if (metaData.containsKey(fieldName)) {
            try {
                if (!getType(fieldName).equals(DataType.OBJECT) && !getType(fieldName).equals(DataType.OBJECT_LIST) && !getType(fieldName).equals(DataType.OBJECT_LINK)) {
                    return true;
                }
            } catch (MetaDataException e) {
                throw new BlufinAlertDeveloperException(e);
            }
        }
        return false;
    }
    public boolean containsField(String fieldName) {
        return metaData.containsKey(fieldName);
    }
    public List<String> getAllFields() {
        return getAllFields(false);
    }
    public List<String> getAllFieldsIncludingContainers() {
        return getAllFields(true);
    }
    public List<String> getRequiredFields() {
        return getRequiredFields(false);
    }
    public List<String> getRequiredFieldsIncludingContainers() {
        return getRequiredFields(true);
    }
    public Set<String> getAllNestedTables() {
        Set<String> allNestedTables = new TreeSet<>();
        this.tableHierarchy.forEach(table -> allNestedTables.add(table.replaceAll("/", "_")));
        return allNestedTables;
    }
    public DataType getType(String fieldName) throws MetaDataException {
        checkFieldExists(fieldName);
        return (DataType) metaData.get(fieldName).get(TYPE);
    }
    public Integer getMaxLength(String fieldName) throws MetaDataException {
        checkFieldExists(fieldName);
        if (metaData.get(fieldName).keySet().contains(MAX_LENGTH)) {
            return (Integer) metaData.get(fieldName).get(MAX_LENGTH);
        } else {
            return null;
        }
    }
    public String getDescription(String fieldName) throws MetaDataException {
        checkFieldExists(fieldName);
        if (metaData.get(fieldName).keySet().contains(DESCRIPTION)) {
            return (String) metaData.get(fieldName).get(DESCRIPTION);
        } else {
            return null;
        }
    }
    @SuppressWarnings("unchecked")
    public Pair<Integer, Integer> getDecimalDistribution(String fieldName) throws MetaDataException {
        checkFieldAndKeyExists(fieldName, DECIMAL_DISTRIBUTION);
        return (Pair<Integer, Integer>) metaData.get(fieldName).get(DECIMAL_DISTRIBUTION);
    }
    @SuppressWarnings("unchecked")
    public String getEnumName(String fieldName) throws MetaDataException {
        checkFieldAndKeyExists(fieldName, ENUM_NAME);
        return (String) metaData.get(fieldName).get(ENUM_NAME);
    }
    @SuppressWarnings("unchecked")
    public List<String> getEnumValues(String fieldName) throws MetaDataException {
        checkFieldAndKeyExists(fieldName, ENUM_VALUES);
        return (List<String>) metaData.get(fieldName).get(ENUM_VALUES);
    }
    @SuppressWarnings("unchecked")
    public String getForeignKey(String fieldName) throws MetaDataException {
        checkFieldAndKeyExists(fieldName, FKEY);
        return (String) metaData.get(fieldName).get(FKEY);
    }
    @SuppressWarnings("unchecked")
    public String getLink(String fieldName) throws MetaDataException {
        checkFieldAndKeyExists(fieldName, LINK);
        return (String) metaData.get(fieldName).get(LINK);
    }
    public String getParentIdFieldName() {
        if (parentTable == null) {
            throw new BlufinAlertDeveloperException(MessageFormat.format("Table ''{0}'' does not have a parent, and therefore, {1}.getParentIdFieldName() should never be called.", table, AbstractMetaData.class.getSimpleName()));
        }
        return parentTable + Constants.UNDERSCORE + AbstractMetaData.ID;
    }
    public boolean isAutoIncrement(String fieldName) throws MetaDataException {
        return hasFlag(fieldName, FLAG_AUTO_INCREMENT);
    }
    public boolean isIndex(String fieldName) throws MetaDataException {
        return hasFlag(fieldName, FLAG_INDEX);
    }
    public boolean isNullAllowed(String fieldName) throws MetaDataException {
        return hasFlag(fieldName, FLAG_NULLABLE);
    }
    public boolean isPrimaryKey(String fieldName) throws MetaDataException {
        return hasFlag(fieldName, FLAG_PRIMARY_KEY);
    }
    public boolean isUnique(String fieldName) throws MetaDataException {
        return hasFlag(fieldName, FLAG_UNIQUE) || isAutoIncrement(fieldName);
    }
    public boolean isEncrypted(String fieldName) throws MetaDataException {
        return hasKey(fieldName, ENCRYPTED);
    }
    public boolean isRequired(String fieldName) throws MetaDataException {
        return hasKey(fieldName, REQUIRED);
    }
    public boolean isTransient(String fieldName) throws MetaDataException {
        return hasKey(fieldName, TRANSIENT);
    }
    public boolean isForeignKeyed(String fieldName) throws MetaDataException { 
        return hasKey(fieldName, FKEY);
    }
    public boolean isForeignKeyedAsChild(String fieldName) throws MetaDataException { 
        return hasKey(fieldName, CHILD_OF);
    }
    public boolean isLinked(String fieldName) throws MetaDataException {
        return hasKey(fieldName, LINK);
    }
    public boolean isObject(String fieldName) throws MetaDataException {
        return getType(fieldName).equals(DataType.OBJECT);
    }
    public boolean isObjectMap(String fieldName) throws MetaDataException {
        return getType(fieldName).equals(DataType.OBJECT_LIST);
    }
    public boolean isNestedObject() { 
        return childType != null && childType.equals(DataType.OBJECT);
    }
    public boolean isNestedObjectList() { 
        return childType != null && childType.equals(DataType.OBJECT_LIST);
    }
    protected List<String> getEnumValues(Enum[] values) {
        List<String> valueList = new ArrayList<>();
        for (Enum value : values) {
            valueList.add(value.toString());
        }
        return valueList;
    }
    private boolean hasKey(String fieldName, String key) throws MetaDataException {
        checkFieldExists(fieldName);
        return metaData.get(fieldName).keySet().contains(key);
    }
    private boolean hasFlag(String fieldName, String key) throws MetaDataException {
        checkFieldExists(fieldName);
        if (metaData.get(fieldName).keySet().contains(FLAG)) {
            List<String> flags = (List<String>) metaData.get(fieldName).get(FLAG);
            return flags.contains(key);
        } else {
            return false;
        }
    }
    private void checkFieldExists(String fieldName) throws MetaDataException {
        if (!metaData.containsKey(fieldName)) {
            throw new MetaDataException(MessageFormat.format("Field not found: {0}", fieldName));
        }
    }
    private void checkFieldAndKeyExists(String fieldName, String key) throws MetaDataException {
        checkFieldExists(fieldName);
        if (!metaData.get(fieldName).keySet().contains(key)) {
            throw new MetaDataException(MessageFormat.format("Field ''{0}'' does not have key → {1}", fieldName, key));
        }
    }
    private List<String> getAllFields(boolean includeContainers) {
        List<String> allFields = new ArrayList<>();
        metaData.keySet().stream().forEach(fieldName -> {
            try {
                if (includeContainers || (!getType(fieldName).equals(DataType.OBJECT) && !getType(fieldName).equals(DataType.OBJECT_LIST) && !getType(fieldName).equals(DataType.OBJECT_LINK))) {
                    allFields.add(fieldName);
                }
            } catch (MetaDataException e) {
                throw new BlufinAlertDeveloperException(e);
            }
        });
        return allFields;
    }
    private List<String> getRequiredFields(boolean includeContainers) {
        List<String> requiredFields = new ArrayList<>();
        metaData.keySet().stream().forEach(fieldName -> {
            try {
                if (getType(fieldName).equals(DataType.OBJECT_LIST)) {
                    if (!includeContainers) {
                        return;
                    }
                }
                if (getType(fieldName).equals(DataType.OBJECT_LINK) || getType(fieldName).equals(DataType.OBJECT)) {
                    if (!includeContainers || !isRequired(fieldName)) {
                        return;
                    }
                }
                if (fieldName.equals(ID) ||
                        isNullAllowed(fieldName) ||
                        isTransient(fieldName) ||
                        isForeignKeyedAsChild(fieldName)) { 
                    return;
                }
                requiredFields.add(fieldName);
            } catch (MetaDataException e) {
                throw new BlufinAlertDeveloperException(e);
            }
        });
        return requiredFields;
    }
}
