
package org.blufin.sdk.normalization;
import org.apache.commons.lang3.StringUtils;
import org.blufin.base.constants.Constants;
import org.blufin.base.enums.TimeZone;
import org.blufin.base.helper.Pair;
import org.blufin.base.helper.Triplet;
import org.blufin.base.utils.UtilsRegex;
import org.blufin.sdk.filters.dynamic.FilterBoolean;
import org.blufin.sdk.response.AckError;
import org.blufin.sdk.response.AckResolver;
import org.blufin.sdk.response.AckWarning;
import java.math.BigDecimal;
import java.text.MessageFormat;
import java.time.DateTimeException;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.regex.Pattern;
public class DataNormalizer {
    public static final String ENUM_SEPARATOR = ", ";
    private static final Pattern REGEX_INTEGER = Pattern.compile("^(-)?\\d+$");
    private static final Pattern REGEX_DECIMAL = Pattern.compile("^(-)?[\\d\\.]+$");
    private static final Pattern REGEX_ZEROS = Pattern.compile("^[0]+$");
    private static final Pattern REGEX_MILLISECONDS = Pattern.compile("^\\d{3}0+$");
    public final static String normalizeString(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        return normalizeString(value, element, ackResolver, 0);
    }
    public final static String normalizeString(String value, String element, AckResolver ackResolver, Integer maxLength) throws DataNormalizationException {
        if (maxLength != null && maxLength > 0) {
            if (value.length() > maxLength) {
                ackResolver.addError(AckError.STRING_MAX_LENGTH_EXCEEDED, value, element, String.valueOf(maxLength), String.valueOf(value.length()));
                throw new DataNormalizationException();
            }
        }
        return value;
    }
    public final static String normalizeStringEnum(String value, String element, AckResolver ackResolver, List<String> enumValues) throws DataNormalizationException {
        value = trimWhitespace(value);
        String valueUpperCase = value.toUpperCase();
        checkValueIsNotBlank(valueUpperCase, ackResolver, AckError.STRING_ENUM_BLANK, element);
        if (!enumValues.contains(valueUpperCase)) {
            ackResolver.addError(AckError.STRING_ENUM_INVALID, value, element, StringUtils.join(enumValues, ENUM_SEPARATOR));
            throw new DataNormalizationException();
        }
        if (!value.equals(valueUpperCase)) {
            ackResolver.addWarning(AckWarning.STRING_ENUM_NOT_UPPERCASE, value, valueUpperCase, element);
        }
        return valueUpperCase;
    }
    public final static boolean normalizeBoolean(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        value = trimWhitespace(value);
        String valueLowercase = value.toLowerCase();
        checkValueIsNotBlank(valueLowercase, ackResolver, AckError.BOOLEAN_BLANK, element);
        if (valueLowercase.equals(FilterBoolean.TRUE) || valueLowercase.equals(FilterBoolean.FALSE)) {
            if (!value.equals(valueLowercase)) {
                ackResolver.addWarning(AckWarning.BOOLEAN_NOT_LOWERCASE, value, value.toLowerCase(), element);
            }
            return Boolean.valueOf(valueLowercase);
        } else {
            ackResolver.addError(AckError.BOOLEAN_INVALID, value, element);
            throw new DataNormalizationException();
        }
    }
    public final static int normalizeInt(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = convertToWholeNumber(value, element, ackResolver, AckError.INT_CONTAINED_DECIMAL_POINT, AckError.INT_INVALID, AckError.INT_BLANK, AckWarning.INT_CONTAINED_LEADING_ZEROS);
            return Integer.parseInt(value);
        } catch (NumberFormatException e) {
            ackResolver.addError(AckError.INT_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }
    public final static long normalizeIntBig(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = convertToWholeNumber(value, element, ackResolver, AckError.LONG_CONTAINED_DECIMAL_POINT, AckError.LONG_INVALID, AckError.LONG_BLANK, AckWarning.LONG_CONTAINED_LEADING_ZEROS);
            return Long.parseLong(value);
        } catch (NumberFormatException e) {
            ackResolver.addError(AckError.LONG_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }
    public final static short normalizeIntSmall(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = convertToWholeNumber(value, element, ackResolver, AckError.SHORT_CONTAINED_DECIMAL_POINT, AckError.SHORT_INVALID, AckError.SHORT_BLANK, AckWarning.SHORT_CONTAINED_LEADING_ZEROS);
            return Short.parseShort(value);
        } catch (NumberFormatException e) {
            ackResolver.addError(AckError.SHORT_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }
    public final static byte normalizeIntTiny(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = convertToWholeNumber(value, element, ackResolver, AckError.BYTE_CONTAINED_DECIMAL_POINT, AckError.BYTE_INVALID, AckError.BYTE_BLANK, AckWarning.BYTE_CONTAINED_LEADING_ZEROS);
            return Byte.parseByte(value);
        } catch (NumberFormatException e) {
            ackResolver.addError(AckError.BYTE_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }
    public final static BigDecimal normalizeDecimal(String value, String element, AckResolver ackResolver, Pair decimalDistribution) throws DataNormalizationException {
        value = trimWhitespace(value);
        checkValueIsNotBlank(value, ackResolver, AckError.DECIMAL_BLANK, element);
        String valueWithLeadingZerosRemoved = removeLeadingZeros(value, element, ackResolver, AckWarning.DECIMAL_CONTAINED_LEADING_ZEROS);
        int m = (int) decimalDistribution.getKey();
        int d = (int) decimalDistribution.getValue();
        if (UtilsRegex.stringMatchesRegex(valueWithLeadingZerosRemoved, REGEX_INTEGER)) {
            String valueWithDecimal = valueWithLeadingZerosRemoved + Constants.PERIOD + StringUtils.repeat("0", d);
            return new BigDecimal(valueWithDecimal);
        } else if (!UtilsRegex.stringMatchesRegex(valueWithLeadingZerosRemoved, REGEX_DECIMAL)) {
            ackResolver.addError(AckError.DECIMAL_INVALID, valueWithLeadingZerosRemoved, element);
            throw new DataNormalizationException();
        }
        String[] parts = valueWithLeadingZerosRemoved.split("\\.");
        BigDecimal fin;
        if (parts.length == 1) {
            fin = new BigDecimal(parts[0] + Constants.PERIOD + StringUtils.repeat("0", d));
        } else if (parts.length == 2) {
            if (parts[0].equals("-")) {
                parts[0] = "-0";
            } else if (parts[0].equals("")) {
                parts[0] = "0";
            }
            boolean partsNegative = false;
            if (parts[0].substring(0, 1).equals("-")) {
                parts[0] = parts[0].substring(1);
                partsNegative = true;
            }
            if (parts[1].length() > d) {
                if (UtilsRegex.stringMatchesRegex(parts[1].substring(d), REGEX_ZEROS)) {
                    parts[1] = parts[1].substring(0, d);
                    ackResolver.addWarning(AckWarning.DECIMAL_CONTAINED_TRAILING_ZEROS, valueWithLeadingZerosRemoved, (partsNegative ? "-" : "") + parts[0] + Constants.PERIOD + parts[1], element, String.valueOf(m), String.valueOf(d));
                } else {
                    ackResolver.addError(AckError.DECIMAL_TOO_GRANULAR, valueWithLeadingZerosRemoved, element, String.valueOf(m), String.valueOf(d));
                    throw new DataNormalizationException();
                }
            } else if (parts[1].length() < d) {
                parts[1] = parts[1] + StringUtils.repeat("0", (d - parts[1].length()));
            }
            fin = new BigDecimal((partsNegative ? "-" : "") + parts[0] + Constants.PERIOD + parts[1]);
        } else {
            ackResolver.addError(AckError.DECIMAL_INVALID, valueWithLeadingZerosRemoved, element);
            throw new DataNormalizationException();
        }
        BigDecimal min = DecimalHelper.getMinimumAllowedValue(decimalDistribution);
        BigDecimal max = DecimalHelper.getMaximumAllowedValue(decimalDistribution);
        if (fin.compareTo(min) == -1 || fin.compareTo(max) == 1) {
            ackResolver.addError(AckError.DECIMAL_OUT_OF_BOUNDS, valueWithLeadingZerosRemoved, element, String.valueOf(min), String.valueOf(max));
            throw new DataNormalizationException();
        }
        return fin;
    }
    public final static LocalDate normalizeDate(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        if (value == null || value.trim().equals("")) {
            ackResolver.addError(AckError.DATE_BLANK, element);
            throw new DataNormalizationException();
        }
        value = trimWhitespace(value);
        try {
            return LocalDate.parse(value);
        } catch (Exception e1) {
            try {
                String[] datePartsModified = getModifiedDateParts(value.split("-"));
                LocalDate modifiedDate = LocalDate.of(Integer.parseInt(datePartsModified[0]), Integer.parseInt(datePartsModified[1]), Integer.parseInt(datePartsModified[2]));
                ackResolver.addWarning(AckWarning.DATE_UNCONVENTIONAL_FORMAT, value, modifiedDate.toString(), element);
                return modifiedDate;
            } catch (DataNormalizationException | NumberFormatException | DateTimeException e2) {
            }
            ackResolver.addError(AckError.DATE_INVALID_FORMAT, value, element);
            throw new DataNormalizationException();
        }
    }
    public final static ZonedDateTime normalizeDateTime(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        return normalizeDateTime(value, element, ackResolver, TimeZone.UTC);
    }
    public final static ZonedDateTime normalizeDateTime(String value, String element, AckResolver ackResolver, TimeZone timeZone) throws DataNormalizationException {
        if (value == null || value.trim().equals("")) {
            ackResolver.addError(AckError.DATETIME_BLANK, element);
            throw new DataNormalizationException();
        }
        value = trimWhitespace(value);
        int initialErrorCount = ackResolver.getErrors().size();
        try {
            boolean dateUnconventional = false;
            String[] dateTimeSplit = value.split("T");
            if (dateTimeSplit.length != 2) {
                throw new DataNormalizationException();
            }
            String[] dateParts = dateTimeSplit[0].split("-");
            String[] datePartsModified = getModifiedDateParts(dateParts);
            if (!StringUtils.join(dateParts, "-").equals(StringUtils.join(datePartsModified, "-"))) {
                dateUnconventional = true;
            }
            Triplet<String, String, String> timeSecondsAndZone = DateTimeSplitter.splitTimeNanoAndZone(dateTimeSplit[1], element, ackResolver);
            String[] timePartsModified = getModifiedTimeParts(timeSecondsAndZone.getFirst().split(":"));
            String t = StringUtils.join(timePartsModified, ":");
            String m = timeSecondsAndZone.getSecond();
            String z = timeSecondsAndZone.getThird();
            if (z == null) {
                if (timeZone.equals(TimeZone.UTC)) {
                    z = TimeZone.UTC.getZoneIdDisplay();
                } else {
                    z = TimeZone.UTC.getZoneIdDisplay() + "[" + timeZone.getZoneIdDisplay() + "]";
                }
            }
            if (m == null || m.equals("")) {
                m = "000";
            } else if (m.length() < 3) {
                m = StringUtils.rightPad(m, 3, "0");
            } else if (m.length() > 3) {
                if (UtilsRegex.stringMatchesRegex(m, REGEX_MILLISECONDS)) {
                    m = m.substring(0, 3);
                } else {
                    ackResolver.addError(AckError.DATETIME_TIME_TOO_GRANULAR, value, element);
                    throw new DataNormalizationException();
                }
            }
            String finalDatetimeString = StringUtils.join(datePartsModified, "-") + "T" + t + Constants.PERIOD + m + z;
            if (dateUnconventional) {
                ackResolver.addWarning(AckWarning.DATETIME_UNCONVENTIONAL_FORMAT, value, finalDatetimeString, element);
            }
            try {
                return ZonedDateTime.parse(finalDatetimeString, DateTimeFormatter.ISO_ZONED_DATE_TIME);
            } catch (Exception e) {
                throw new DataNormalizationException();
            }
        } catch (DataNormalizationException | NumberFormatException e) {
            if (ackResolver.getErrors().size() == initialErrorCount) {
                ackResolver.addError(AckError.DATETIME_INVALID_FORMAT, value, element);
            }
            throw new DataNormalizationException();
        }
    }
    private static String convertToWholeNumber(String value, String element, AckResolver ackResolver, AckError containsDecimalPointError, AckError invalidError, AckError blankError, AckWarning containsLeadingZerosWarning) throws DataNormalizationException {
        value = trimWhitespace(value);
        checkValueIsNotBlank(value, ackResolver, blankError, element);
        if (UtilsRegex.stringMatchesRegex(value.trim(), REGEX_INTEGER)) {
            value = removeLeadingZeros(value, element, ackResolver, containsLeadingZerosWarning);
            return value;
        } else if (UtilsRegex.stringMatchesRegex(value.trim(), REGEX_DECIMAL)) {
            value = removeLeadingZeros(value, element, ackResolver, containsLeadingZerosWarning);
            int decimalPointMatches = StringUtils.countMatches(value, ".");
            if (decimalPointMatches == 0) {
                return value;
            } else if (decimalPointMatches == 1) {
                ackResolver.addError(containsDecimalPointError, value, element);
                throw new DataNormalizationException();
            }
        }
        ackResolver.addError(invalidError, value, element);
        throw new DataNormalizationException();
    }
    private static String removeLeadingZeros(String value, String element, AckResolver ackResolver, AckWarning containsLeadingZerosWarning) {
        boolean valueNegative = false;
        if (value.substring(0, 1).equals("-")) {
            value = value.substring(1);
            valueNegative = true;
        }
        if (value.equals("0")) {
            return value;
        } else if (value.length() >= 2 && value.substring(0, 2).equals("0.")) {
            return (valueNegative ? "-" : "") + value;
        }
        String valueWithoutLeadingZeros = StringUtils.stripStart(value, "0");
        if (valueWithoutLeadingZeros.equals("")) {
            valueWithoutLeadingZeros = "0";
        }
        String adjustedAmount = ((valueNegative && !valueWithoutLeadingZeros.equals("0")) ? "-" : "") + valueWithoutLeadingZeros;
        if (!value.equals(valueWithoutLeadingZeros)) {
            ackResolver.addWarning(containsLeadingZerosWarning, (valueNegative ? "-" : "") + value, adjustedAmount, element);
        }
        return adjustedAmount;
    }
    private static String[] getModifiedDateParts(String[] dateParts) throws DataNormalizationException {
        if (dateParts.length != 3) {
            throw new DataNormalizationException();
        }
        int idx = -1;
        String[] datePartsModified = new String[3];
        for (String datePart : dateParts) {
            if (datePart == null || datePart.equals("") || !datePart.trim().equals(datePart)) {
                throw new DataNormalizationException();
            }
            idx++;
            switch (idx) {
                case 0:
                    if (Integer.parseInt(datePart) < 0) {
                        throw new DataNormalizationException();
                    }
                    if (datePart.length() <= 4) {
                        datePartsModified[idx] = StringUtils.leftPad(datePart, 4, "0");
                    } else {
                        throw new DataNormalizationException();
                    }
                    break;
                case 1:
                case 2:
                    if (Integer.parseInt(datePart) <= 0) {
                        throw new DataNormalizationException();
                    }
                    if (datePart.length() <= 2) {
                        datePartsModified[idx] = StringUtils.leftPad(datePart, 2, "0");
                    } else {
                        throw new DataNormalizationException();
                    }
                    break;
                default:
                    throw new RuntimeException(MessageFormat.format("Index out of bounds: {0} — This statement should never be reached.", idx));
            }
        }
        return datePartsModified;
    }
    private static String[] getModifiedTimeParts(String[] timeParts) throws DataNormalizationException {
        if (timeParts.length != 3) {
            throw new DataNormalizationException();
        }
        int idx = -1;
        String[] timePartsModified = new String[3];
        for (String timePart : timeParts) {
            if (timePart == null || timePart.equals("") || !timePart.trim().equals(timePart)) {
                throw new DataNormalizationException();
            }
            idx++;
            if (timePart.length() <= 2) {
                timePartsModified[idx] = StringUtils.leftPad(timePart, 2, "0");
            } else {
                throw new DataNormalizationException();
            }
        }
        return timePartsModified;
    }
    private static String trimWhitespace(String value) {
        if (value == null) {
            return StringUtils.EMPTY;
        }
        return value.trim();
    }
    private static void checkValueIsNotBlank(String value, AckResolver ackResolver, AckError blankErrorMessage, String element) throws DataNormalizationException {
        if (value.trim().equals("")) {
            ackResolver.addError(blankErrorMessage, element);
            throw new DataNormalizationException();
        }
    }
}
