
package org.blufin.sdk.request;
import com.mashape.unirest.http.HttpMethod;
import com.mashape.unirest.http.HttpResponse;
import com.mashape.unirest.http.JsonNode;
import com.mashape.unirest.http.Unirest;
import com.mashape.unirest.http.exceptions.UnirestException;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.blufin.base.annotations.TestOnly;
import org.blufin.base.exceptions.BlufinAlertDeveloperException;
import org.blufin.base.exceptions.BlufinClientException;
import org.blufin.base.exceptions.BlufinNotImplementedException;
import org.blufin.base.exceptions.BlufinServerException;
import org.blufin.base.helper.IdSet;
import org.blufin.base.helper.Pair;
import org.blufin.jackson.Jackson;
import org.blufin.sdk.base.AbstractField;
import org.blufin.sdk.base.AbstractMetaData;
import org.blufin.sdk.base.AbstractSort;
import org.blufin.sdk.base.PersistentDto;
import org.blufin.sdk.constants.SdkConstants;
import org.blufin.sdk.embedded.base.EmbeddedResourceData;
import org.blufin.sdk.enums.SortOrder;
import org.blufin.sdk.filters.Filter;
import org.blufin.sdk.filters.FilterContainer;
import org.blufin.sdk.filters.FilterUriBuilder;
import org.blufin.sdk.filters.Match;
import org.blufin.sdk.filters.SortUriBuilder;
import org.blufin.sdk.request.credentials.RequestCredentials;
import org.blufin.sdk.request.interfaces.ApiGetExecutor;
import org.blufin.sdk.response.ApiResponse;
import org.blufin.sdk.response.HttpStatus;
import org.blufin.sdk.rest.GetRequest;
import org.blufin.sdk.service.credentials.ApiCredentials;
import org.json.JSONException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.text.MessageFormat;
import java.util.*;
public abstract class GetRequestBase<DTO extends PersistentDto, FIELD extends AbstractField, SORT extends AbstractSort> extends RequestBase<DTO> implements ApiGetExecutor<DTO, FIELD> {
    @Setter(AccessLevel.PROTECTED)
    private AbstractMetaData metaData;
    @Setter(AccessLevel.PROTECTED)
    private IdSet idSet;
    @Getter(AccessLevel.PROTECTED)
    private List<Filter> filterList = new LinkedList<>();
    private FilterContainer filterContainer;
    private List<Pair<String, SortOrder>> sort;
    private int page;
    private int entries;
    public GetRequestBase(final ApiCredentials apiCredentials, final String endPoint, final Class<DTO> responseDto, final AbstractMetaData metaData, final RequestCredentials requestCredentials) {
        super(apiCredentials, endPoint, responseDto, requestCredentials, HttpMethod.GET);
        this.metaData = metaData;
    }
    @Override
    public ApiResponse<DTO> execute() {
        try {
            if (filterContainer == null) {
                filterContainer = new FilterContainer(Match.ALL, filterList);
            }
            GetRequest getRequest = new GetRequest(metaData, idSet);
            if (getApiCredentials().getApiCredentialsType().equals(ApiCredentials.ApiCredentialsType.INTERNAL)) {
                if (idSet == null) {
                    if (filterList.size() > 0) {
                        getRequest.setFilters(filterContainer);
                    }
                    if (sort != null) {
                        getRequest.setSort(sort);
                    }
                    if (page > 0) {
                        getRequest.setPage(page);
                    }
                    if (entries > 0) {
                        getRequest.setEntries(entries);
                    }
                } else {
                    if (sort != null) {
                        getRequest.setSort(sort);
                    }
                }
                return mapIfEmbedded(getApiCredentials().getResourceData().executeGet(getEndPoint(), getRequest));
            } else {
                String uri = getApiCredentials().getApiUri() + "/" + getEndPoint();
                HttpResponse<JsonNode> httpResponse = Unirest.get(uri).headers(buildHeaders()).asJson();
                JsonNode jsonNode = httpResponse.getBody();
                ApiResponse<DTO> apiResponse = new ApiResponse<>(HttpStatus.getFromCode((int) jsonNode.getObject().get(ApiResponse.RESPONSE_CODE)));
                try {
                    apiResponse.setBody(Jackson.getObjectMapper().readValue(jsonNode.getObject().get(ApiResponse.DATA).toString(), List.class));
                } catch (JSONException e) {
                    apiResponse.setBody(Collections.EMPTY_LIST);
                }
                return mapIfEmbedded(apiResponse);
            }
        } catch (IOException | UnirestException | BlufinClientException | BlufinServerException e) {
            return new ApiResponse<>(HttpStatus.BAD_REQUEST);
        }
    }
    @Override
    public ApiResponse<Map<FIELD, Object>> executeWithLimitedFields(FIELD... fields) {
        throw new BlufinNotImplementedException();
    }
    protected void setSortInternal(Pair<SORT, SortOrder> sort) {
        if (sort.getKey() == null || sort.getValue() == null) {
            throw new BlufinAlertDeveloperException("Field and/or SortOrder cannot be null.");
        }
        setSortInternal(Arrays.asList(sort));
    }
    protected void setSortInternal(List<Pair<SORT, SortOrder>> sortList) {
        List<Pair<String, SortOrder>> list = new ArrayList<>();
        sortList.forEach(pair -> list.add(Pair.of(pair.getKey().getFieldName(), pair.getValue())));
        this.sort = list;
    }
    protected void setPageInternal(int page) {
        if (page < 1) {
            throw new BlufinAlertDeveloperException("Page cannot be less than 1.");
        }
        this.page = page;
    }
    protected void setEntriesInternal(int entries) {
        if (entries < 1) {
            throw new BlufinAlertDeveloperException("Entries cannot be less than 1.");
        } else if (entries > SdkConstants.API_MAXIMUM_ENTRIES) {
            throw new BlufinAlertDeveloperException(MessageFormat.format("Exceeded maximum entries of: {0}", SdkConstants.API_MAXIMUM_ENTRIES));
        }
        this.entries = entries;
    }
    protected String buildUriParameterString() {
        StringBuilder sb = new StringBuilder();
        if (this.entries > 0) {
            sb.append(SdkConstants.ENTRIES).append("=").append(this.entries).append("&");
        }
        if (this.filterList != null && this.filterList.size() > 0) {
            sb.append(FilterUriBuilder.build(filterList)).append("&");
        }
        if (this.page > 0) {
            sb.append(SdkConstants.PAGE).append("=").append(this.page).append("&");
        }
        if (this.sort != null) {
            sb.append(SortUriBuilder.build(sort)).append("&");
        }
        String uriParameterString = sb.toString();
        if (uriParameterString.length() > 0 && uriParameterString.charAt(uriParameterString.length() - 1) == '&') {
            uriParameterString = uriParameterString.substring(0, uriParameterString.length() - 1);
        }
        return uriParameterString;
    }
    public ApiResponse<DTO> mapIfEmbedded(ApiResponse apiResponse) {
        if (!apiResponse.getBody().get(0).getClass().getName().equals(getResponseDto().getName())) {
            ArrayList<DTO> newData = new ArrayList<>();
            apiResponse.getBody().forEach(source -> {
                try {
                    newData.add((DTO) EmbeddedResourceData.getInstance().mapper(getResponseDto().getSimpleName()).map(apiResponse.getBody().get(0)));
                } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                    throw new BlufinAlertDeveloperException(MessageFormat.format("Embedded mapping failed: ({0}) {1}", e.getClass().getSimpleName(), e.getMessage()), e);
                }
            });
            apiResponse.setBody(newData);
        }
        return apiResponse;
    }
    @TestOnly
    protected String getEndPointForTesting() {
        StringBuilder sb = new StringBuilder(getEndPoint());
        String uriParameterString = buildUriParameterString();
        if (uriParameterString != null && uriParameterString.length() > 0) {
            sb.append("?").append(uriParameterString);
        }
        return sb.toString();
    }
    @TestOnly
    protected String getEndPointForTesting(List<FIELD> fields) {
        if (true) {
            throw new BlufinNotImplementedException();
        }
        return null;
    }
}
