
package org.blufin.sdk.rest;
import lombok.Getter;
import org.blufin.base.exceptions.BlufinAlertDeveloperException;
import org.blufin.base.exceptions.BlufinServerException;
import org.blufin.base.helper.IdSet;
import org.blufin.base.helper.Triplet;
import org.blufin.sdk.base.AbstractMetaData;
import org.blufin.sdk.base.AbstractValidatorInterface;
import org.blufin.sdk.base.PersistentDto;
import org.blufin.sdk.enums.HttpMethod;
import org.blufin.sdk.enums.PayloadType;
import org.blufin.sdk.helper.IdHelper;
import org.blufin.sdk.response.AckResolver;
import org.blufin.sdk.response.AckResolverThreadLocal;
import java.sql.Connection;
import java.text.MessageFormat;
import java.util.List;
@Getter
public abstract class AbstractPayLoadRequest<D extends PersistentDto> extends AbstractRequest {
    private final List<D> payload;
    private final PayloadType payloadType;
    private final IdSet modifiedIds;
    private Triplet<String, String, Integer> modifiedParentData;
    protected AbstractPayLoadRequest(Connection connection, AbstractMetaData metaData, AbstractValidatorInterface<D> validator, List<D> payload, PayloadType payloadType) throws BlufinServerException {
        super((payloadType == PayloadType.POST ? HttpMethod.POST : HttpMethod.PUT), metaData, AckResolverThreadLocal.get());
        this.payload = payload;
        this.payloadType = payloadType;
        this.modifiedIds = new IdSet();
        if (ackResolver == null) {
            throw new BlufinAlertDeveloperException(MessageFormat.format("{0} should never be null within {1} constructor.", AckResolver.class.getSimpleName(), AbstractPayLoadRequest.class.getSimpleName()));
        }
        try {
            if (payload.size() > 0) {
                switch (payloadType) {
                    case POST:
                        validator.validatePost(connection, payload, ackResolver);
                        break;
                    case PUT:
                        validator.validatePut(connection, payload, ackResolver);
                        break;
                    default:
                        throw new BlufinAlertDeveloperException(MessageFormat.format("Unsupported payloadType: {0}", payloadType.toString()));
                }
            }
        } finally {
            AckResolverThreadLocal.destroy();
        }
    }
    public void addIdToResponse(int id) {
        this.modifiedIds.add(id);
    }
    public void addParentDataToResponse(List<D> dtos, String table, String parentIdFieldName) {
        if (this.modifiedParentData != null) {
            throw new BlufinAlertDeveloperException(MessageFormat.format("{0}.addParentIdToResponse() was called when ''modifiedParentData'' already had a value: {1}. This property was designed in a way to only ever expects a single Pair of values.", this.getClass().getSimpleName(), modifiedParentData.toString()));
        }
        Integer parentId = IdHelper.extractParentId(dtos);
        if (!IdHelper.isValid(parentId)) {
            throw new BlufinAlertDeveloperException(MessageFormat.format("Parent ID was ''{1}'' in {0}.addParentIdToResponse().", this.getClass().getSimpleName(), parentId));
        }
        this.modifiedParentData = Triplet.of(table, parentIdFieldName, parentId);
    }
}
