
package org.blufin.sdk.rest;
import org.blufin.base.helper.IdSet;
import org.blufin.base.helper.Pair;
import org.blufin.base.utils.UtilsUrl;
import org.blufin.sdk.base.AbstractMetaData;
import org.blufin.sdk.constants.SdkConstants;
import org.blufin.sdk.enums.HttpMethod;
import org.blufin.sdk.enums.SortOrder;
import org.blufin.sdk.filters.FilterContainer;
import org.blufin.sdk.normalization.DataNormalizationException;
import org.blufin.sdk.normalization.DataNormalizer;
import org.blufin.sdk.response.AckError;
import org.blufin.sdk.validators.FieldValidator;
import org.blufin.sdk.validators.FilterValidator;
import org.blufin.sdk.validators.SortValidator;
import lombok.Getter;
import java.io.UnsupportedEncodingException;
import java.util.List;
@Getter
public class GetRequest extends AbstractRequest {
    private final IdSet idSet;
    private List<String> fields;
    private FilterContainer filters;
    private List<Pair<String, SortOrder>> sort;
    private int page = SdkConstants.API_DEFAULT_PAGE;
    private int entries = SdkConstants.API_DEFAULT_ENTRIES;
    public GetRequest(AbstractMetaData metaData) {
        super(HttpMethod.GET, metaData);
        this.idSet = null;
    }
    public GetRequest(AbstractMetaData metaData, IdSet idSet) {
        super(HttpMethod.GET, metaData);
        this.idSet = idSet;
    }
    public void setFields(String queryParameter) {
        if (queryParameter != null) {
            try {
                setFields(FieldValidator.deserialize(UtilsUrl.decode(queryParameter), ackResolver, metaData));
            } catch (IllegalArgumentException | UnsupportedEncodingException e) {
                ackResolver.addError(AckError.URL_ENCODING_ERROR, queryParameter, SdkConstants.FILTERS);
            }
        }
    }
    public void setFields(List<String> fields) {
        this.fields = fields;
    }
    public void setFilters(String queryParameters) {
        if (queryParameters != null) {
            try {
                setFilters(FilterValidator.deserialize(UtilsUrl.decode(queryParameters), ackResolver, metaData));
            } catch (IllegalArgumentException | UnsupportedEncodingException e) {
                ackResolver.addError(AckError.URL_ENCODING_ERROR, queryParameters, SdkConstants.FILTERS);
            }
        }
    }
    public void setFilters(FilterContainer filters) {
        this.filters = filters;
    }
    public void setSort(String queryParameters) {
        if (queryParameters != null) {
            try {
                setSort(SortValidator.deserialize(UtilsUrl.decode(queryParameters), ackResolver, metaData));
            } catch (IllegalArgumentException | UnsupportedEncodingException e) {
                ackResolver.addError(AckError.URL_ENCODING_ERROR, queryParameters, SdkConstants.SORT);
            }
        }
    }
    public void setSort(List<Pair<String, SortOrder>> sortList) {
        this.sort = sortList;
    }
    public void setPage(String page) {
        if (page != null) {
            try {
                setPage(DataNormalizer.normalizeInt(UtilsUrl.decode(page), SdkConstants.PAGE, ackResolver));
            } catch (DataNormalizationException e) {
            } catch (IllegalArgumentException e) {
                ackResolver.addError(AckError.URL_ENCODING_ERROR, page, SdkConstants.PAGE);
            } catch (UnsupportedEncodingException e) {
                ackResolver.addError(AckError.URL_ENCODING_ERROR, page, SdkConstants.PAGE);
            }
        }
    }
    public void setPage(int page) {
        if (page < 1) {
            ackResolver.addError(AckError.PAGE_UNDER_MINIMUM, String.valueOf(page));
        } else {
            this.page = page;
        }
    }
    public void setEntries(String entries) {
        if (entries != null) {
            try {
                setEntries(DataNormalizer.normalizeInt(UtilsUrl.decode(entries), SdkConstants.ENTRIES, ackResolver));
            } catch (DataNormalizationException e) {
            } catch (IllegalArgumentException e) {
                ackResolver.addError(AckError.URL_ENCODING_ERROR, entries, SdkConstants.ENTRIES);
            } catch (UnsupportedEncodingException e) {
                ackResolver.addError(AckError.URL_ENCODING_ERROR, entries, SdkConstants.ENTRIES);
            }
        }
    }
    public void setEntries(int entries) {
        if (entries < 1) {
            ackResolver.addError(AckError.ENTRIES_UNDER_MINIMUM, String.valueOf(entries));
        } else if (entries > SdkConstants.API_MAXIMUM_ENTRIES) {
            ackResolver.addError(AckError.ENTRIES_EXCEEDED_MAXIMUM, String.valueOf(entries));
        } else {
            this.entries = entries;
        }
    }
    public String getTableName() {
        return this.metaData.getTable();
    }
    public List<String> getTableFields() {
        return this.metaData.getAllFields();
    }
    public List<String> getTableHierarchy() {
        return this.metaData.getTableHierarchy();
    }
}
