
package org.blufin.sdk.validators;
import org.blufin.base.constants.Constants;
import org.blufin.base.exceptions.BlufinAlertDeveloperException;
import org.blufin.base.helper.Quadruplet;
import org.blufin.base.helper.Triplet;
import org.blufin.sdk.base.AbstractMetaData;
import org.blufin.sdk.base.MetaDataException;
import org.blufin.sdk.constants.SdkConstants;
import org.blufin.sdk.response.AckError;
import org.blufin.sdk.response.AckResolver;
import org.blufin.sdk.response.AckWarning;
import java.text.MessageFormat;
public abstract class AbstractGetValidator {
    public static final String BLANK = "[BLANK]";
    public static final String FIELD = "Field";
    public static final String FIELD_NESTED = "Nested field";
    public static final String COLLECTION_MANY_TO_MANY = "many-to-many";
    public static final String COLLECTION_ONE_TO_ONE = "one-to-one";
    public static final String COLLECTION_ONE_TO_MANY = "one-to-many";
    protected static final Triplet<String, String, String> FRAGMENT_FILTERS = Triplet.of(SdkConstants.FILTERS, "filter", "filterable");
    protected static final Triplet<String, String, String> FRAGMENT_FIELDS = Triplet.of(SdkConstants.FIELDS, null, null);
    protected static final Triplet<String, String, String> FRAGMENT_SORT = Triplet.of(SdkConstants.SORT, "sort", "sortable");
    protected static Quadruplet<String, String, String, Boolean> resolveAndValidateFieldName(String fieldName, String tableName, AckResolver ackResolver, AbstractMetaData metaData, ValidationType validationType) {
        try {
            Triplet<String, String, String> textFragment;
            switch (validationType) {
                case FILTERS:
                    textFragment = FRAGMENT_FILTERS;
                    break;
                case FIELDS:
                    textFragment = FRAGMENT_FIELDS;
                    break;
                case SORT:
                    textFragment = FRAGMENT_SORT;
                    break;
                default:
                    throw new RuntimeException("Unhandled validation type: " + validationType.toString());
            }
            boolean isNested = false;
            if (!fieldName.equals(fieldName.toLowerCase())) {
                ackResolver.addWarning(AckWarning.FIELD_NAME_NOT_LOWERCASE, fieldName, fieldName.toLowerCase(), textFragment.getFirst());
                fieldName = fieldName.toLowerCase();
            }
            String fieldNameWithTable = fieldName;
            if (!fieldName.equals(fieldName.trim())) {
                ackResolver.addWarning(AckWarning.FIELD_NAME_CONTAINED_UNNECESSARY_WHITESPACE, fieldName, fieldName.trim().equals("") ? BLANK : fieldName.trim(), textFragment.getFirst());
                fieldName = fieldName.trim();
            }
            if (fieldName.contains(".")) {
                String[] fieldNameSplit = fieldName.split("\\.");
                if (fieldNameSplit.length == 2) {
                    tableName = fieldNameSplit[0];
                    fieldName = fieldNameSplit[1];
                    if (!tableName.equals(metaData.getTable())) {
                        isNested = true;
                        if (!metaData.getAllNestedTables().contains(tableName)) {
                            ackResolver.addError(AckError.FIELD_NAME_INVALID_NESTED_TABLE, tableName, textFragment.getFirst(), String.join(", ", metaData.getAllNestedTables()));
                            return null;
                        }
                        if (!metaData.getNestedMetaData(tableName).containsFieldExcludingObjects(fieldName)) {
                            if (metaData.getNestedMetaData(tableName).containsField(fieldName)) {
                                if (metaData.getNestedMetaData(tableName).isLinked(fieldName)) {
                                    ackResolver.addError(AckError.FIELD_HOLDS_COLLECTION, MessageFormat.format("{0}.{1}", tableName, fieldName), FIELD_NESTED, textFragment.getThird(), COLLECTION_MANY_TO_MANY, textFragment.getSecond());
                                } else if (metaData.getNestedMetaData(tableName).isObject(fieldName)) {
                                    ackResolver.addError(AckError.FIELD_HOLDS_COLLECTION, MessageFormat.format("{0}.{1}", tableName, fieldName), FIELD_NESTED, textFragment.getThird(), COLLECTION_ONE_TO_ONE, textFragment.getSecond());
                                } else if (metaData.getNestedMetaData(tableName).isObjectMap(fieldName)) {
                                    ackResolver.addError(AckError.FIELD_HOLDS_COLLECTION, MessageFormat.format("{0}.{1}", tableName, fieldName), FIELD_NESTED, textFragment.getThird(), COLLECTION_ONE_TO_MANY, textFragment.getSecond());
                                } else {
                                    throw new Exception(MessageFormat.format("Unsupported DataType for field {0}.{1}: {2}", tableName, fieldName, metaData.getNestedMetaData(tableName).getType(fieldName)));
                                }
                            } else {
                                ackResolver.addError(AckError.FIELD_NAME_INVALID_NESTED, fieldNameWithTable, textFragment.getFirst(), tableName, String.join(", ", metaData.getNestedMetaData(tableName).getAllFieldsIncludingContainers()));
                            }
                            return null;
                        }
                        if (metaData.getNestedMetaData(tableName).isEncrypted(fieldName)) {
                            ackResolver.addError(AckError.FIELD_IS_ENCRYPTED, fieldNameWithTable, textFragment.getThird());
                            return null;
                        }
                        if (validationType.equals(ValidationType.SORT)) {
                            ackResolver.addError(AckError.SORT_NESTED_NOT_SUPPORTED, fieldNameWithTable, textFragment.getThird());
                            return null;
                        }
                    }
                } else {
                    ackResolver.addError(AckError.FIELD_NAME_INVALID_SYNTAX, fieldName, textFragment.getFirst());
                    return null;
                }
            } else {
                fieldNameWithTable = metaData.getTable() + Constants.PERIOD + fieldName;
            }
            if (!isNested) {
                if (metaData.containsFieldExcludingObjects(fieldName)) {
                    if (metaData.isEncrypted(fieldName)) {
                        ackResolver.addError(AckError.FIELD_IS_ENCRYPTED, fieldName, textFragment.getThird());
                    }
                } else {
                    if (metaData.containsField(fieldName)) {
                        if (metaData.isLinked(fieldName)) {
                            ackResolver.addError(AckError.FIELD_HOLDS_COLLECTION, fieldName, FIELD, textFragment.getThird(), COLLECTION_MANY_TO_MANY, textFragment.getSecond());
                        } else if (metaData.isObject(fieldName)) {
                            ackResolver.addError(AckError.FIELD_HOLDS_COLLECTION, fieldName, FIELD, textFragment.getThird(), COLLECTION_ONE_TO_ONE, textFragment.getSecond());
                        } else if (metaData.isObjectMap(fieldName)) {
                            ackResolver.addError(AckError.FIELD_HOLDS_COLLECTION, fieldName, FIELD, textFragment.getThird(), COLLECTION_ONE_TO_MANY, textFragment.getSecond());
                        } else {
                            throw new Exception(MessageFormat.format("Unsupported DataType for field {0}: {1}", fieldName, metaData.getNestedMetaData(tableName).getType(fieldName)));
                        }
                    } else {
                        ackResolver.addError(AckError.FIELD_NAME_INVALID, fieldName.equals("") ? BLANK : fieldName, textFragment.getFirst(), String.join(", ", metaData.getAllFieldsIncludingContainers()));
                    }
                    return null;
                }
            }
            return Quadruplet.of(fieldName, tableName, fieldNameWithTable, isNested);
        } catch (MetaDataException e) {
            throw new BlufinAlertDeveloperException(MessageFormat.format("{0} threw a {1} which should never be happen. Error message was: {2}", AbstractGetValidator.class.getSimpleName(), MetaDataException.class.getSimpleName(), e.getMessage()), e);
        } catch (Exception e) {
            throw new BlufinAlertDeveloperException(e);
        }
    }
    protected enum ValidationType {
        FIELDS, FILTERS, SORT
    }
}
