
package org.blufin.sdk.validators;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.blufin.base.enums.DataType;
import org.blufin.base.exceptions.BlufinAlertDeveloperException;
import org.blufin.base.exceptions.BlufinNotImplementedException;
import org.blufin.base.helper.Quadruplet;
import org.blufin.base.utils.UtilsDate;
import org.blufin.jackson.Jackson;
import org.blufin.sdk.base.AbstractMetaData;
import org.blufin.sdk.base.MetaDataException;
import org.blufin.sdk.filters.Filter;
import org.blufin.sdk.filters.FilterContainer;
import org.blufin.sdk.filters.FilterContainerData;
import org.blufin.sdk.filters.FilterType;
import org.blufin.sdk.filters.Modifier;
import org.blufin.sdk.filters.interfaces.*;
import org.blufin.sdk.normalization.DataNormalizationException;
import org.blufin.sdk.normalization.DataNormalizer;
import org.blufin.sdk.response.AckError;
import org.blufin.sdk.response.AckResolver;
import org.blufin.sdk.response.AckWarning;
import java.io.IOException;
import java.text.MessageFormat;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
public final class FilterValidator extends AbstractGetValidator {
    private static final ObjectMapper objectMapper = Jackson.getMinimalObjectMapper();
    public static final String DESERIALIZER_EXPECTED_FORMAT = "'[\"id\",\">\",\"0\"]' OR '[[\"field-1\",\"==\",\"abc\"],[\"field-2\",\"=><=\",\"2015-01-01\",\"2015-02-01\"]]'";
    public static final FilterContainer deserialize(String filterString, AckResolver ackResolver, AbstractMetaData metaData) {
        if (filterString == null || filterString.trim().equals("")) {
            return null;
        } else {
            List<List<String>> deserializedData;
            try {
                try {
                    List<String> singleFilter = objectMapper.readValue(filterString, new TypeReference<List<String>>() {});
                    deserializedData = Arrays.asList(singleFilter);
                } catch (IOException e1) {
                    try {
                        deserializedData = objectMapper.readValue(filterString, new TypeReference<List<List<String>>>() {});
                    } catch (IOException e2) {
                        objectMapper.readValue(filterString, FilterContainerData.class);
                        throw new BlufinNotImplementedException();
                    }
                }
                List<Filter> filtersList = new ArrayList<>();
                validateDeserializedData(deserializedData, filtersList, ackResolver, metaData);
                if (filtersList.size() == 0) {
                    return null;
                }
                return new FilterContainer(filtersList);
            } catch (DataNormalizationException | IOException | BlufinNotImplementedException e) {
                ackResolver.addError(AckError.FILTER_DESERIALIZATION_FAILURE, DESERIALIZER_EXPECTED_FORMAT, filterString);
                return null;
            }
        }
    }
    private static void validateDeserializedData(List<List<String>> deserializedData, List<Filter> filterList, AckResolver ackResolver, AbstractMetaData metaData) throws DataNormalizationException {
        if (deserializedData == null || deserializedData.size() == 0) {
            throw new DataNormalizationException();
        }
        for (List<String> filterParameters : deserializedData) {
            try {
                if (filterParameters == null || filterParameters.size() == 0) {
                    throw new DataNormalizationException();
                } else {
                    for (String content : filterParameters) {
                        if (content == null) {
                            throw new DataNormalizationException();
                        }
                    }
                }
                Quadruplet<String, String, String, Boolean> fieldData = resolveAndValidateFieldName(filterParameters.get(0), metaData.getTable(), ackResolver, metaData, ValidationType.FILTERS);
                if (fieldData == null) {
                    continue;
                }
                String fieldName = fieldData.getFirst();
                String tableName = fieldData.getSecond();
                String fieldNameWithTable = fieldData.getThird();
                boolean isNested = fieldData.getFourth();
                if (filterParameters.size() == 1) {
                    ackResolver.addError(AckError.FILTER_MISSING_MODIFIER, fieldNameWithTable);
                    continue;
                }
                AbstractMetaData metaDataForThisField = isNested ? metaData.getNestedMetaData(tableName) : metaData;
                FilterType filterType = filterType(fieldName, metaDataForThisField);
                Modifier modifier = Modifier.get(filterParameters.get(1));
                if (modifier == null) {
                    if (filterParameters.get(1).trim().equals("")) {
                        ackResolver.addError(AckError.FILTER_INVALID_MODIFIER_BLANK, filterParameters.get(0));
                    } else {
                        ackResolver.addError(AckError.FILTER_INVALID_MODIFIER, filterParameters.get(1), filterParameters.get(0));
                    }
                    continue;
                } else if (!modifierValidForType(filterType, modifier)) {
                    ackResolver.addError(AckError.FILTER_INVALID_MODIFIER_FOR_TYPE, modifier.toString(), fieldNameWithTable, filterType.name());
                    continue;
                }
                if (filterParameters.size() == 2) {
                    ackResolver.addError(AckError.FILTER_MISSING_PRIMARY_VALUE, fieldNameWithTable);
                    continue;
                }
                Filter filterFinal;
                String primaryValue = filterParameters.get(2);
                int expectedFilterDataSize;
                if (modifier.equals(Modifier.BETWEEN) || modifier.equals(Modifier.BETWEEN_OR_EQUAL)) {
                    if (filterParameters.size() == 3) {
                        ackResolver.addError(AckError.FILTER_MISSING_SECONDARY_VALUE, fieldNameWithTable);
                        continue;
                    }
                    String secondaryValue = filterParameters.get(3);
                    filterFinal = assembleFilter(metaDataForThisField, ackResolver, filterType, modifier, fieldNameWithTable, primaryValue, secondaryValue);
                    expectedFilterDataSize = 4;
                } else {
                    filterFinal = assembleFilter(metaDataForThisField, ackResolver, filterType, modifier, fieldNameWithTable, primaryValue);
                    expectedFilterDataSize = 3;
                }
                if (filterParameters.size() > expectedFilterDataSize) {
                    List<String> ignoredParameters = new ArrayList<>();
                    for (int i = 0; i < filterParameters.size(); i++) {
                        if (i >= (expectedFilterDataSize)) {
                            ignoredParameters.add(filterParameters.get(i));
                        }
                    }
                    ackResolver.addWarning(AckWarning.FILTER_TOO_MANY_PARAMETERS, ignoredParameters.toString(), fieldNameWithTable, String.valueOf(expectedFilterDataSize), String.valueOf(filterParameters.size()));
                }
                if (filterFinal != null) {
                    filterList.add(filterFinal);
                }
            } catch (MetaDataException e) {
                throw new BlufinAlertDeveloperException(MessageFormat.format("{0} threw a {1} which should never be happen. Error message was: {2}", FilterValidator.class.getSimpleName(), MetaDataException.class.getSimpleName(), e.getMessage()), e);
            }
        }
    }
    private static final FilterType filterType(String fieldName, AbstractMetaData metaData) throws MetaDataException {
        DataType dataType = metaData.getType(fieldName);
        switch (dataType) {
            case BOOLEAN:
                return FilterType.BOOLEAN;
            case DATE:
                return FilterType.DATE;
            case DATETIME:
            case DATETIME_INSERT:
            case DATETIME_UPDATE:
                return FilterType.DATETIME;
            case DECIMAL:
                return FilterType.DECIMAL;
            case ENUM:
            case ENUM_CUSTOM:
            case ENUM_SYSTEM:
                return FilterType.ENUM;
            case INT:
            case INT_AUTO:
                return FilterType.INTEGER;
            case INT_TINY:
                return FilterType.INTEGER_TINY;
            case INT_SMALL:
                return FilterType.INTEGER_SMALL;
            case INT_BIG:
                return FilterType.INTEGER_BIG;
            case TEXT:
                return FilterType.TEXT;
            case TEXT_LONG:
                return FilterType.TEXT_LONG;
            case VARCHAR:
                return FilterType.VARCHAR;
            default:
                throw new RuntimeException(MessageFormat.format("Unable to map DataType.{0} to an equivalent FilterType.", dataType.name()));
        }
    }
    private static final boolean modifierValidForType(FilterType filterType, Modifier modifier) {
        switch (filterType) {
            case BOOLEAN:
                return BooleanFilterable.modifiers.contains(modifier);
            case DATE:
            case DATETIME:
                return MatchComparisonFilterable.modifiers.contains(modifier);
            case ENUM:
                return MatchFilterable.modifiers.contains(modifier);
            case DECIMAL:
            case INTEGER:
            case INTEGER_BIG:
            case INTEGER_SMALL:
            case INTEGER_TINY:
                return IntegerFilterable.modifiers.contains(modifier);
            case TEXT:
            case TEXT_LONG:
                return ContentFilterable.modifiers.contains(modifier);
            case VARCHAR:
                return MatchContentFilterable.modifiers.contains(modifier);
            default:
                throw new RuntimeException(MessageFormat.format("Unsupported FilterType: {0}", filterType.name()));
        }
    }
    private static final Filter assembleFilter(AbstractMetaData metaData, AckResolver ackResolver, FilterType filterType, Modifier modifier, String fieldName, String primaryValue) throws MetaDataException {
        String normalizedPrimaryValue = normalize(primaryValue, fieldName, ackResolver, metaData, filterType);
        return new Filter(filterType, fieldName, modifier, normalizedPrimaryValue);
    }
    private static final Filter assembleFilter(AbstractMetaData metaData, AckResolver ackResolver, FilterType filterType, Modifier modifier, String fieldNameWithTable, String primaryValue, String secondaryValue) throws MetaDataException {
        String normalizedPrimaryValue = normalize(primaryValue, fieldNameWithTable, ackResolver, metaData, filterType);
        String normalizedSecondaryValue = normalize(secondaryValue, fieldNameWithTable, ackResolver, metaData, filterType);
        if (ackResolver.getErrors().size() > 0) {
            return null;
        }
        Filter filter = new Filter(filterType, fieldNameWithTable, modifier, normalizedPrimaryValue, normalizedSecondaryValue);
        if (modifier.equals(Modifier.BETWEEN) || modifier.equals(Modifier.BETWEEN_OR_EQUAL)) {
            if (filterType.equals(FilterType.DATE)) {
                LocalDate dateOne = LocalDate.parse(normalizedPrimaryValue);
                LocalDate dateTwo = LocalDate.parse(normalizedSecondaryValue);
                if (dateOne.isAfter(dateTwo)) {
                    ackResolver.addWarning(AckWarning.FILTER_DATE_REVERSED, normalizedSecondaryValue, modifier.toString(), normalizedPrimaryValue, fieldNameWithTable);
                    filter = new Filter(filterType, fieldNameWithTable, modifier, normalizedSecondaryValue, normalizedPrimaryValue);
                }
            } else if (filterType.equals(FilterType.DATETIME)) {
                ZonedDateTime dateOne = ZonedDateTime.parse(normalizedPrimaryValue);
                ZonedDateTime dateTwo = ZonedDateTime.parse(normalizedSecondaryValue);
                if (dateOne.isAfter(dateTwo)) {
                    ackResolver.addWarning(AckWarning.FILTER_DATETIME_REVERSED, normalizedSecondaryValue, modifier.toString(), normalizedPrimaryValue, fieldNameWithTable);
                    filter = new Filter(filterType, fieldNameWithTable, modifier, normalizedSecondaryValue, normalizedPrimaryValue);
                }
            } else {
                if (Long.parseLong(normalizedPrimaryValue) > Long.parseLong(normalizedSecondaryValue)) {
                    ackResolver.addWarning(AckWarning.FILTER_VALUES_REVERSED, normalizedSecondaryValue, modifier.toString(), normalizedPrimaryValue, fieldNameWithTable);
                    filter = new Filter(filterType, fieldNameWithTable, modifier, normalizedSecondaryValue, normalizedPrimaryValue);
                }
            }
        }
        return filter;
    }
    public static final String normalize(String value, String fieldNameWithTable, AckResolver ackResolver, AbstractMetaData metaData, FilterType filterType) throws MetaDataException {
        try {
            switch (filterType) {
                case BOOLEAN:
                    return String.valueOf(DataNormalizer.normalizeBoolean(value, fieldNameWithTable, ackResolver));
                case DATE:
                    return UtilsDate.convertLocalDateToString(DataNormalizer.normalizeDate(value, fieldNameWithTable, ackResolver));
                case DATETIME:
                    return UtilsDate.convertZonedDateTimeToString(DataNormalizer.normalizeDateTime(value, fieldNameWithTable, ackResolver));
                case DECIMAL:
                    return DataNormalizer.normalizeDecimal(value, fieldNameWithTable, ackResolver, metaData.getDecimalDistribution(fieldNameWithTable.split("\\.")[1])).toString();
                case ENUM:
                    return DataNormalizer.normalizeStringEnum(value, fieldNameWithTable, ackResolver, metaData.getEnumValues(fieldNameWithTable));
                case INTEGER:
                    return String.valueOf(DataNormalizer.normalizeInt(value, fieldNameWithTable, ackResolver));
                case INTEGER_BIG:
                    return String.valueOf(DataNormalizer.normalizeIntBig(value, fieldNameWithTable, ackResolver));
                case INTEGER_SMALL:
                    return String.valueOf(DataNormalizer.normalizeIntSmall(value, fieldNameWithTable, ackResolver));
                case INTEGER_TINY:
                    return String.valueOf(DataNormalizer.normalizeIntTiny(value, fieldNameWithTable, ackResolver));
                case TEXT:
                case TEXT_LONG:
                case VARCHAR:
                    return DataNormalizer.normalizeString(value, fieldNameWithTable, ackResolver);
                default:
                    throw new RuntimeException(MessageFormat.format("Unsupported FilterType: {0}", filterType.name()));
            }
        } catch (DataNormalizationException e) {
            return null;
        }
    }
}
