/*
 * Decompiled with CFR 0.152.
 */
package org.blufin.sdk.normalization;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.time.DateTimeException;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.blufin.base.enums.TimeZone;
import org.blufin.base.helper.Pair;
import org.blufin.base.helper.Triplet;
import org.blufin.base.utils.UtilsRegex;
import org.blufin.sdk.normalization.DataNormalizationException;
import org.blufin.sdk.normalization.DateTimeSplitter;
import org.blufin.sdk.normalization.DecimalHelper;
import org.blufin.sdk.response.AckError;
import org.blufin.sdk.response.AckResolver;
import org.blufin.sdk.response.AckWarning;

public class DataNormalizer {
    public static final String ENUM_SEPARATOR = ", ";
    private static final Pattern REGEX_INTEGER = Pattern.compile("^(-)?\\d+$");
    private static final Pattern REGEX_DECIMAL = Pattern.compile("^(-)?[\\d\\.]+$");
    private static final Pattern REGEX_ZEROS = Pattern.compile("^[0]+$");
    private static final Pattern REGEX_MILLISECONDS = Pattern.compile("^\\d{3}0+$");

    public static final String normalizeString(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        return DataNormalizer.normalizeString(value, element, ackResolver, 0);
    }

    public static final String normalizeString(String value, String element, AckResolver ackResolver, Integer maxLength) throws DataNormalizationException {
        if (maxLength != null && maxLength > 0 && value.length() > maxLength) {
            ackResolver.addError(AckError.STRING_MAX_LENGTH_EXCEEDED, value, element, String.valueOf(maxLength), String.valueOf(value.length()));
            throw new DataNormalizationException();
        }
        return value;
    }

    public static final String normalizeStringEnum(String value, String element, AckResolver ackResolver, List<String> enumValues) throws DataNormalizationException {
        value = DataNormalizer.trimWhitespace(value);
        String valueUpperCase = value.toUpperCase();
        DataNormalizer.checkValueIsNotBlank(valueUpperCase, ackResolver, AckError.STRING_ENUM_BLANK, element);
        if (!enumValues.contains(valueUpperCase)) {
            ackResolver.addError(AckError.STRING_ENUM_INVALID, value, element, StringUtils.join(enumValues, (String)ENUM_SEPARATOR));
            throw new DataNormalizationException();
        }
        if (!value.equals(valueUpperCase)) {
            ackResolver.addWarning(AckWarning.STRING_ENUM_NOT_UPPERCASE, value, valueUpperCase, element);
        }
        return valueUpperCase;
    }

    public static final boolean normalizeBoolean(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        value = DataNormalizer.trimWhitespace(value);
        String valueLowercase = value.toLowerCase();
        DataNormalizer.checkValueIsNotBlank(valueLowercase, ackResolver, AckError.BOOLEAN_BLANK, element);
        if (valueLowercase.equals("true") || valueLowercase.equals("false")) {
            if (!value.equals(valueLowercase)) {
                ackResolver.addWarning(AckWarning.BOOLEAN_NOT_LOWERCASE, value, value.toLowerCase(), element);
            }
            return Boolean.valueOf(valueLowercase);
        }
        ackResolver.addError(AckError.BOOLEAN_INVALID, value, element);
        throw new DataNormalizationException();
    }

    public static final int normalizeInt(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = DataNormalizer.convertToWholeNumber(value, element, ackResolver, AckError.INT_CONTAINED_DECIMAL_POINT, AckError.INT_INVALID, AckError.INT_BLANK, AckWarning.INT_CONTAINED_LEADING_ZEROS);
            return Integer.parseInt(value);
        }
        catch (NumberFormatException e) {
            ackResolver.addError(AckError.INT_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }

    public static final long normalizeIntBig(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = DataNormalizer.convertToWholeNumber(value, element, ackResolver, AckError.LONG_CONTAINED_DECIMAL_POINT, AckError.LONG_INVALID, AckError.LONG_BLANK, AckWarning.LONG_CONTAINED_LEADING_ZEROS);
            return Long.parseLong(value);
        }
        catch (NumberFormatException e) {
            ackResolver.addError(AckError.LONG_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }

    public static final short normalizeIntSmall(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = DataNormalizer.convertToWholeNumber(value, element, ackResolver, AckError.SHORT_CONTAINED_DECIMAL_POINT, AckError.SHORT_INVALID, AckError.SHORT_BLANK, AckWarning.SHORT_CONTAINED_LEADING_ZEROS);
            return Short.parseShort(value);
        }
        catch (NumberFormatException e) {
            ackResolver.addError(AckError.SHORT_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }

    public static final byte normalizeIntTiny(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        try {
            value = DataNormalizer.convertToWholeNumber(value, element, ackResolver, AckError.BYTE_CONTAINED_DECIMAL_POINT, AckError.BYTE_INVALID, AckError.BYTE_BLANK, AckWarning.BYTE_CONTAINED_LEADING_ZEROS);
            return Byte.parseByte(value);
        }
        catch (NumberFormatException e) {
            ackResolver.addError(AckError.BYTE_OUT_OF_BOUNDS, value, element);
            throw new DataNormalizationException();
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    public static final BigDecimal normalizeDecimal(String value, String element, AckResolver ackResolver, Pair decimalDistribution) throws DataNormalizationException {
        BigDecimal fin;
        value = DataNormalizer.trimWhitespace(value);
        DataNormalizer.checkValueIsNotBlank(value, ackResolver, AckError.DECIMAL_BLANK, element);
        String valueWithLeadingZerosRemoved = DataNormalizer.removeLeadingZeros(value, element, ackResolver, AckWarning.DECIMAL_CONTAINED_LEADING_ZEROS);
        int m = (Integer)decimalDistribution.getKey();
        int d = (Integer)decimalDistribution.getValue();
        if (UtilsRegex.stringMatchesRegex((String)valueWithLeadingZerosRemoved, (Pattern)REGEX_INTEGER)) {
            String valueWithDecimal = valueWithLeadingZerosRemoved + "." + StringUtils.repeat((String)"0", (int)d);
            return new BigDecimal(valueWithDecimal);
        }
        if (!UtilsRegex.stringMatchesRegex((String)valueWithLeadingZerosRemoved, (Pattern)REGEX_DECIMAL)) {
            ackResolver.addError(AckError.DECIMAL_INVALID, valueWithLeadingZerosRemoved, element);
            throw new DataNormalizationException();
        }
        String[] parts = valueWithLeadingZerosRemoved.split("\\.");
        if (parts.length == 1) {
            fin = new BigDecimal(parts[0] + "." + StringUtils.repeat((String)"0", (int)d));
        } else {
            if (parts.length != 2) {
                ackResolver.addError(AckError.DECIMAL_INVALID, valueWithLeadingZerosRemoved, element);
                throw new DataNormalizationException();
            }
            if (parts[0].equals("-")) {
                parts[0] = "-0";
            } else if (parts[0].equals("")) {
                parts[0] = "0";
            }
            boolean partsNegative = false;
            if (parts[0].substring(0, 1).equals("-")) {
                parts[0] = parts[0].substring(1);
                partsNegative = true;
            }
            if (parts[1].length() > d) {
                if (!UtilsRegex.stringMatchesRegex((String)parts[1].substring(d), (Pattern)REGEX_ZEROS)) {
                    ackResolver.addError(AckError.DECIMAL_TOO_GRANULAR, valueWithLeadingZerosRemoved, element, String.valueOf(m), String.valueOf(d));
                    throw new DataNormalizationException();
                }
                parts[1] = parts[1].substring(0, d);
                ackResolver.addWarning(AckWarning.DECIMAL_CONTAINED_TRAILING_ZEROS, valueWithLeadingZerosRemoved, (partsNegative ? "-" : "") + parts[0] + "." + parts[1], element, String.valueOf(m), String.valueOf(d));
            } else if (parts[1].length() < d) {
                parts[1] = parts[1] + StringUtils.repeat((String)"0", (int)(d - parts[1].length()));
            }
            fin = new BigDecimal((partsNegative ? "-" : "") + parts[0] + "." + parts[1]);
        }
        BigDecimal min = DecimalHelper.getMinimumAllowedValue(decimalDistribution);
        BigDecimal max = DecimalHelper.getMaximumAllowedValue(decimalDistribution);
        if (fin.compareTo(min) != -1 && fin.compareTo(max) != 1) {
            return fin;
        }
        ackResolver.addError(AckError.DECIMAL_OUT_OF_BOUNDS, valueWithLeadingZerosRemoved, element, String.valueOf(min), String.valueOf(max));
        throw new DataNormalizationException();
    }

    public static final LocalDate normalizeDate(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        if (value == null || value.trim().equals("")) {
            ackResolver.addError(AckError.DATE_BLANK, element);
            throw new DataNormalizationException();
        }
        value = DataNormalizer.trimWhitespace(value);
        try {
            return LocalDate.parse(value);
        }
        catch (Exception e1) {
            try {
                String[] datePartsModified = DataNormalizer.getModifiedDateParts(value.split("-"));
                LocalDate modifiedDate = LocalDate.of(Integer.parseInt(datePartsModified[0]), Integer.parseInt(datePartsModified[1]), Integer.parseInt(datePartsModified[2]));
                ackResolver.addWarning(AckWarning.DATE_UNCONVENTIONAL_FORMAT, value, modifiedDate.toString(), element);
                return modifiedDate;
            }
            catch (NumberFormatException | DateTimeException | DataNormalizationException exception) {
                ackResolver.addError(AckError.DATE_INVALID_FORMAT, value, element);
                throw new DataNormalizationException();
            }
        }
    }

    public static final ZonedDateTime normalizeDateTime(String value, String element, AckResolver ackResolver) throws DataNormalizationException {
        return DataNormalizer.normalizeDateTime(value, element, ackResolver, TimeZone.UTC);
    }

    public static final ZonedDateTime normalizeDateTime(String value, String element, AckResolver ackResolver, TimeZone timeZone) throws DataNormalizationException {
        if (value == null || value.trim().equals("")) {
            ackResolver.addError(AckError.DATETIME_BLANK, element);
            throw new DataNormalizationException();
        }
        value = DataNormalizer.trimWhitespace(value);
        int initialErrorCount = ackResolver.getErrors().size();
        try {
            boolean dateUnconventional = false;
            String[] dateTimeSplit = value.split("T");
            if (dateTimeSplit.length != 2) {
                throw new DataNormalizationException();
            }
            Object[] dateParts = dateTimeSplit[0].split("-");
            Object[] datePartsModified = DataNormalizer.getModifiedDateParts((String[])dateParts);
            if (!StringUtils.join((Object[])dateParts, (String)"-").equals(StringUtils.join((Object[])datePartsModified, (String)"-"))) {
                dateUnconventional = true;
            }
            Triplet<String, String, String> timeSecondsAndZone = DateTimeSplitter.splitTimeNanoAndZone(dateTimeSplit[1], element, ackResolver);
            Object[] timePartsModified = DataNormalizer.getModifiedTimeParts(((String)timeSecondsAndZone.getFirst()).split(":"));
            String t = StringUtils.join((Object[])timePartsModified, (String)":");
            String m = (String)timeSecondsAndZone.getSecond();
            String z = (String)timeSecondsAndZone.getThird();
            if (z == null) {
                z = timeZone.equals((Object)TimeZone.UTC) ? TimeZone.UTC.getZoneIdDisplay() : TimeZone.UTC.getZoneIdDisplay() + "[" + timeZone.getZoneIdDisplay() + "]";
            }
            if (m == null || m.equals("")) {
                m = "000";
            } else if (m.length() < 3) {
                m = StringUtils.rightPad((String)m, (int)3, (String)"0");
            } else if (m.length() > 3) {
                if (UtilsRegex.stringMatchesRegex((String)m, (Pattern)REGEX_MILLISECONDS)) {
                    m = m.substring(0, 3);
                } else {
                    ackResolver.addError(AckError.DATETIME_TIME_TOO_GRANULAR, value, element);
                    throw new DataNormalizationException();
                }
            }
            String finalDatetimeString = StringUtils.join((Object[])datePartsModified, (String)"-") + "T" + t + "." + m + z;
            if (dateUnconventional) {
                ackResolver.addWarning(AckWarning.DATETIME_UNCONVENTIONAL_FORMAT, value, finalDatetimeString, element);
            }
            try {
                return ZonedDateTime.parse(finalDatetimeString, DateTimeFormatter.ISO_ZONED_DATE_TIME);
            }
            catch (Exception e) {
                throw new DataNormalizationException();
            }
        }
        catch (NumberFormatException | DataNormalizationException e) {
            if (ackResolver.getErrors().size() == initialErrorCount) {
                ackResolver.addError(AckError.DATETIME_INVALID_FORMAT, value, element);
            }
            throw new DataNormalizationException();
        }
    }

    private static String convertToWholeNumber(String value, String element, AckResolver ackResolver, AckError containsDecimalPointError, AckError invalidError, AckError blankError, AckWarning containsLeadingZerosWarning) throws DataNormalizationException {
        value = DataNormalizer.trimWhitespace(value);
        DataNormalizer.checkValueIsNotBlank(value, ackResolver, blankError, element);
        if (UtilsRegex.stringMatchesRegex((String)value.trim(), (Pattern)REGEX_INTEGER)) {
            value = DataNormalizer.removeLeadingZeros(value, element, ackResolver, containsLeadingZerosWarning);
            return value;
        }
        if (UtilsRegex.stringMatchesRegex((String)value.trim(), (Pattern)REGEX_DECIMAL)) {
            int decimalPointMatches = StringUtils.countMatches((CharSequence)(value = DataNormalizer.removeLeadingZeros(value, element, ackResolver, containsLeadingZerosWarning)), (CharSequence)".");
            if (decimalPointMatches == 0) {
                return value;
            }
            if (decimalPointMatches == 1) {
                ackResolver.addError(containsDecimalPointError, value, element);
                throw new DataNormalizationException();
            }
        }
        ackResolver.addError(invalidError, value, element);
        throw new DataNormalizationException();
    }

    private static String removeLeadingZeros(String value, String element, AckResolver ackResolver, AckWarning containsLeadingZerosWarning) {
        boolean valueNegative = false;
        if (value.substring(0, 1).equals("-")) {
            value = value.substring(1);
            valueNegative = true;
        }
        if (value.equals("0")) {
            return value;
        }
        if (value.length() >= 2 && value.substring(0, 2).equals("0.")) {
            return (valueNegative ? "-" : "") + value;
        }
        String valueWithoutLeadingZeros = StringUtils.stripStart((String)value, (String)"0");
        if (valueWithoutLeadingZeros.equals("")) {
            valueWithoutLeadingZeros = "0";
        }
        String adjustedAmount = (valueNegative && !valueWithoutLeadingZeros.equals("0") ? "-" : "") + valueWithoutLeadingZeros;
        if (!value.equals(valueWithoutLeadingZeros)) {
            ackResolver.addWarning(containsLeadingZerosWarning, (valueNegative ? "-" : "") + value, adjustedAmount, element);
        }
        return adjustedAmount;
    }

    private static String[] getModifiedDateParts(String[] dateParts) throws DataNormalizationException {
        if (dateParts.length != 3) {
            throw new DataNormalizationException();
        }
        int idx = -1;
        String[] datePartsModified = new String[3];
        block4: for (String datePart : dateParts) {
            if (datePart == null || datePart.equals("") || !datePart.trim().equals(datePart)) {
                throw new DataNormalizationException();
            }
            switch (++idx) {
                case 0: {
                    if (Integer.parseInt(datePart) < 0) {
                        throw new DataNormalizationException();
                    }
                    if (datePart.length() <= 4) {
                        datePartsModified[idx] = StringUtils.leftPad((String)datePart, (int)4, (String)"0");
                        continue block4;
                    }
                    throw new DataNormalizationException();
                }
                case 1: 
                case 2: {
                    if (Integer.parseInt(datePart) <= 0) {
                        throw new DataNormalizationException();
                    }
                    if (datePart.length() <= 2) {
                        datePartsModified[idx] = StringUtils.leftPad((String)datePart, (int)2, (String)"0");
                        continue block4;
                    }
                    throw new DataNormalizationException();
                }
                default: {
                    throw new RuntimeException(MessageFormat.format("Index out of bounds: {0} \u2014 This statement should never be reached.", idx));
                }
            }
        }
        return datePartsModified;
    }

    private static String[] getModifiedTimeParts(String[] timeParts) throws DataNormalizationException {
        if (timeParts.length != 3) {
            throw new DataNormalizationException();
        }
        int idx = -1;
        String[] timePartsModified = new String[3];
        for (String timePart : timeParts) {
            if (timePart == null || timePart.equals("") || !timePart.trim().equals(timePart)) {
                throw new DataNormalizationException();
            }
            ++idx;
            if (timePart.length() > 2) {
                throw new DataNormalizationException();
            }
            timePartsModified[idx] = StringUtils.leftPad((String)timePart, (int)2, (String)"0");
        }
        return timePartsModified;
    }

    private static String trimWhitespace(String value) {
        if (value == null) {
            return "";
        }
        return value.trim();
    }

    private static void checkValueIsNotBlank(String value, AckResolver ackResolver, AckError blankErrorMessage, String element) throws DataNormalizationException {
        if (value.trim().equals("")) {
            ackResolver.addError(blankErrorMessage, element);
            throw new DataNormalizationException();
        }
    }
}

