package org.bndly.common.html;

/*-
 * #%L
 * HTML
 * %%
 * Copyright (C) 2013 - 2020 Cybercon GmbH
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 *
 * @author cybercon &lt;bndly@cybercon.de&gt;
 */
public final class HTML5EntityMap {
	public static final HTML5EntityMap INSTANCE;
	private class Entry {
		private final char character;
		private final String[] entities;
		private final String defaultName;

		public Entry(char character, String... entities) {
			this.character = character;
			if (entities == null || entities.length < 1) {
				throw new IllegalArgumentException("a entity map entry requires at least on entity");
			}
			this.entities = entities;
			for (String entity : entities) {
				if (entity.length() < 2) {
					throw new IllegalArgumentException("entities need at least 2 characters");
				}
				if (entity.charAt(0) != '&') {
					throw new IllegalArgumentException("entity did not start with &");
				}
				if (entity.charAt(entity.length() - 1) != ';') {
					throw new IllegalArgumentException("entity did not end with ;");
				}
			}
			defaultName = entities[0].substring(1, entities[0].length() - 1);
		}

		public final char getCharacter() {
			return character;
		}

		public final String[] getEntities() {
			return entities;
		}

		public final String getDefaultEntityName() {
			return defaultName;
		}
		
	}
	
	private final List<Entry> entries = new ArrayList<>();
	private final Map<Character, Entry> entriesByChar = new LinkedHashMap<>();
	private final Map<String, Entry> entriesByEntity = new LinkedHashMap<>();

	private HTML5EntityMap() {
	}
	
	static {
		INSTANCE = new HTML5EntityMap();
		INSTANCE.init();
	}
	public boolean hasEntity(char c) {
		return entriesByChar.containsKey(c);
	}
	public String getEntityName(char c) {
		Entry get = entriesByChar.get(c);
		if (get == null) {
			return null;
		}
		return get.getDefaultEntityName();
	}

	public String getEntity(char c) {
		Entry get = entriesByChar.get(c);
		if (get == null) {
			return null;
		}
		return get.entities[0];
	}
	
	private void init() {
		entries.add(new Entry('\t', "&Tab;", "&#x00009;", "&#9;"));
		entries.add(new Entry('\n', "&NewLine;","&#x0000A;","&#10;"));
		entries.add(new Entry('!', "&excl;","&#x00021;","&#33;"));
		entries.add(new Entry('"',"&quot; &QUOT;","&#x00022;","&#34;"));
		entries.add(new Entry('#',"&num;","&#x00023;","&#35;"));
		entries.add(new Entry('$',"&dollar;","&#x00024;","&#36;"));
		entries.add(new Entry('%',"&percnt;","&#x00025;","&#37;"));
		entries.add(new Entry('&',"&amp;","&AMP;","&#x00026;","&#38;"));
		entries.add(new Entry('\'',"&apos;","&#x00027;","&#39;"));
		entries.add(new Entry('(',"&lpar;","&#x00028;","&#40;"));
		entries.add(new Entry(')',"&rpar;","&#x00029;","&#41;"));
		entries.add(new Entry('*',"&ast;","&midast;","&#x0002A;","&#42;"));
		entries.add(new Entry('+',"&plus;","&#x0002B;","&#43;"));
		entries.add(new Entry(',',"&comma;","&#x0002C;","&#44;"));
		entries.add(new Entry('.',"&period;","&#x0002E;","&#46;"));
		entries.add(new Entry('/',"&sol;","&#x0002F;","&#47;"));
		entries.add(new Entry(':',"&colon;","&#x0003A;","&#58;"));
		entries.add(new Entry(';',"&semi;","&#x0003B;","&#59;"));
		entries.add(new Entry('<',"&lt;","&LT;","&#x0003C;","&#60;"));
		entries.add(new Entry('=',"&equals;","&#x0003D;","&#61;"));
		entries.add(new Entry('>',"&gt;","&GT;","&#x0003E;","&#62;"));
		entries.add(new Entry('?',"&quest;","&#x0003F;","&#63;"));
		entries.add(new Entry('@',"&commat;","&#x00040;","&#64;"));
		entries.add(new Entry('[',"&lsqb;","&lbrack;","&#x0005B;","&#91;"));
		entries.add(new Entry('\\',"&bsol;","&#x0005C;","&#92;"));
		entries.add(new Entry(']',"&rsqb;","&rbrack;","&#x0005D;","&#93;"));
		entries.add(new Entry('^',"&Hat;","&#x0005E;","&#94;"));
		entries.add(new Entry('_',"&lowbar;","&#x0005F;","&#95;"));
		entries.add(new Entry('`',"&grave;","&DiacriticalGrave;","&#x00060;","&#96;"));
		entries.add(new Entry('{',"&lcub;","&lbrace;","&#x0007B;","&#123;"));
		entries.add(new Entry('|',"&verbar;","&vert;","&VerticalLine;","&#x0007C;","&#124;"));
		entries.add(new Entry('}',"&rcub;","&rbrace;","&#x0007D;","&#125;"));
		entries.add(new Entry(' ',"&nbsp;","&NonBreakingSpace;","&#x000A0;","&#160;"));
		entries.add(new Entry('¡',"&iexcl;","&#x000A1;","&#161;"));
		entries.add(new Entry('¢',"&cent;","&#x000A2;","&#162;"));
		entries.add(new Entry('£',"&pound;","&#x000A3;","&#163;"));
		entries.add(new Entry('¤',"&curren;","&#x000A4;","&#164;"));
		entries.add(new Entry('¥',"&yen;","&#x000A5;","&#165;"));
		entries.add(new Entry('¦',"&brvbar;","&#x000A6;","&#166;"));
		entries.add(new Entry('§',"&sect;","&#x000A7;","&#167;"));
		entries.add(new Entry('¨',"&Dot;","&die;","&DoubleDot;","&uml;","&#x000A8;","&#168;"));
		entries.add(new Entry('©',"&copy;","&COPY;","&#x000A9;","&#169;"));
		entries.add(new Entry('ª',"&ordf;","&#x000AA;","&#170;"));
		entries.add(new Entry('«',"&laquo;","&#x000AB;","&#171;"));
		entries.add(new Entry('¬',"&not;","&#x000AC;","&#172;"));
		entries.add(new Entry((char)0x000AD,"&shy;","&#x000AD;","&#173;"));
		entries.add(new Entry('®',"&reg;","&circledR;","&REG;","&#x000AE;","&#174;"));
		entries.add(new Entry('¯',"&macr;","&OverBar;","&strns;","&#x000AF;","&#175;"));
		entries.add(new Entry('°',"&deg;","&#x000B0;","&#176;"));
		entries.add(new Entry('±',"&plusmn;","&pm;","&PlusMinus;","&#x000B1;","&#177;"));
		entries.add(new Entry('²',"&sup2;","&#x000B2;","&#178;"));
		entries.add(new Entry('³',"&sup3;","&#x000B3;","&#179;"));
		entries.add(new Entry('´',"&acute;","&DiacriticalAcute;","&#x000B4;","&#180;"));
		entries.add(new Entry('µ',"&micro;","&#x000B5;","&#181;"));
		entries.add(new Entry('¶',"&para;","&#x000B6;","&#182;"));
		entries.add(new Entry('·',"&middot;","&centerdot;","&CenterDot;","&#x000B7;","&#183;"));
		entries.add(new Entry('¸',"&cedil;","&Cedilla;","&#x000B8;","&#184;"));
		entries.add(new Entry('¹',"&sup1;","&#x000B9;","&#185;"));
		entries.add(new Entry('º',"&ordm;","&#x000BA;","&#186;"));
		entries.add(new Entry('»',"&raquo;","&#x000BB;","&#187;"));
		entries.add(new Entry('¼',"&frac14;","&#x000BC;","&#188;"));
		entries.add(new Entry('½',"&frac12;","&half;","&#x000BD;","&#189;"));
		entries.add(new Entry('¾',"&frac34;","&#x000BE;","&#190;"));
		entries.add(new Entry('¿',"&iquest;","&#x000BF;","&#191;"));
		entries.add(new Entry('À',"&Agrave;","&#x000C0;","&#192;"));
		entries.add(new Entry('Á',"&Aacute;","&#x000C1;","&#193;"));
		entries.add(new Entry('Â',"&Acirc;","&#x000C2;","&#194;"));
		entries.add(new Entry('Ã',"&Atilde;","&#x000C3;","&#195;"));
		entries.add(new Entry('Ä',"&Auml;","&#x000C4;","&#196;"));
		entries.add(new Entry('Å',"&Aring;","&#x000C5;","&#197;"));
		entries.add(new Entry('Æ',"&AElig;","&#x000C6;","&#198;"));
		entries.add(new Entry('Ç',"&Ccedil;","&#x000C7;","&#199;"));
		entries.add(new Entry('È',"&Egrave;","&#x000C8;","&#200;"));
		entries.add(new Entry('É',"&Eacute;","&#x000C9;","&#201;"));
		entries.add(new Entry('Ê',"&Ecirc;","&#x000CA;","&#202;"));
		entries.add(new Entry('Ë',"&Euml;","&#x000CB;","&#203;"));
		entries.add(new Entry('Ì',"&Igrave;","&#x000CC;","&#204;"));
		entries.add(new Entry('Í',"&Iacute;","&#x000CD;","&#205;"));
		entries.add(new Entry('Î',"&Icirc;","&#x000CE;","&#206;"));
		entries.add(new Entry('Ï',"&Iuml;","&#x000CF;","&#207;"));
		entries.add(new Entry('Ð',"&ETH;","&#x000D0;","&#208;"));
		entries.add(new Entry('Ñ',"&Ntilde;","&#x000D1;","&#209;"));
		entries.add(new Entry('Ò',"&Ograve;","&#x000D2;","&#210;"));
		entries.add(new Entry('Ó',"&Oacute;","&#x000D3;","&#211;"));
		entries.add(new Entry('Ô',"&Ocirc;","&#x000D4;","&#212;"));
		entries.add(new Entry('Õ',"&Otilde;","&#x000D5;","&#213;"));
		entries.add(new Entry('Ö',"&Ouml;","&#x000D6;","&#214;"));
		entries.add(new Entry('×',"&times;","&#x000D7;","&#215;"));
		entries.add(new Entry('Ø',"&Oslash;","&#x000D8;","&#216;"));
		entries.add(new Entry('Ù',"&Ugrave;","&#x000D9;","&#217;"));
		entries.add(new Entry('Ú',"&Uacute;","&#x000DA;","&#218;"));
		entries.add(new Entry('Û',"&Ucirc;","&#x000DB;","&#219;"));
		entries.add(new Entry('Ü',"&Uuml;","&#x000DC;","&#220;"));
		entries.add(new Entry('Ý',"&Yacute;","&#x000DD;","&#221;"));
		entries.add(new Entry('Þ',"&THORN;","&#x000DE;","&#222;"));
		entries.add(new Entry('ß',"&szlig;","&#x000DF;","&#223;"));
		entries.add(new Entry('à',"&agrave;","&#x000E0;","&#224;"));
		entries.add(new Entry('á',"&aacute;","&#x000E1;","&#225;"));
		entries.add(new Entry('â',"&acirc;","&#x000E2;","&#226;"));
		entries.add(new Entry('ã',"&atilde;","&#x000E3;","&#227;"));
		entries.add(new Entry('ä',"&auml;","&#x000E4;","&#228;"));
		entries.add(new Entry('å',"&aring;","&#x000E5;","&#229;"));
		entries.add(new Entry('æ',"&aelig;","&#x000E6;","&#230;"));
		entries.add(new Entry('ç',"&ccedil;","&#x000E7;","&#231;"));
		entries.add(new Entry('è',"&egrave;","&#x000E8;","&#232;"));
		entries.add(new Entry('é',"&eacute;","&#x000E9;","&#233;"));
		entries.add(new Entry('ê',"&ecirc;","&#x000EA;","&#234;"));
		entries.add(new Entry('ë',"&euml;","&#x000EB;","&#235;"));
		entries.add(new Entry('ì',"&igrave;","&#x000EC;","&#236;"));
		entries.add(new Entry('í',"&iacute;","&#x000ED;","&#237;"));
		entries.add(new Entry('î',"&icirc;","&#x000EE;","&#238;"));
		entries.add(new Entry('ï',"&iuml;","&#x000EF;","&#239;"));
		entries.add(new Entry('ð',"&eth;","&#x000F0;","&#240;"));
		entries.add(new Entry('ñ',"&ntilde;","&#x000F1;","&#241;"));
		entries.add(new Entry('ò',"&ograve;","&#x000F2;","&#242;"));
		entries.add(new Entry('ó',"&oacute;","&#x000F3;","&#243;"));
		entries.add(new Entry('ô',"&ocirc;","&#x000F4;","&#244;"));
		entries.add(new Entry('õ',"&otilde;","&#x000F5;","&#245;"));
		entries.add(new Entry('ö',"&ouml;","&#x000F6;","&#246;"));
		entries.add(new Entry('÷',"&divide;","&div;","&#x000F7;","&#247;"));
		entries.add(new Entry('ø',"&oslash;","&#x000F8;","&#248;"));
		entries.add(new Entry('ù',"&ugrave;","&#x000F9;","&#249;"));
		entries.add(new Entry('ú',"&uacute;","&#x000FA;","&#250;"));
		entries.add(new Entry('û',"&ucirc;","&#x000FB;","&#251;"));
		entries.add(new Entry('ü',"&uuml;","&#x000FC;","&#252;"));
		entries.add(new Entry('ý',"&yacute;","&#x000FD;","&#253;"));
		entries.add(new Entry('þ',"&thorn;","&#x000FE;","&#254;"));
		entries.add(new Entry('ÿ',"&yuml;","&#x000FF;","&#255;"));
		entries.add(new Entry('Ā',"&Amacr;","&#x00100;","&#256;"));
		entries.add(new Entry('ā',"&amacr;","&#x00101;","&#257;"));
		entries.add(new Entry('Ă',"&Abreve;","&#x00102;","&#258;"));
		entries.add(new Entry('ă',"&abreve;","&#x00103;","&#259;"));
		entries.add(new Entry('Ą',"&Aogon;","&#x00104;","&#260;"));
		entries.add(new Entry('ą',"&aogon;","&#x00105;","&#261;"));
		entries.add(new Entry('Ć',"&Cacute;","&#x00106;","&#262;"));
		entries.add(new Entry('ć',"&cacute;","&#x00107;","&#263;"));
		entries.add(new Entry('Ĉ',"&Ccirc;","&#x00108;","&#264;"));
		entries.add(new Entry('ĉ',"&ccirc;","&#x00109;","&#265;"));
		entries.add(new Entry('Ċ',"&Cdot;","&#x0010A;","&#266;"));
		entries.add(new Entry('ċ',"&cdot;","&#x0010B;","&#267;"));
		entries.add(new Entry('Č',"&Ccaron;","&#x0010C;","&#268;"));
		entries.add(new Entry('č',"&ccaron;","&#x0010D;","&#269;"));
		entries.add(new Entry('Ď',"&Dcaron;","&#x0010E;","&#270;"));
		entries.add(new Entry('ď',"&dcaron;","&#x0010F;","&#271;"));
		entries.add(new Entry('Đ',"&Dstrok;","&#x00110;","&#272;"));
		entries.add(new Entry('đ',"&dstrok;","&#x00111;","&#273;"));
		entries.add(new Entry('Ē',"&Emacr;","&#x00112;","&#274;"));
		entries.add(new Entry('ē',"&emacr;","&#x00113;","&#275;"));
		entries.add(new Entry('Ė',"&Edot;","&#x00116;","&#278;"));
		entries.add(new Entry('ė',"&edot;","&#x00117;","&#279;"));
		entries.add(new Entry('Ę',"&Eogon;","&#x00118;","&#280;"));
		entries.add(new Entry('ę',"&eogon;","&#x00119;","&#281;"));
		entries.add(new Entry('Ě',"&Ecaron;","&#x0011A;","&#282;"));
		entries.add(new Entry('ě',"&ecaron;","&#x0011B;","&#283;"));
		entries.add(new Entry('Ĝ',"&Gcirc;","&#x0011C;","&#284;"));
		entries.add(new Entry('ĝ',"&gcirc;","&#x0011D;","&#285;"));
		entries.add(new Entry('Ğ',"&Gbreve;","&#x0011E;","&#286;"));
		entries.add(new Entry('ğ',"&gbreve;","&#x0011F;","&#287;"));
		entries.add(new Entry('Ġ',"&Gdot;","&#x00120;","&#288;"));
		entries.add(new Entry('ġ',"&gdot;","&#x00121;","&#289;"));
		entries.add(new Entry('Ģ',"&Gcedil;","&#x00122;","&#290;"));
		entries.add(new Entry('Ĥ',"&Hcirc;","&#x00124;","&#292;"));
		entries.add(new Entry('ĥ',"&hcirc;","&#x00125;","&#293;"));
		entries.add(new Entry('Ħ',"&Hstrok;","&#x00126;","&#294;"));
		entries.add(new Entry('ħ',"&hstrok;","&#x00127;","&#295;"));
		entries.add(new Entry('Ĩ',"&Itilde;","&#x00128;","&#296;"));
		entries.add(new Entry('ĩ',"&itilde;","&#x00129;","&#297;"));
		entries.add(new Entry('Ī',"&Imacr;","&#x0012A;","&#298;"));
		entries.add(new Entry('ī',"&imacr;","&#x0012B;","&#299;"));
		entries.add(new Entry('Į',"&Iogon;","&#x0012E;","&#302;"));
		entries.add(new Entry('į',"&iogon;","&#x0012F;","&#303;"));
		entries.add(new Entry('İ',"&Idot;","&#x00130;","&#304;"));
		entries.add(new Entry('ı',"&imath;","&inodot;","&#x00131;","&#305;"));
		entries.add(new Entry('Ĳ',"&IJlig;","&#x00132;","&#306;"));
		entries.add(new Entry('ĳ',"&ijlig;","&#x00133;","&#307;"));
		entries.add(new Entry('Ĵ',"&Jcirc;","&#x00134;","&#308;"));
		entries.add(new Entry('ĵ',"&jcirc;","&#x00135;","&#309;"));
		entries.add(new Entry('Ķ',"&Kcedil;","&#x00136;","&#310;"));
		entries.add(new Entry('ķ',"&kcedil;","&#x00137;","&#311;"));
		entries.add(new Entry('ĸ',"&kgreen;","&#x00138;","&#312;"));
		entries.add(new Entry('Ĺ',"&Lacute;","&#x00139;","&#313;"));
		entries.add(new Entry('ĺ',"&lacute;","&#x0013A;","&#314;"));
		entries.add(new Entry('Ļ',"&Lcedil;","&#x0013B;","&#315;"));
		entries.add(new Entry('ļ',"&lcedil;","&#x0013C;","&#316;"));
		entries.add(new Entry('Ľ',"&Lcaron;","&#x0013D;","&#317;"));
		entries.add(new Entry('ľ',"&lcaron;","&#x0013E;","&#318;"));
		entries.add(new Entry('Ŀ',"&Lmidot;","&#x0013F;","&#319;"));
		entries.add(new Entry('ŀ',"&lmidot;","&#x00140;","&#320;"));
		entries.add(new Entry('Ł',"&Lstrok;","&#x00141;","&#321;"));
		entries.add(new Entry('ł',"&lstrok;","&#x00142;","&#322;"));
		entries.add(new Entry('Ń',"&Nacute;","&#x00143;","&#323;"));
		entries.add(new Entry('ń',"&nacute;","&#x00144;","&#324;"));
		entries.add(new Entry('Ņ',"&Ncedil;","&#x00145;","&#325;"));
		entries.add(new Entry('ņ',"&ncedil;","&#x00146;","&#326;"));
		entries.add(new Entry('Ň',"&Ncaron;","&#x00147;","&#327;"));
		entries.add(new Entry('ň',"&ncaron;","&#x00148;","&#328;"));
		entries.add(new Entry('ŉ',"&napos;","&#x00149;","&#329;"));
		entries.add(new Entry('Ŋ',"&ENG;","&#x0014A;","&#330;"));
		entries.add(new Entry('ŋ',"&eng;","&#x0014B;","&#331;"));
		entries.add(new Entry('Ō',"&Omacr;","&#x0014C;","&#332;"));
		entries.add(new Entry('ō',"&omacr;","&#x0014D;","&#333;"));
		entries.add(new Entry('Ő',"&Odblac;","&#x00150;","&#336;"));
		entries.add(new Entry('ő',"&odblac;","&#x00151;","&#337;"));
		entries.add(new Entry('Œ',"&OElig;","&#x00152;","&#338;"));
		entries.add(new Entry('œ',"&oelig;","&#x00153;","&#339;"));
		entries.add(new Entry('Ŕ',"&Racute;","&#x00154;","&#340;"));
		entries.add(new Entry('ŕ',"&racute;","&#x00155;","&#341;"));
		entries.add(new Entry('Ŗ',"&Rcedil;","&#x00156;","&#342;"));
		entries.add(new Entry('ŗ',"&rcedil;","&#x00157;","&#343;"));
		entries.add(new Entry('Ř',"&Rcaron;","&#x00158;","&#344;"));
		entries.add(new Entry('ř',"&rcaron;","&#x00159;","&#345;"));
		entries.add(new Entry('Ś',"&Sacute;","&#x0015A;","&#346;"));
		entries.add(new Entry('ś',"&sacute;","&#x0015B;","&#347;"));
		entries.add(new Entry('Ŝ',"&Scirc;","&#x0015C;","&#348;"));
		entries.add(new Entry('ŝ',"&scirc;","&#x0015D;","&#349;"));
		entries.add(new Entry('Ş',"&Scedil;","&#x0015E;","&#350;"));
		entries.add(new Entry('ş',"&scedil;","&#x0015F;","&#351;"));
		entries.add(new Entry('Š',"&Scaron;","&#x00160;","&#352;"));
		entries.add(new Entry('š',"&scaron;","&#x00161;","&#353;"));
		entries.add(new Entry('Ţ',"&Tcedil;","&#x00162;","&#354;"));
		entries.add(new Entry('ţ',"&tcedil;","&#x00163;","&#355;"));
		entries.add(new Entry('Ť',"&Tcaron;","&#x00164;","&#356;"));
		entries.add(new Entry('ť',"&tcaron;","&#x00165;","&#357;"));
		entries.add(new Entry('Ŧ',"&Tstrok;","&#x00166;","&#358;"));
		entries.add(new Entry('ŧ',"&tstrok;","&#x00167;","&#359;"));
		entries.add(new Entry('Ũ',"&Utilde;","&#x00168;","&#360;"));
		entries.add(new Entry('ũ',"&utilde;","&#x00169;","&#361;"));
		entries.add(new Entry('Ū',"&Umacr;","&#x0016A;","&#362;"));
		entries.add(new Entry('ū',"&umacr;","&#x0016B;","&#363;"));
		entries.add(new Entry('Ŭ',"&Ubreve;","&#x0016C;","&#364;"));
		entries.add(new Entry('ŭ',"&ubreve;","&#x0016D;","&#365;"));
		entries.add(new Entry('Ů',"&Uring;","&#x0016E;","&#366;"));
		entries.add(new Entry('ů',"&uring;","&#x0016F;","&#367;"));
		entries.add(new Entry('Ű',"&Udblac;","&#x00170;","&#368;"));
		entries.add(new Entry('ű',"&udblac;","&#x00171;","&#369;"));
		entries.add(new Entry('Ų',"&Uogon;","&#x00172;","&#370;"));
		entries.add(new Entry('ų',"&uogon;","&#x00173;","&#371;"));
		entries.add(new Entry('Ŵ',"&Wcirc;","&#x00174;","&#372;"));
		entries.add(new Entry('ŵ',"&wcirc;","&#x00175;","&#373;"));
		entries.add(new Entry('Ŷ',"&Ycirc;","&#x00176;","&#374;"));
		entries.add(new Entry('ŷ',"&ycirc;","&#x00177;","&#375;"));
		entries.add(new Entry('Ÿ',"&Yuml;","&#x00178;","&#376;"));
		entries.add(new Entry('Ź',"&Zacute;","&#x00179;","&#377;"));
		entries.add(new Entry('ź',"&zacute;","&#x0017A;","&#378;"));
		entries.add(new Entry('Ż',"&Zdot;","&#x0017B;","&#379;"));
		entries.add(new Entry('ż',"&zdot;","&#x0017C;","&#380;"));
		entries.add(new Entry('Ž',"&Zcaron;","&#x0017D;","&#381;"));
		entries.add(new Entry('ž',"&zcaron;","&#x0017E;","&#382;"));
		entries.add(new Entry('ƒ',"&fnof;","&#x00192;","&#402;"));
		entries.add(new Entry('Ƶ',"&imped;","&#x001B5;","&#437;"));
		entries.add(new Entry('ǵ',"&gacute;","&#x001F5;","&#501;"));
		entries.add(new Entry('ȷ',"&jmath;","&#x00237;","&#567;"));
		entries.add(new Entry('ˆ',"&circ;","&#x002C6;","&#710;"));
		entries.add(new Entry('ˇ',"&caron;","&Hacek;","&#x002C7;","&#711;"));
		entries.add(new Entry('˘',"&breve;","&Breve;","&#x002D8;","&#728;"));
		entries.add(new Entry('˙',"&dot;","&DiacriticalDot;","&#x002D9;","&#729;"));
		entries.add(new Entry('˚',"&ring;","&#x002DA;","&#730;"));
		entries.add(new Entry('˛',"&ogon;","&#x002DB;","&#731;"));
		entries.add(new Entry('˜',"&tilde;","&DiacriticalTilde;","&#x002DC;","&#732;"));
		entries.add(new Entry('˝',"&dblac;","&DiacriticalDoubleAcute;","&#x002DD;","&#733;"));
		entries.add(new Entry('̑',"&DownBreve;","&#x00311;","&#785;"));
		entries.add(new Entry('̲',"&UnderBar;","&#x00332;","&#818;"));
		entries.add(new Entry('Α',"&Alpha;","&#x00391;","&#913;"));
		entries.add(new Entry('Β',"&Beta;","&#x00392;","&#914;"));
		entries.add(new Entry('Γ',"&Gamma;","&#x00393;","&#915;"));
		entries.add(new Entry('Δ',"&Delta;","&#x00394;","&#916;"));
		entries.add(new Entry('Ε',"&Epsilon;","&#x00395;","&#917;"));
		entries.add(new Entry('Ζ',"&Zeta;","&#x00396;","&#918;"));
		entries.add(new Entry('Η',"&Eta;","&#x00397;","&#919;"));
		entries.add(new Entry('Θ',"&Theta;","&#x00398;","&#920;"));
		entries.add(new Entry('Ι',"&Iota;","&#x00399;","&#921;"));
		entries.add(new Entry('Κ',"&Kappa;","&#x0039A;","&#922;"));
		entries.add(new Entry('Λ',"&Lambda;","&#x0039B;","&#923;"));
		entries.add(new Entry('Μ',"&Mu;","&#x0039C;","&#924;"));
		entries.add(new Entry('Ν',"&Nu;","&#x0039D;","&#925;"));
		entries.add(new Entry('Ξ',"&Xi;","&#x0039E;","&#926;"));
		entries.add(new Entry('Ο',"&Omicron;","&#x0039F;","&#927;"));
		entries.add(new Entry('Π',"&Pi;","&#x003A0;","&#928;"));
		entries.add(new Entry('Ρ',"&Rho;","&#x003A1;","&#929;"));
		entries.add(new Entry('Σ',"&Sigma;","&#x003A3;","&#931;"));
		entries.add(new Entry('Τ',"&Tau;","&#x003A4;","&#932;"));
		entries.add(new Entry('Υ',"&Upsilon;","&#x003A5;","&#933;"));
		entries.add(new Entry('Φ',"&Phi;","&#x003A6;","&#934;"));
		entries.add(new Entry('Χ',"&Chi;","&#x003A7;","&#935;"));
		entries.add(new Entry('Ψ',"&Psi;","&#x003A8;","&#936;"));
		entries.add(new Entry('Ω',"&Omega;","&#x003A9;","&#937;"));
		entries.add(new Entry('α',"&alpha;","&#x003B1;","&#945;"));
		entries.add(new Entry('β',"&beta;","&#x003B2;","&#946;"));
		entries.add(new Entry('γ',"&gamma;","&#x003B3;","&#947;"));
		entries.add(new Entry('δ',"&delta;","&#x003B4;","&#948;"));
		entries.add(new Entry('ε',"&epsiv;","&varepsilon;","&epsilon;","&#x003B5;","&#949;"));
		entries.add(new Entry('ζ',"&zeta;","&#x003B6;","&#950;"));
		entries.add(new Entry('η',"&eta;","&#x003B7;","&#951;"));
		entries.add(new Entry('θ',"&theta;","&#x003B8;","&#952;"));
		entries.add(new Entry('ι',"&iota;","&#x003B9;","&#953;"));
		entries.add(new Entry('κ',"&kappa;","&#x003BA;","&#954;"));
		entries.add(new Entry('λ',"&lambda;","&#x003BB;","&#955;"));
		entries.add(new Entry('μ',"&mu;","&#x003BC;","&#956;"));
		entries.add(new Entry('ν',"&nu;","&#x003BD;","&#957;"));
		entries.add(new Entry('ξ',"&xi;","&#x003BE;","&#958;"));
		entries.add(new Entry('ο',"&omicron;","&#x003BF;","&#959;"));
		entries.add(new Entry('π',"&pi;","&#x003C0;","&#960;"));
		entries.add(new Entry('ρ',"&rho;","&#x003C1;","&#961;"));
		entries.add(new Entry('ς',"&sigmav;","&varsigma;","&sigmaf;","&#x003C2;","&#962;"));
		entries.add(new Entry('σ',"&sigma;","&#x003C3;","&#963;"));
		entries.add(new Entry('τ',"&tau;","&#x003C4;","&#964;"));
		entries.add(new Entry('υ',"&upsi;","&upsilon;","&#x003C5;","&#965;"));
		entries.add(new Entry('φ',"&phi;","&phiv;","&varphi;","&#x003C6;","&#966;"));
		entries.add(new Entry('χ',"&chi;","&#x003C7;","&#967;"));
		entries.add(new Entry('ψ',"&psi;","&#x003C8;","&#968;"));
		entries.add(new Entry('ω',"&omega;","&#x003C9;","&#969;"));
		entries.add(new Entry('ϑ',"&thetav;","&vartheta;","&thetasym;","&#x003D1;","&#977;"));
		entries.add(new Entry('ϒ',"&Upsi;","&upsih;","&#x003D2;","&#978;"));
		entries.add(new Entry('ϕ',"&straightphi;","&#x003D5;","&#981;"));
		entries.add(new Entry('ϖ',"&piv;","&varpi;","&#x003D6;","&#982;"));
		entries.add(new Entry('Ϝ',"&Gammad;","&#x003DC;","&#988;"));
		entries.add(new Entry('ϝ',"&gammad;","&digamma;","&#x003DD;","&#989;"));
		entries.add(new Entry('ϰ',"&kappav;","&varkappa;","&#x003F0;","&#1008;"));
		entries.add(new Entry('ϱ',"&rhov;","&varrho;","&#x003F1;","&#1009;"));
		entries.add(new Entry('ϵ',"&epsi;","&straightepsilon;","&#x003F5;","&#1013;"));
		entries.add(new Entry('϶',"&bepsi;","&backepsilon;","&#x003F6;","&#1014;"));
		entries.add(new Entry('Ё',"&IOcy;","&#x00401;","&#1025;"));
		entries.add(new Entry('Ђ',"&DJcy;","&#x00402;","&#1026;"));
		entries.add(new Entry('Ѓ',"&GJcy;","&#x00403;","&#1027;"));
		entries.add(new Entry('Є',"&Jukcy;","&#x00404;","&#1028;"));
		entries.add(new Entry('Ѕ',"&DScy;","&#x00405;","&#1029;"));
		entries.add(new Entry('І',"&Iukcy;","&#x00406;","&#1030;"));
		entries.add(new Entry('Ї',"&YIcy;","&#x00407;","&#1031;"));
		entries.add(new Entry('Ј',"&Jsercy;","&#x00408;","&#1032;"));
		entries.add(new Entry('Љ',"&LJcy;","&#x00409;","&#1033;"));
		entries.add(new Entry('Њ',"&NJcy;","&#x0040A;","&#1034;"));
		entries.add(new Entry('Ћ',"&TSHcy;","&#x0040B;","&#1035;"));
		entries.add(new Entry('Ќ',"&KJcy;","&#x0040C;","&#1036;"));
		entries.add(new Entry('Ў',"&Ubrcy;","&#x0040E;","&#1038;"));
		entries.add(new Entry('Џ',"&DZcy;","&#x0040F;","&#1039;"));
		entries.add(new Entry('А',"&Acy;","&#x00410;","&#1040;"));
		entries.add(new Entry('Б',"&Bcy;","&#x00411;","&#1041;"));
		entries.add(new Entry('В',"&Vcy;","&#x00412;","&#1042;"));
		entries.add(new Entry('Г',"&Gcy;","&#x00413;","&#1043;"));
		entries.add(new Entry('Д',"&Dcy;","&#x00414;","&#1044;"));
		entries.add(new Entry('Е',"&IEcy;","&#x00415;","&#1045;"));
		entries.add(new Entry('Ж',"&ZHcy;","&#x00416;","&#1046;"));
		entries.add(new Entry('З',"&Zcy;","&#x00417;","&#1047;"));
		entries.add(new Entry('И',"&Icy;","&#x00418;","&#1048;"));
		entries.add(new Entry('Й',"&Jcy;","&#x00419;","&#1049;"));
		entries.add(new Entry('К',"&Kcy;","&#x0041A;","&#1050;"));
		entries.add(new Entry('Л',"&Lcy;","&#x0041B;","&#1051;"));
		entries.add(new Entry('М',"&Mcy;","&#x0041C;","&#1052;"));
		entries.add(new Entry('Н',"&Ncy;","&#x0041D;","&#1053;"));
		entries.add(new Entry('О',"&Ocy;","&#x0041E;","&#1054;"));
		entries.add(new Entry('П',"&Pcy;","&#x0041F;","&#1055;"));
		entries.add(new Entry('Р',"&Rcy;","&#x00420;","&#1056;"));
		entries.add(new Entry('С',"&Scy;","&#x00421;","&#1057;"));
		entries.add(new Entry('Т',"&Tcy;","&#x00422;","&#1058;"));
		entries.add(new Entry('У',"&Ucy;","&#x00423;","&#1059;"));
		entries.add(new Entry('Ф',"&Fcy;","&#x00424;","&#1060;"));
		entries.add(new Entry('Х',"&KHcy;","&#x00425;","&#1061;"));
		entries.add(new Entry('Ц',"&TScy;","&#x00426;","&#1062;"));
		entries.add(new Entry('Ч',"&CHcy;","&#x00427;","&#1063;"));
		entries.add(new Entry('Ш',"&SHcy;","&#x00428;","&#1064;"));
		entries.add(new Entry('Щ',"&SHCHcy;","&#x00429;","&#1065;"));
		entries.add(new Entry('Ъ',"&HARDcy;","&#x0042A;","&#1066;"));
		entries.add(new Entry('Ы',"&Ycy;","&#x0042B;","&#1067;"));
		entries.add(new Entry('Ь',"&SOFTcy;","&#x0042C;","&#1068;"));
		entries.add(new Entry('Э',"&Ecy;","&#x0042D;","&#1069;"));
		entries.add(new Entry('Ю',"&YUcy;","&#x0042E;","&#1070;"));
		entries.add(new Entry('Я',"&YAcy;","&#x0042F;","&#1071;"));
		entries.add(new Entry('а',"&acy;","&#x00430;","&#1072;"));
		entries.add(new Entry('б',"&bcy;","&#x00431;","&#1073;"));
		entries.add(new Entry('в',"&vcy;","&#x00432;","&#1074;"));
		entries.add(new Entry('г',"&gcy;","&#x00433;","&#1075;"));
		entries.add(new Entry('д',"&dcy;","&#x00434;","&#1076;"));
		entries.add(new Entry('е',"&iecy;","&#x00435;","&#1077;"));
		entries.add(new Entry('ж',"&zhcy;","&#x00436;","&#1078;"));
		entries.add(new Entry('з',"&zcy;","&#x00437;","&#1079;"));
		entries.add(new Entry('и',"&icy;","&#x00438;","&#1080;"));
		entries.add(new Entry('й',"&jcy;","&#x00439;","&#1081;"));
		entries.add(new Entry('к',"&kcy;","&#x0043A;","&#1082;"));
		entries.add(new Entry('л',"&lcy;","&#x0043B;","&#1083;"));
		entries.add(new Entry('м',"&mcy;","&#x0043C;","&#1084;"));
		entries.add(new Entry('н',"&ncy;","&#x0043D;","&#1085;"));
		entries.add(new Entry('о',"&ocy;","&#x0043E;","&#1086;"));
		entries.add(new Entry('п',"&pcy;","&#x0043F;","&#1087;"));
		entries.add(new Entry('р',"&rcy;","&#x00440;","&#1088;"));
		entries.add(new Entry('с',"&scy;","&#x00441;","&#1089;"));
		entries.add(new Entry('т',"&tcy;","&#x00442;","&#1090;"));
		entries.add(new Entry('у',"&ucy;","&#x00443;","&#1091;"));
		entries.add(new Entry('ф',"&fcy;","&#x00444;","&#1092;"));
		entries.add(new Entry('х',"&khcy;","&#x00445;","&#1093;"));
		entries.add(new Entry('ц',"&tscy;","&#x00446;","&#1094;"));
		entries.add(new Entry('ч',"&chcy;","&#x00447;","&#1095;"));
		entries.add(new Entry('ш',"&shcy;","&#x00448;","&#1096;"));
		entries.add(new Entry('щ',"&shchcy;","&#x00449;","&#1097;"));
		entries.add(new Entry('ъ',"&hardcy;","&#x0044A;","&#1098;"));
		entries.add(new Entry('ы',"&ycy;","&#x0044B;","&#1099;"));
		entries.add(new Entry('ь',"&softcy;","&#x0044C;","&#1100;"));
		entries.add(new Entry('э',"&ecy;","&#x0044D;","&#1101;"));
		entries.add(new Entry('ю',"&yucy;","&#x0044E;","&#1102;"));
		entries.add(new Entry('я',"&yacy;","&#x0044F;","&#1103;"));
		entries.add(new Entry('ё',"&iocy;","&#x00451;","&#1105;"));
		entries.add(new Entry('ђ',"&djcy;","&#x00452;","&#1106;"));
		entries.add(new Entry('ѓ',"&gjcy;","&#x00453;","&#1107;"));
		entries.add(new Entry('є',"&jukcy;","&#x00454;","&#1108;"));
		entries.add(new Entry('ѕ',"&dscy;","&#x00455;","&#1109;"));
		entries.add(new Entry('і',"&iukcy;","&#x00456;","&#1110;"));
		entries.add(new Entry('ї',"&yicy;","&#x00457;","&#1111;"));
		entries.add(new Entry('ј',"&jsercy;","&#x00458;","&#1112;"));
		entries.add(new Entry('љ',"&ljcy;","&#x00459;","&#1113;"));
		entries.add(new Entry('њ',"&njcy;","&#x0045A;","&#1114;"));
		entries.add(new Entry('ћ',"&tshcy;","&#x0045B;","&#1115;"));
		entries.add(new Entry('ќ',"&kjcy;","&#x0045C;","&#1116;"));
		entries.add(new Entry('ў',"&ubrcy;","&#x0045E;","&#1118;"));
		entries.add(new Entry('џ',"&dzcy;","&#x0045F;","&#1119;"));
		entries.add(new Entry(' ',"&ensp;","&#x02002;","&#8194;"));
		entries.add(new Entry(' ',"&emsp;","&#x02003;","&#8195;"));
		entries.add(new Entry(' ',"&emsp13;","&#x02004;","&#8196;"));
		entries.add(new Entry(' ',"&emsp14;","&#x02005;","&#8197;"));
		entries.add(new Entry(' ',"&numsp;","&#x02007;","&#8199;"));
		entries.add(new Entry(' ',"&puncsp;","&#x02008;","&#8200;"));
		entries.add(new Entry(' ',"&thinsp;","&ThinSpace;","&#x02009;","&#8201;"));
		entries.add(new Entry(' ',"&hairsp;","&VeryThinSpace;","&#x0200A;","&#8202;"));
		entries.add(new Entry('​',"&ZeroWidthSpace;","&NegativeVeryThinSpace;","&NegativeThinSpace;","&NegativeMediumSpace;","&NegativeThickSpace;","&#x0200B;","&#8203;"));
		entries.add(new Entry('‌',"&zwnj;","&#x0200C;","&#8204;"));
		entries.add(new Entry('‍',"&zwj;","&#x0200D;","&#8205;"));
		entries.add(new Entry('‎',"&lrm;","&#x0200E;","&#8206;"));
		entries.add(new Entry('‏',"&rlm;","&#x0200F;","&#8207;"));
		entries.add(new Entry('‐',"&hyphen;","&dash;","&#x02010;","&#8208;"));
		entries.add(new Entry('–',"&ndash;","&#x02013;","&#8211;"));
		entries.add(new Entry('—',"&mdash;","&#x02014;","&#8212;"));
		entries.add(new Entry('―',"&horbar;","&#x02015;","&#8213;"));
		entries.add(new Entry('‖',"&Verbar;","&Vert;","&#x02016;","&#8214;"));
		entries.add(new Entry('‘',"&lsquo;","&OpenCurlyQuote;","&#x02018;","&#8216;"));
		entries.add(new Entry('’',"&rsquo;","&rsquor;","&CloseCurlyQuote;","&#x02019;","&#8217;"));
		entries.add(new Entry('‚',"&lsquor;","&sbquo;","&#x0201A;","&#8218;"));
		entries.add(new Entry('“',"&ldquo;","&OpenCurlyDoubleQuote;","&#x0201C;","&#8220;"));
		entries.add(new Entry('”',"&rdquo;","&rdquor;","&CloseCurlyDoubleQuote;","&#x0201D;","&#8221;"));
		entries.add(new Entry('„',"&ldquor;","&bdquo;","&#x0201E;","&#8222;"));
		entries.add(new Entry('†',"&dagger;","&#x02020;","&#8224;"));
		entries.add(new Entry('‡',"&Dagger;","&ddagger;","&#x02021;","&#8225;"));
		entries.add(new Entry('•',"&bull;","&bullet;","&#x02022;","&#8226;"));
		entries.add(new Entry('‥',"&nldr;","&#x02025;","&#8229;"));
		entries.add(new Entry('…',"&hellip;","&mldr;","&#x02026;","&#8230;"));
		entries.add(new Entry('‰',"&permil;","&#x02030;","&#8240;"));
		entries.add(new Entry('‱',"&pertenk;","&#x02031;","&#8241;"));
		entries.add(new Entry('′',"&prime;","&#x02032;","&#8242;"));
		entries.add(new Entry('″',"&Prime;","&#x02033;","&#8243;"));
		entries.add(new Entry('‴',"&tprime;","&#x02034;","&#8244;"));
		entries.add(new Entry('‵',"&bprime;","&backprime;","&#x02035;","&#8245;"));
		entries.add(new Entry('‹',"&lsaquo;","&#x02039;","&#8249;"));
		entries.add(new Entry('›',"&rsaquo;","&#x0203A;","&#8250;"));
		entries.add(new Entry('‾',"&oline;","&#x0203E;","&#8254;"));
		entries.add(new Entry('⁁',"&caret;","&#x02041;","&#8257;"));
		entries.add(new Entry('⁃',"&hybull;","&#x02043;","&#8259;"));
		entries.add(new Entry('⁄',"&frasl;","&#x02044;","&#8260;"));
		entries.add(new Entry('⁏',"&bsemi;","&#x0204F;","&#8271;"));
		entries.add(new Entry('⁗',"&qprime;","&#x02057;","&#8279;"));
		entries.add(new Entry(' ',"&MediumSpace;","&#x0205F;","&#8287;"));
		entries.add(new Entry('⁠',"&NoBreak;","&#x02060;","&#8288;"));
		entries.add(new Entry('⁡',"&ApplyFunction;","&af;","&#x02061;","&#8289;"));
		entries.add(new Entry('⁢',"&InvisibleTimes;","&it;","&#x02062;","&#8290;"));
		entries.add(new Entry('⁣',"&InvisibleComma;","&ic;","&#x02063;","&#8291;"));
		entries.add(new Entry('€',"&euro;","&#x020AC;","&#8364;"));
		entries.add(new Entry('⃛',"&tdot;","&TripleDot;","&#x020DB;","&#8411;"));
		entries.add(new Entry('⃜',"&DotDot;","&#x020DC;","&#8412;"));
		entries.add(new Entry('ℂ',"&Copf;","&complexes;","&#x02102;","&#8450;"));
		entries.add(new Entry('℅',"&incare;","&#x02105;","&#8453;"));
		entries.add(new Entry('ℊ',"&gscr;","&#x0210A;","&#8458;"));
		entries.add(new Entry('ℋ',"&hamilt;","&HilbertSpace;","&Hscr;","&#x0210B;","&#8459;"));
		entries.add(new Entry('ℌ',"&Hfr;","&Poincareplane;","&#x0210C;","&#8460;"));
		entries.add(new Entry('ℍ',"&quaternions;","&Hopf;","&#x0210D;","&#8461;"));
		entries.add(new Entry('ℎ',"&planckh;","&#x0210E;","&#8462;"));
		entries.add(new Entry('ℏ',"&planck;","&hbar;","&plankv;","&hslash;","&#x0210F;","&#8463;"));
		entries.add(new Entry('ℐ',"&Iscr;","&imagline;","&#x02110;","&#8464;"));
		entries.add(new Entry('ℑ',"&image;","&Im;","&imagpart;","&Ifr;","&#x02111;","&#8465;"));
		entries.add(new Entry('ℒ',"&Lscr;","&lagran;","&Laplacetrf;","&#x02112;","&#8466;"));
		entries.add(new Entry('ℓ',"&ell;","&#x02113;","&#8467;"));
		entries.add(new Entry('ℕ',"&Nopf;","&naturals;","&#x02115;","&#8469;"));
		entries.add(new Entry('№',"&numero;","&#x02116;","&#8470;"));
		entries.add(new Entry('℗',"&copysr;","&#x02117;","&#8471;"));
		entries.add(new Entry('℘',"&weierp;","&wp;","&#x02118;","&#8472;"));
		entries.add(new Entry('ℙ',"&Popf;","&primes;","&#x02119;","&#8473;"));
		entries.add(new Entry('ℚ',"&rationals;","&Qopf;","&#x0211A;","&#8474;"));
		entries.add(new Entry('ℛ',"&Rscr;","&realine;","&#x0211B;","&#8475;"));
		entries.add(new Entry('ℜ',"&real;","&Re;","&realpart;","&Rfr;","&#x0211C;","&#8476;"));
		entries.add(new Entry('ℝ',"&reals;","&Ropf;","&#x0211D;","&#8477;"));
		entries.add(new Entry('℞',"&rx;","&#x0211E;","&#8478;"));
		entries.add(new Entry('™',"&trade;","&TRADE;","&#x02122;","&#8482;"));
		entries.add(new Entry('ℤ',"&integers;","&Zopf;","&#x02124;","&#8484;"));
		entries.add(new Entry('Ω',"&ohm;","&#x02126;","&#8486;"));
		entries.add(new Entry('℧',"&mho;","&#x02127;","&#8487;"));
		entries.add(new Entry('ℨ',"&Zfr;","&zeetrf;","&#x02128;","&#8488;"));
		entries.add(new Entry('℩',"&iiota;","&#x02129;","&#8489;"));
		entries.add(new Entry('Å',"&angst;","&#x0212B;","&#8491;"));
		entries.add(new Entry('ℬ',"&bernou;","&Bernoullis;","&Bscr;","&#x0212C;","&#8492;"));
		entries.add(new Entry('ℭ',"&Cfr;","&Cayleys;","&#x0212D;","&#8493;"));
		entries.add(new Entry('ℯ',"&escr;","&#x0212F;","&#8495;"));
		entries.add(new Entry('ℰ',"&Escr;","&expectation;","&#x02130;","&#8496;"));
		entries.add(new Entry('ℱ',"&Fscr;","&Fouriertrf;","&#x02131;","&#8497;"));
		entries.add(new Entry('ℳ',"&phmmat;","&Mellintrf;","&Mscr;","&#x02133;","&#8499;"));
		entries.add(new Entry('ℴ',"&order;","&orderof;","&oscr;","&#x02134;","&#8500;"));
		entries.add(new Entry('ℵ',"&alefsym;","&aleph;","&#x02135;","&#8501;"));
		entries.add(new Entry('ℶ',"&beth;","&#x02136;","&#8502;"));
		entries.add(new Entry('ℷ',"&gimel;","&#x02137;","&#8503;"));
		entries.add(new Entry('ℸ',"&daleth;","&#x02138;","&#8504;"));
		entries.add(new Entry('ⅅ',"&CapitalDifferentialD;","&DD;","&#x02145;","&#8517;"));
		entries.add(new Entry('ⅆ',"&DifferentialD;","&dd;","&#x02146;","&#8518;"));
		entries.add(new Entry('ⅇ',"&ExponentialE;","&exponentiale;","&ee;","&#x02147;","&#8519;"));
		entries.add(new Entry('ⅈ',"&ImaginaryI;","&ii;","&#x02148;","&#8520;"));
		entries.add(new Entry('⅓',"&frac13;","&#x02153;","&#8531;"));
		entries.add(new Entry('⅔',"&frac23;","&#x02154;","&#8532;"));
		entries.add(new Entry('⅕',"&frac15;","&#x02155;","&#8533;"));
		entries.add(new Entry('⅖',"&frac25;","&#x02156;","&#8534;"));
		entries.add(new Entry('⅗',"&frac35;","&#x02157;","&#8535;"));
		entries.add(new Entry('⅘',"&frac45;","&#x02158;","&#8536;"));
		entries.add(new Entry('⅙',"&frac16;","&#x02159;","&#8537;"));
		entries.add(new Entry('⅚',"&frac56;","&#x0215A;","&#8538;"));
		entries.add(new Entry('⅛',"&frac18;","&#x0215B;","&#8539;"));
		entries.add(new Entry('⅜',"&frac38;","&#x0215C;","&#8540;"));
		entries.add(new Entry('⅝',"&frac58;","&#x0215D;","&#8541;"));
		entries.add(new Entry('⅞',"&frac78;","&#x0215E;","&#8542;"));
		entries.add(new Entry('←',"&larr;","&leftarrow;","&LeftArrow;","&slarr;","&ShortLeftArrow;","&#x02190;","&#8592;"));
		entries.add(new Entry('↑',"&uarr;","&uparrow;","&UpArrow;","&ShortUpArrow;","&#x02191;","&#8593;"));
		entries.add(new Entry('→',"&rarr;","&rightarrow;","&RightArrow;","&srarr;","&ShortRightArrow;","&#x02192;","&#8594;"));
		entries.add(new Entry('↓',"&darr;","&downarrow;","&DownArrow;","&ShortDownArrow;","&#x02193;","&#8595;"));
		entries.add(new Entry('↔',"&harr;","&leftrightarrow;","&LeftRightArrow;","&#x02194;","&#8596;"));
		entries.add(new Entry('↕',"&varr;","&updownarrow;","&UpDownArrow;","&#x02195;","&#8597;"));
		entries.add(new Entry('↖',"&nwarr;","&UpperLeftArrow;","&nwarrow;","&#x02196;","&#8598;"));
		entries.add(new Entry('↗',"&nearr;","&UpperRightArrow;","&nearrow;","&#x02197;","&#8599;"));
		entries.add(new Entry('↘',"&searr;","&searrow;","&LowerRightArrow;","&#x02198;","&#8600;"));
		entries.add(new Entry('↙',"&swarr;","&swarrow;","&LowerLeftArrow;","&#x02199;","&#8601;"));
		entries.add(new Entry('↚',"&nlarr;","&nleftarrow;","&#x0219A;","&#8602;"));
		entries.add(new Entry('↛',"&nrarr;","&nrightarrow;","&#x0219B;","&#8603;"));
		entries.add(new Entry('↝',"&rarrw;","&rightsquigarrow;","&#x0219D;","&#8605;"));
		entries.add(new Entry('↞',"&Larr;","&twoheadleftarrow;","&#x0219E;","&#8606;"));
		entries.add(new Entry('↟',"&Uarr;","&#x0219F;","&#8607;"));
		entries.add(new Entry('↠',"&Rarr;","&twoheadrightarrow;","&#x021A0;","&#8608;"));
		entries.add(new Entry('↡',"&Darr;","&#x021A1;","&#8609;"));
		entries.add(new Entry('↢',"&larrtl;","&leftarrowtail;","&#x021A2;","&#8610;"));
		entries.add(new Entry('↣',"&rarrtl;","&rightarrowtail;","&#x021A3;","&#8611;"));
		entries.add(new Entry('↤',"&LeftTeeArrow;","&mapstoleft;","&#x021A4;","&#8612;"));
		entries.add(new Entry('↥',"&UpTeeArrow;","&mapstoup;","&#x021A5;","&#8613;"));
		entries.add(new Entry('↦',"&map;","&RightTeeArrow;","&mapsto;","&#x021A6;","&#8614;"));
		entries.add(new Entry('↧',"&DownTeeArrow;","&mapstodown;","&#x021A7;","&#8615;"));
		entries.add(new Entry('↩',"&larrhk;","&hookleftarrow;","&#x021A9;","&#8617;"));
		entries.add(new Entry('↪',"&rarrhk;","&hookrightarrow;","&#x021AA;","&#8618;"));
		entries.add(new Entry('↫',"&larrlp;","&looparrowleft;","&#x021AB;","&#8619;"));
		entries.add(new Entry('↬',"&rarrlp;","&looparrowright;","&#x021AC;","&#8620;"));
		entries.add(new Entry('↭',"&harrw;","&leftrightsquigarrow;","&#x021AD;","&#8621;"));
		entries.add(new Entry('↮',"&nharr;","&nleftrightarrow;","&#x021AE;","&#8622;"));
		entries.add(new Entry('↰',"&lsh;","&Lsh;","&#x021B0;","&#8624;"));
		entries.add(new Entry('↱',"&rsh;","&Rsh;","&#x021B1;","&#8625;"));
		entries.add(new Entry('↲',"&ldsh;","&#x021B2;","&#8626;"));
		entries.add(new Entry('↳',"&rdsh;","&#x021B3;","&#8627;"));
		entries.add(new Entry('↵',"&crarr;","&#x021B5;","&#8629;"));
		entries.add(new Entry('↶',"&cularr;","&curvearrowleft;","&#x021B6;","&#8630;"));
		entries.add(new Entry('↷',"&curarr;","&curvearrowright;","&#x021B7;","&#8631;"));
		entries.add(new Entry('↺',"&olarr;","&circlearrowleft;","&#x021BA;","&#8634;"));
		entries.add(new Entry('↻',"&orarr;","&circlearrowright;","&#x021BB;","&#8635;"));
		entries.add(new Entry('↼',"&lharu;","&LeftVector;","&leftharpoonup;","&#x021BC;","&#8636;"));
		entries.add(new Entry('↽',"&lhard;","&leftharpoondown;","&DownLeftVector;","&#x021BD;","&#8637;"));
		entries.add(new Entry('↾',"&uharr;","&upharpoonright;","&RightUpVector;","&#x021BE;","&#8638;"));
		entries.add(new Entry('↿',"&uharl;","&upharpoonleft;","&LeftUpVector;","&#x021BF;","&#8639;"));
		entries.add(new Entry('⇀',"&rharu;","&RightVector;","&rightharpoonup;","&#x021C0;","&#8640;"));
		entries.add(new Entry('⇁',"&rhard;","&rightharpoondown;","&DownRightVector;","&#x021C1;","&#8641;"));
		entries.add(new Entry('⇂',"&dharr;","&RightDownVector;","&downharpoonright;","&#x021C2;","&#8642;"));
		entries.add(new Entry('⇃',"&dharl;","&LeftDownVector;","&downharpoonleft;","&#x021C3;","&#8643;"));
		entries.add(new Entry('⇄',"&rlarr;","&rightleftarrows;","&RightArrowLeftArrow;","&#x021C4;","&#8644;"));
		entries.add(new Entry('⇅',"&udarr;","&UpArrowDownArrow;","&#x021C5;","&#8645;"));
		entries.add(new Entry('⇆',"&lrarr;","&leftrightarrows;","&LeftArrowRightArrow;","&#x021C6;","&#8646;"));
		entries.add(new Entry('⇇',"&llarr;","&leftleftarrows;","&#x021C7;","&#8647;"));
		entries.add(new Entry('⇈',"&uuarr;","&upuparrows;","&#x021C8;","&#8648;"));
		entries.add(new Entry('⇉',"&rrarr;","&rightrightarrows;","&#x021C9;","&#8649;"));
		entries.add(new Entry('⇊',"&ddarr;","&downdownarrows;","&#x021CA;","&#8650;"));
		entries.add(new Entry('⇋',"&lrhar;","&ReverseEquilibrium;","&leftrightharpoons;","&#x021CB;","&#8651;"));
		entries.add(new Entry('⇌',"&rlhar;","&rightleftharpoons;","&Equilibrium;","&#x021CC;","&#8652;"));
		entries.add(new Entry('⇍',"&nlArr;","&nLeftarrow;","&#x021CD;","&#8653;"));
		entries.add(new Entry('⇎',"&nhArr;","&nLeftrightarrow;","&#x021CE;","&#8654;"));
		entries.add(new Entry('⇏',"&nrArr;","&nRightarrow;","&#x021CF;","&#8655;"));
		entries.add(new Entry('⇐',"&lArr;","&Leftarrow;","&DoubleLeftArrow;","&#x021D0;","&#8656;"));
		entries.add(new Entry('⇑',"&uArr;","&Uparrow;","&DoubleUpArrow;","&#x021D1;","&#8657;"));
		entries.add(new Entry('⇒',"&rArr;","&Rightarrow;","&Implies;","&DoubleRightArrow;","&#x021D2;","&#8658;"));
		entries.add(new Entry('⇓',"&dArr;","&Downarrow;","&DoubleDownArrow;","&#x021D3;","&#8659;"));
		entries.add(new Entry('⇔',"&hArr;","&Leftrightarrow;","&DoubleLeftRightArrow;","&iff;","&#x021D4;","&#8660;"));
		entries.add(new Entry('⇕',"&vArr;","&Updownarrow;","&DoubleUpDownArrow;","&#x021D5;","&#8661;"));
		entries.add(new Entry('⇖',"&nwArr;","&#x021D6;","&#8662;"));
		entries.add(new Entry('⇗',"&neArr;","&#x021D7;","&#8663;"));
		entries.add(new Entry('⇘',"&seArr;","&#x021D8;","&#8664;"));
		entries.add(new Entry('⇙',"&swArr;","&#x021D9;","&#8665;"));
		entries.add(new Entry('⇚',"&lAarr;","&Lleftarrow;","&#x021DA;","&#8666;"));
		entries.add(new Entry('⇛',"&rAarr;","&Rrightarrow;","&#x021DB;","&#8667;"));
		entries.add(new Entry('⇝',"&zigrarr;","&#x021DD;","&#8669;"));
		entries.add(new Entry('⇤',"&larrb;","&LeftArrowBar;","&#x021E4;","&#8676;"));
		entries.add(new Entry('⇥',"&rarrb;","&RightArrowBar;","&#x021E5;","&#8677;"));
		entries.add(new Entry('⇵',"&duarr;","&DownArrowUpArrow;","&#x021F5;","&#8693;"));
		entries.add(new Entry('⇽',"&loarr;","&#x021FD;","&#8701;"));
		entries.add(new Entry('⇾',"&roarr;","&#x021FE;","&#8702;"));
		entries.add(new Entry('⇿',"&hoarr;","&#x021FF;","&#8703;"));
		entries.add(new Entry('∀',"&forall;","&ForAll;","&#x02200;","&#8704;"));
		entries.add(new Entry('∁',"&comp;","&complement;","&#x02201;","&#8705;"));
		entries.add(new Entry('∂',"&part;","&PartialD;","&#x02202;","&#8706;"));
		entries.add(new Entry('∃',"&exist;","&Exists;","&#x02203;","&#8707;"));
		entries.add(new Entry('∄',"&nexist;","&NotExists;","&nexists;","&#x02204;","&#8708;"));
		entries.add(new Entry('∅',"&empty;","&emptyset;","&emptyv;","&varnothing;","&#x02205;","&#8709;"));
		entries.add(new Entry('∇',"&nabla;","&Del;","&#x02207;","&#8711;"));
		entries.add(new Entry('∈',"&isin;","&isinv;","&Element;","&in;","&#x02208;","&#8712;"));
		entries.add(new Entry('∉',"&notin;","&NotElement;","&notinva;","&#x02209;","&#8713;"));
		entries.add(new Entry('∋',"&niv;","&ReverseElement;","&ni;","&SuchThat;","&#x0220B;","&#8715;"));
		entries.add(new Entry('∌',"&notni;","&notniva;","&NotReverseElement;","&#x0220C;","&#8716;"));
		entries.add(new Entry('∏',"&prod;","&Product;","&#x0220F;","&#8719;"));
		entries.add(new Entry('∐',"&coprod;","&Coproduct;","&#x02210;","&#8720;"));
		entries.add(new Entry('∑',"&sum;","&Sum;","&#x02211;","&#8721;"));
		entries.add(new Entry('−',"&minus;","&#x02212;","&#8722;"));
		entries.add(new Entry('∓',"&mnplus;","&mp;","&MinusPlus;","&#x02213;","&#8723;"));
		entries.add(new Entry('∔',"&plusdo;","&dotplus;","&#x02214;","&#8724;"));
		entries.add(new Entry('∖',"&setmn;","&setminus;","&Backslash;","&ssetmn;","&smallsetminus;","&#x02216;","&#8726;"));
		entries.add(new Entry('∗',"&lowast;","&#x02217;","&#8727;"));
		entries.add(new Entry('∘',"&compfn;","&SmallCircle;","&#x02218;","&#8728;"));
		entries.add(new Entry('√',"&radic;","&Sqrt;","&#x0221A;","&#8730;"));
		entries.add(new Entry('∝',"&prop;","&propto;","&Proportional;","&vprop;","&varpropto;","&#x0221D;","&#8733;"));
		entries.add(new Entry('∞',"&infin;","&#x0221E;","&#8734;"));
		entries.add(new Entry('∟',"&angrt;","&#x0221F;","&#8735;"));
		entries.add(new Entry('∠',"&ang;","&angle;","&#x02220;","&#8736;"));
		entries.add(new Entry('∡',"&angmsd;","&measuredangle;","&#x02221;","&#8737;"));
		entries.add(new Entry('∢',"&angsph;","&#x02222;","&#8738;"));
		entries.add(new Entry('∣',"&mid;","&VerticalBar;","&smid;","&shortmid;","&#x02223;","&#8739;"));
		entries.add(new Entry('∤',"&nmid;","&NotVerticalBar;","&nsmid;","&nshortmid;","&#x02224;","&#8740;"));
		entries.add(new Entry('∥',"&par;","&parallel;","&DoubleVerticalBar;","&spar;","&shortparallel;","&#x02225;","&#8741;"));
		entries.add(new Entry('∦',"&npar;","&nparallel;","&NotDoubleVerticalBar;","&nspar;","&nshortparallel;","&#x02226;","&#8742;"));
		entries.add(new Entry('∧',"&and;","&wedge;","&#x02227;","&#8743;"));
		entries.add(new Entry('∨',"&or;","&vee;","&#x02228;","&#8744;"));
		entries.add(new Entry('∩',"&cap;","&#x02229;","&#8745;"));
		entries.add(new Entry('∪',"&cup;","&#x0222A;","&#8746;"));
		entries.add(new Entry('∫',"&int;","&Integral;","&#x0222B;","&#8747;"));
		entries.add(new Entry('∬',"&Int;","&#x0222C;","&#8748;"));
		entries.add(new Entry('∭',"&tint;","&iiint;","&#x0222D;","&#8749;"));
		entries.add(new Entry('∮',"&conint;","&oint;","&ContourIntegral;","&#x0222E;","&#8750;"));
		entries.add(new Entry('∯',"&Conint;","&DoubleContourIntegral;","&#x0222F;","&#8751;"));
		entries.add(new Entry('∰',"&Cconint;","&#x02230;","&#8752;"));
		entries.add(new Entry('∱',"&cwint;","&#x02231;","&#8753;"));
		entries.add(new Entry('∲',"&cwconint;","&ClockwiseContourIntegral;","&#x02232;","&#8754;"));
		entries.add(new Entry('∳',"&awconint;","&CounterClockwiseContourIntegral;","&#x02233;","&#8755;"));
		entries.add(new Entry('∴',"&there4;","&therefore;","&Therefore;","&#x02234;","&#8756;"));
		entries.add(new Entry('∵',"&becaus;","&because;","&Because;","&#x02235;","&#8757;"));
		entries.add(new Entry('∶',"&ratio;","&#x02236;","&#8758;"));
		entries.add(new Entry('∷',"&Colon;","&Proportion;","&#x02237;","&#8759;"));
		entries.add(new Entry('∸',"&minusd;","&dotminus;","&#x02238;","&#8760;"));
		entries.add(new Entry('∺',"&mDDot;","&#x0223A;","&#8762;"));
		entries.add(new Entry('∻',"&homtht;","&#x0223B;","&#8763;"));
		entries.add(new Entry('∼',"&sim;","&Tilde;","&thksim;","&thicksim;","&#x0223C;","&#8764;"));
		entries.add(new Entry('∽',"&bsim;","&backsim;","&#x0223D;","&#8765;"));
		entries.add(new Entry('∾',"&ac;","&mstpos;","&#x0223E;","&#8766;"));
		entries.add(new Entry('∿',"&acd;","&#x0223F;","&#8767;"));
		entries.add(new Entry('≀',"&wreath;","&VerticalTilde;","&wr;","&#x02240;","&#8768;"));
		entries.add(new Entry('≁',"&nsim;","&NotTilde;","&#x02241;","&#8769;"));
		entries.add(new Entry('≂',"&esim;","&EqualTilde;","&eqsim;","&#x02242;","&#8770;"));
		entries.add(new Entry('≃',"&sime;","&TildeEqual;","&simeq;","&#x02243;","&#8771;"));
		entries.add(new Entry('≄',"&nsime;","&nsimeq;","&NotTildeEqual;","&#x02244;","&#8772;"));
		entries.add(new Entry('≅',"&cong;","&TildeFullEqual;","&#x02245;","&#8773;"));
		entries.add(new Entry('≆',"&simne;","&#x02246;","&#8774;"));
		entries.add(new Entry('≇',"&ncong;","&NotTildeFullEqual;","&#x02247;","&#8775;"));
		entries.add(new Entry('≈',"&asymp;","&ap;","&TildeTilde;","&approx;","&thkap;","&thickapprox;","&#x02248;","&#8776;"));
		entries.add(new Entry('≉',"&nap;","&NotTildeTilde;","&napprox;","&#x02249;","&#8777;"));
		entries.add(new Entry('≊',"&ape;","&approxeq;","&#x0224A;","&#8778;"));
		entries.add(new Entry('≋',"&apid;","&#x0224B;","&#8779;"));
		entries.add(new Entry('≌',"&bcong;","&backcong;","&#x0224C;","&#8780;"));
		entries.add(new Entry('≍',"&asympeq;","&CupCap;","&#x0224D;","&#8781;"));
		entries.add(new Entry('≎',"&bump;","&HumpDownHump;","&Bumpeq;","&#x0224E;","&#8782;"));
		entries.add(new Entry('≏',"&bumpe;","&HumpEqual;","&bumpeq;","&#x0224F;","&#8783;"));
		entries.add(new Entry('≐',"&esdot;","&DotEqual;","&doteq;","&#x02250;","&#8784;"));
		entries.add(new Entry('≑',"&eDot;","&doteqdot;","&#x02251;","&#8785;"));
		entries.add(new Entry('≒',"&efDot;","&fallingdotseq;","&#x02252;","&#8786;"));
		entries.add(new Entry('≓',"&erDot;","&risingdotseq;","&#x02253;","&#8787;"));
		entries.add(new Entry('≔',"&colone;","&coloneq;","&Assign;","&#x02254;","&#8788;"));
		entries.add(new Entry('≕',"&ecolon;","&eqcolon;","&#x02255;","&#8789;"));
		entries.add(new Entry('≖',"&ecir;","&eqcirc;","&#x02256;","&#8790;"));
		entries.add(new Entry('≗',"&cire;","&circeq;","&#x02257;","&#8791;"));
		entries.add(new Entry('≙',"&wedgeq;","&#x02259;","&#8793;"));
		entries.add(new Entry('≚',"&veeeq;","&#x0225A;","&#8794;"));
		entries.add(new Entry('≜',"&trie;","&triangleq;","&#x0225C;","&#8796;"));
		entries.add(new Entry('≟',"&equest;","&questeq;","&#x0225F;","&#8799;"));
		entries.add(new Entry('≠',"&ne;","&NotEqual;","&#x02260;","&#8800;"));
		entries.add(new Entry('≡',"&equiv;","&Congruent;","&#x02261;","&#8801;"));
		entries.add(new Entry('≢',"&nequiv;","&NotCongruent;","&#x02262;","&#8802;"));
		entries.add(new Entry('≤',"&le;","&leq;","&#x02264;","&#8804;"));
		entries.add(new Entry('≥',"&ge;","&GreaterEqual;","&geq;","&#x02265;","&#8805;"));
		entries.add(new Entry('≦',"&lE;","&LessFullEqual;","&leqq;","&#x02266;","&#8806;"));
		entries.add(new Entry('≧',"&gE;","&GreaterFullEqual;","&geqq;","&#x02267;","&#8807;"));
		entries.add(new Entry('≨',"&lnE;","&lneqq;","&#x02268;","&#8808;"));
		entries.add(new Entry('≩',"&gnE;","&gneqq;","&#x02269;","&#8809;"));
		entries.add(new Entry('≪',"&Lt;","&NestedLessLess;","&ll;","&#x0226A;","&#8810;"));
		entries.add(new Entry('≫',"&Gt;","&NestedGreaterGreater;","&gg;","&#x0226B;","&#8811;"));
		entries.add(new Entry('≬',"&twixt;","&between;","&#x0226C;","&#8812;"));
		entries.add(new Entry('≭',"&NotCupCap;","&#x0226D;","&#8813;"));
		entries.add(new Entry('≮',"&nlt;","&NotLess;","&nless;","&#x0226E;","&#8814;"));
		entries.add(new Entry('≯',"&ngt;","&NotGreater;","&ngtr;","&#x0226F;","&#8815;"));
		entries.add(new Entry('≰',"&nle;","&NotLessEqual;","&nleq;","&#x02270;","&#8816;"));
		entries.add(new Entry('≱',"&nge;","&NotGreaterEqual;","&ngeq;","&#x02271;","&#8817;"));
		entries.add(new Entry('≲',"&lsim;","&LessTilde;","&lesssim;","&#x02272;","&#8818;"));
		entries.add(new Entry('≳',"&gsim;","&gtrsim;","&GreaterTilde;","&#x02273;","&#8819;"));
		entries.add(new Entry('≴',"&nlsim;","&NotLessTilde;","&#x02274;","&#8820;"));
		entries.add(new Entry('≵',"&ngsim;","&NotGreaterTilde;","&#x02275;","&#8821;"));
		entries.add(new Entry('≶',"&lg;","&lessgtr;","&LessGreater;","&#x02276;","&#8822;"));
		entries.add(new Entry('≷',"&gl;","&gtrless;","&GreaterLess;","&#x02277;","&#8823;"));
		entries.add(new Entry('≸',"&ntlg;","&NotLessGreater;","&#x02278;","&#8824;"));
		entries.add(new Entry('≹',"&ntgl;","&NotGreaterLess;","&#x02279;","&#8825;"));
		entries.add(new Entry('≺',"&pr;","&Precedes;","&prec;","&#x0227A;","&#8826;"));
		entries.add(new Entry('≻',"&sc;","&Succeeds;","&succ;","&#x0227B;","&#8827;"));
		entries.add(new Entry('≼',"&prcue;","&PrecedesSlantEqual;","&preccurlyeq;","&#x0227C;","&#8828;"));
		entries.add(new Entry('≽',"&sccue;","&SucceedsSlantEqual;","&succcurlyeq;","&#x0227D;","&#8829;"));
		entries.add(new Entry('≾',"&prsim;","&precsim;","&PrecedesTilde;","&#x0227E;","&#8830;"));
		entries.add(new Entry('≿',"&scsim;","&succsim;","&SucceedsTilde;","&#x0227F;","&#8831;"));
		entries.add(new Entry('⊀',"&npr;","&nprec;","&NotPrecedes;","&#x02280;","&#8832;"));
		entries.add(new Entry('⊁',"&nsc;","&nsucc;","&NotSucceeds;","&#x02281;","&#8833;"));
		entries.add(new Entry('⊂',"&sub;","&subset;","&#x02282;","&#8834;"));
		entries.add(new Entry('⊃',"&sup;","&supset;","&Superset;","&#x02283;","&#8835;"));
		entries.add(new Entry('⊄',"&nsub;","&#x02284;","&#8836;"));
		entries.add(new Entry('⊅',"&nsup;","&#x02285;","&#8837;"));
		entries.add(new Entry('⊆',"&sube;","&SubsetEqual;","&subseteq;","&#x02286;","&#8838;"));
		entries.add(new Entry('⊇',"&supe;","&supseteq;","&SupersetEqual;","&#x02287;","&#8839;"));
		entries.add(new Entry('⊈',"&nsube;","&nsubseteq;","&NotSubsetEqual;","&#x02288;","&#8840;"));
		entries.add(new Entry('⊉',"&nsupe;","&nsupseteq;","&NotSupersetEqual;","&#x02289;","&#8841;"));
		entries.add(new Entry('⊊',"&subne;","&subsetneq;","&#x0228A;","&#8842;"));
		entries.add(new Entry('⊋',"&supne;","&supsetneq;","&#x0228B;","&#8843;"));
		entries.add(new Entry('⊍',"&cupdot;","&#x0228D;","&#8845;"));
		entries.add(new Entry('⊎',"&uplus;","&UnionPlus;","&#x0228E;","&#8846;"));
		entries.add(new Entry('⊏',"&sqsub;","&SquareSubset;","&sqsubset;","&#x0228F;","&#8847;"));
		entries.add(new Entry('⊐',"&sqsup;","&SquareSuperset;","&sqsupset;","&#x02290;","&#8848;"));
		entries.add(new Entry('⊑',"&sqsube;","&SquareSubsetEqual;","&sqsubseteq;","&#x02291;","&#8849;"));
		entries.add(new Entry('⊒',"&sqsupe;","&SquareSupersetEqual;","&sqsupseteq;","&#x02292;","&#8850;"));
		entries.add(new Entry('⊓',"&sqcap;","&SquareIntersection;","&#x02293;","&#8851;"));
		entries.add(new Entry('⊔',"&sqcup;","&SquareUnion;","&#x02294;","&#8852;"));
		entries.add(new Entry('⊕',"&oplus;","&CirclePlus;","&#x02295;","&#8853;"));
		entries.add(new Entry('⊖',"&ominus;","&CircleMinus;","&#x02296;","&#8854;"));
		entries.add(new Entry('⊗',"&otimes;","&CircleTimes;","&#x02297;","&#8855;"));
		entries.add(new Entry('⊘',"&osol;","&#x02298;","&#8856;"));
		entries.add(new Entry('⊙',"&odot;","&CircleDot;","&#x02299;","&#8857;"));
		entries.add(new Entry('⊚',"&ocir;","&circledcirc;","&#x0229A;","&#8858;"));
		entries.add(new Entry('⊛',"&oast;","&circledast;","&#x0229B;","&#8859;"));
		entries.add(new Entry('⊝',"&odash;","&circleddash;","&#x0229D;","&#8861;"));
		entries.add(new Entry('⊞',"&plusb;","&boxplus;","&#x0229E;","&#8862;"));
		entries.add(new Entry('⊟',"&minusb;","&boxminus;","&#x0229F;","&#8863;"));
		entries.add(new Entry('⊠',"&timesb;","&boxtimes;","&#x022A0;","&#8864;"));
		entries.add(new Entry('⊡',"&sdotb;","&dotsquare;","&#x022A1;","&#8865;"));
		entries.add(new Entry('⊢',"&vdash;","&RightTee;","&#x022A2;","&#8866;"));
		entries.add(new Entry('⊣',"&dashv;","&LeftTee;","&#x022A3;","&#8867;"));
		entries.add(new Entry('⊤',"&top;","&DownTee;","&#x022A4;","&#8868;"));
		entries.add(new Entry('⊥',"&bottom;","&bot;","&perp;","&UpTee;","&#x022A5;","&#8869;"));
		entries.add(new Entry('⊧',"&models;","&#x022A7;","&#8871;"));
		entries.add(new Entry('⊨',"&vDash;","&DoubleRightTee;","&#x022A8;","&#8872;"));
		entries.add(new Entry('⊩',"&Vdash;","&#x022A9;","&#8873;"));
		entries.add(new Entry('⊪',"&Vvdash;","&#x022AA;","&#8874;"));
		entries.add(new Entry('⊫',"&VDash;","&#x022AB;","&#8875;"));
		entries.add(new Entry('⊬',"&nvdash;","&#x022AC;","&#8876;"));
		entries.add(new Entry('⊭',"&nvDash;","&#x022AD;","&#8877;"));
		entries.add(new Entry('⊮',"&nVdash;","&#x022AE;","&#8878;"));
		entries.add(new Entry('⊯',"&nVDash;","&#x022AF;","&#8879;"));
		entries.add(new Entry('⊰',"&prurel;","&#x022B0;","&#8880;"));
		entries.add(new Entry('⊲',"&vltri;","&vartriangleleft;","&LeftTriangle;","&#x022B2;","&#8882;"));
		entries.add(new Entry('⊳',"&vrtri;","&vartriangleright;","&RightTriangle;","&#x022B3;","&#8883;"));
		entries.add(new Entry('⊴',"&ltrie;","&trianglelefteq;","&LeftTriangleEqual;","&#x022B4;","&#8884;"));
		entries.add(new Entry('⊵',"&rtrie;","&trianglerighteq;","&RightTriangleEqual;","&#x022B5;","&#8885;"));
		entries.add(new Entry('⊶',"&origof;","&#x022B6;","&#8886;"));
		entries.add(new Entry('⊷',"&imof;","&#x022B7;","&#8887;"));
		entries.add(new Entry('⊸',"&mumap;","&multimap;","&#x022B8;","&#8888;"));
		entries.add(new Entry('⊹',"&hercon;","&#x022B9;","&#8889;"));
		entries.add(new Entry('⊺',"&intcal;","&intercal;","&#x022BA;","&#8890;"));
		entries.add(new Entry('⊻',"&veebar;","&#x022BB;","&#8891;"));
		entries.add(new Entry('⊽',"&barvee;","&#x022BD;","&#8893;"));
		entries.add(new Entry('⊾',"&angrtvb;","&#x022BE;","&#8894;"));
		entries.add(new Entry('⊿',"&lrtri;","&#x022BF;","&#8895;"));
		entries.add(new Entry('⋀',"&xwedge;","&Wedge;","&bigwedge;","&#x022C0;","&#8896;"));
		entries.add(new Entry('⋁',"&xvee;","&Vee;","&bigvee;","&#x022C1;","&#8897;"));
		entries.add(new Entry('⋂',"&xcap;","&Intersection;","&bigcap;","&#x022C2;","&#8898;"));
		entries.add(new Entry('⋃',"&xcup;","&Union;","&bigcup;","&#x022C3;","&#8899;"));
		entries.add(new Entry('⋄',"&diam;","&diamond;","&Diamond;","&#x022C4;","&#8900;"));
		entries.add(new Entry('⋅',"&sdot;","&#x022C5;","&#8901;"));
		entries.add(new Entry('⋆',"&sstarf;","&Star;","&#x022C6;","&#8902;"));
		entries.add(new Entry('⋇',"&divonx;","&divideontimes;","&#x022C7;","&#8903;"));
		entries.add(new Entry('⋈',"&bowtie;","&#x022C8;","&#8904;"));
		entries.add(new Entry('⋉',"&ltimes;","&#x022C9;","&#8905;"));
		entries.add(new Entry('⋊',"&rtimes;","&#x022CA;","&#8906;"));
		entries.add(new Entry('⋋',"&lthree;","&leftthreetimes;","&#x022CB;","&#8907;"));
		entries.add(new Entry('⋌',"&rthree;","&rightthreetimes;","&#x022CC;","&#8908;"));
		entries.add(new Entry('⋍',"&bsime;","&backsimeq;","&#x022CD;","&#8909;"));
		entries.add(new Entry('⋎',"&cuvee;","&curlyvee;","&#x022CE;","&#8910;"));
		entries.add(new Entry('⋏',"&cuwed;","&curlywedge;","&#x022CF;","&#8911;"));
		entries.add(new Entry('⋐',"&Sub;","&Subset;","&#x022D0;","&#8912;"));
		entries.add(new Entry('⋑',"&Sup;","&Supset;","&#x022D1;","&#8913;"));
		entries.add(new Entry('⋒',"&Cap;","&#x022D2;","&#8914;"));
		entries.add(new Entry('⋓',"&Cup;","&#x022D3;","&#8915;"));
		entries.add(new Entry('⋔',"&fork;","&pitchfork;","&#x022D4;","&#8916;"));
		entries.add(new Entry('⋕',"&epar;","&#x022D5;","&#8917;"));
		entries.add(new Entry('⋖',"&ltdot;","&lessdot;","&#x022D6;","&#8918;"));
		entries.add(new Entry('⋗',"&gtdot;","&gtrdot;","&#x022D7;","&#8919;"));
		entries.add(new Entry('⋘',"&Ll;","&#x022D8;","&#8920;"));
		entries.add(new Entry('⋙',"&Gg;","&ggg;","&#x022D9;","&#8921;"));
		entries.add(new Entry('⋚',"&leg;","&LessEqualGreater;","&lesseqgtr;","&#x022DA;","&#8922;"));
		entries.add(new Entry('⋛',"&gel;","&gtreqless;","&GreaterEqualLess;","&#x022DB;","&#8923;"));
		entries.add(new Entry('⋞',"&cuepr;","&curlyeqprec;","&#x022DE;","&#8926;"));
		entries.add(new Entry('⋟',"&cuesc;","&curlyeqsucc;","&#x022DF;","&#8927;"));
		entries.add(new Entry('⋠',"&nprcue;","&NotPrecedesSlantEqual;","&#x022E0;","&#8928;"));
		entries.add(new Entry('⋡',"&nsccue;","&NotSucceedsSlantEqual;","&#x022E1;","&#8929;"));
		entries.add(new Entry('⋢',"&nsqsube;","&NotSquareSubsetEqual;","&#x022E2;","&#8930;"));
		entries.add(new Entry('⋣',"&nsqsupe;","&NotSquareSupersetEqual;","&#x022E3;","&#8931;"));
		entries.add(new Entry('⋦',"&lnsim;","&#x022E6;","&#8934;"));
		entries.add(new Entry('⋧',"&gnsim;","&#x022E7;","&#8935;"));
		entries.add(new Entry('⋨',"&prnsim;","&precnsim;","&#x022E8;","&#8936;"));
		entries.add(new Entry('⋩',"&scnsim;","&succnsim;","&#x022E9;","&#8937;"));
		entries.add(new Entry('⋪',"&nltri;","&ntriangleleft;","&NotLeftTriangle;","&#x022EA;","&#8938;"));
		entries.add(new Entry('⋫',"&nrtri;","&ntriangleright;","&NotRightTriangle;","&#x022EB;","&#8939;"));
		entries.add(new Entry('⋬',"&nltrie;","&ntrianglelefteq;","&NotLeftTriangleEqual;","&#x022EC;","&#8940;"));
		entries.add(new Entry('⋭',"&nrtrie;","&ntrianglerighteq;","&NotRightTriangleEqual;","&#x022ED;","&#8941;"));
		entries.add(new Entry('⋮',"&vellip;","&#x022EE;","&#8942;"));
		entries.add(new Entry('⋯',"&ctdot;","&#x022EF;","&#8943;"));
		entries.add(new Entry('⋰',"&utdot;","&#x022F0;","&#8944;"));
		entries.add(new Entry('⋱',"&dtdot;","&#x022F1;","&#8945;"));
		entries.add(new Entry('⋲',"&disin;","&#x022F2;","&#8946;"));
		entries.add(new Entry('⋳',"&isinsv;","&#x022F3;","&#8947;"));
		entries.add(new Entry('⋴',"&isins;","&#x022F4;","&#8948;"));
		entries.add(new Entry('⋵',"&isindot;","&#x022F5;","&#8949;"));
		entries.add(new Entry('⋶',"&notinvc;","&#x022F6;","&#8950;"));
		entries.add(new Entry('⋷',"&notinvb;","&#x022F7;","&#8951;"));
		entries.add(new Entry('⋹',"&isinE;","&#x022F9;","&#8953;"));
		entries.add(new Entry('⋺',"&nisd;","&#x022FA;","&#8954;"));
		entries.add(new Entry('⋻',"&xnis;","&#x022FB;","&#8955;"));
		entries.add(new Entry('⋼',"&nis;","&#x022FC;","&#8956;"));
		entries.add(new Entry('⋽',"&notnivc;","&#x022FD;","&#8957;"));
		entries.add(new Entry('⋾',"&notnivb;","&#x022FE;","&#8958;"));
		entries.add(new Entry('⌅',"&barwed;","&barwedge;","&#x02305;","&#8965;"));
		entries.add(new Entry('⌆',"&Barwed;","&doublebarwedge;","&#x02306;","&#8966;"));
		entries.add(new Entry('⌈',"&lceil;","&LeftCeiling;","&#x02308;","&#8968;"));
		entries.add(new Entry('⌉',"&rceil;","&RightCeiling;","&#x02309;","&#8969;"));
		entries.add(new Entry('⌊',"&lfloor;","&LeftFloor;","&#x0230A;","&#8970;"));
		entries.add(new Entry('⌋',"&rfloor;","&RightFloor;","&#x0230B;","&#8971;"));
		entries.add(new Entry('⌌',"&drcrop;","&#x0230C;","&#8972;"));
		entries.add(new Entry('⌍',"&dlcrop;","&#x0230D;","&#8973;"));
		entries.add(new Entry('⌎',"&urcrop;","&#x0230E;","&#8974;"));
		entries.add(new Entry('⌏',"&ulcrop;","&#x0230F;","&#8975;"));
		entries.add(new Entry('⌐',"&bnot;","&#x02310;","&#8976;"));
		entries.add(new Entry('⌒',"&profline;","&#x02312;","&#8978;"));
		entries.add(new Entry('⌓',"&profsurf;","&#x02313;","&#8979;"));
		entries.add(new Entry('⌕',"&telrec;","&#x02315;","&#8981;"));
		entries.add(new Entry('⌖',"&target;","&#x02316;","&#8982;"));
		entries.add(new Entry('⌜',"&ulcorn;","&ulcorner;","&#x0231C;","&#8988;"));
		entries.add(new Entry('⌝',"&urcorn;","&urcorner;","&#x0231D;","&#8989;"));
		entries.add(new Entry('⌞',"&dlcorn;","&llcorner;","&#x0231E;","&#8990;"));
		entries.add(new Entry('⌟',"&drcorn;","&lrcorner;","&#x0231F;","&#8991;"));
		entries.add(new Entry('⌢',"&frown;","&sfrown;","&#x02322;","&#8994;"));
		entries.add(new Entry('⌣',"&smile;","&ssmile;","&#x02323;","&#8995;"));
		entries.add(new Entry('⌭',"&cylcty;","&#x0232D;","&#9005;"));
		entries.add(new Entry('⌮',"&profalar;","&#x0232E;","&#9006;"));
		entries.add(new Entry('⌶',"&topbot;","&#x02336;","&#9014;"));
		entries.add(new Entry('⌽',"&ovbar;","&#x0233D;","&#9021;"));
		entries.add(new Entry('⌿',"&solbar;","&#x0233F;","&#9023;"));
		entries.add(new Entry('⍼',"&angzarr;","&#x0237C;","&#9084;"));
		entries.add(new Entry('⎰',"&lmoust;","&lmoustache;","&#x023B0;","&#9136;"));
		entries.add(new Entry('⎱',"&rmoust;","&rmoustache;","&#x023B1;","&#9137;"));
		entries.add(new Entry('⎴',"&tbrk;","&OverBracket;","&#x023B4;","&#9140;"));
		entries.add(new Entry('⎵',"&bbrk;","&UnderBracket;","&#x023B5;","&#9141;"));
		entries.add(new Entry('⎶',"&bbrktbrk;","&#x023B6;","&#9142;"));
		entries.add(new Entry('⏜',"&OverParenthesis;","&#x023DC;","&#9180;"));
		entries.add(new Entry('⏝',"&UnderParenthesis;","&#x023DD;","&#9181;"));
		entries.add(new Entry('⏞',"&OverBrace;","&#x023DE;","&#9182;"));
		entries.add(new Entry('⏟',"&UnderBrace;","&#x023DF;","&#9183;"));
		entries.add(new Entry('⏢',"&trpezium;","&#x023E2;","&#9186;"));
		entries.add(new Entry('⏧',"&elinters;","&#x023E7;","&#9191;"));
		entries.add(new Entry('␣',"&blank;","&#x02423;","&#9251;"));
		entries.add(new Entry('Ⓢ',"&oS;","&circledS;","&#x024C8;","&#9416;"));
		entries.add(new Entry('─',"&boxh;","&HorizontalLine;","&#x02500;","&#9472;"));
		entries.add(new Entry('│',"&boxv;","&#x02502;","&#9474;"));
		entries.add(new Entry('┌',"&boxdr;","&#x0250C;","&#9484;"));
		entries.add(new Entry('┐',"&boxdl;","&#x02510;","&#9488;"));
		entries.add(new Entry('└',"&boxur;","&#x02514;","&#9492;"));
		entries.add(new Entry('┘',"&boxul;","&#x02518;","&#9496;"));
		entries.add(new Entry('├',"&boxvr;","&#x0251C;","&#9500;"));
		entries.add(new Entry('┤',"&boxvl;","&#x02524;","&#9508;"));
		entries.add(new Entry('┬',"&boxhd;","&#x0252C;","&#9516;"));
		entries.add(new Entry('┴',"&boxhu;","&#x02534;","&#9524;"));
		entries.add(new Entry('┼',"&boxvh;","&#x0253C;","&#9532;"));
		entries.add(new Entry('═',"&boxH;","&#x02550;","&#9552;"));
		entries.add(new Entry('║',"&boxV;","&#x02551;","&#9553;"));
		entries.add(new Entry('╒',"&boxdR;","&#x02552;","&#9554;"));
		entries.add(new Entry('╓',"&boxDr;","&#x02553;","&#9555;"));
		entries.add(new Entry('╔',"&boxDR;","&#x02554;","&#9556;"));
		entries.add(new Entry('╕',"&boxdL;","&#x02555;","&#9557;"));
		entries.add(new Entry('╖',"&boxDl;","&#x02556;","&#9558;"));
		entries.add(new Entry('╗',"&boxDL;","&#x02557;","&#9559;"));
		entries.add(new Entry('╘',"&boxuR;","&#x02558;","&#9560;"));
		entries.add(new Entry('╙',"&boxUr;","&#x02559;","&#9561;"));
		entries.add(new Entry('╚',"&boxUR;","&#x0255A;","&#9562;"));
		entries.add(new Entry('╛',"&boxuL;","&#x0255B;","&#9563;"));
		entries.add(new Entry('╜',"&boxUl;","&#x0255C;","&#9564;"));
		entries.add(new Entry('╝',"&boxUL;","&#x0255D;","&#9565;"));
		entries.add(new Entry('╞',"&boxvR;","&#x0255E;","&#9566;"));
		entries.add(new Entry('╟',"&boxVr;","&#x0255F;","&#9567;"));
		entries.add(new Entry('╠',"&boxVR;","&#x02560;","&#9568;"));
		entries.add(new Entry('╡',"&boxvL;","&#x02561;","&#9569;"));
		entries.add(new Entry('╢',"&boxVl;","&#x02562;","&#9570;"));
		entries.add(new Entry('╣',"&boxVL;","&#x02563;","&#9571;"));
		entries.add(new Entry('╤',"&boxHd;","&#x02564;","&#9572;"));
		entries.add(new Entry('╥',"&boxhD;","&#x02565;","&#9573;"));
		entries.add(new Entry('╦',"&boxHD;","&#x02566;","&#9574;"));
		entries.add(new Entry('╧',"&boxHu;","&#x02567;","&#9575;"));
		entries.add(new Entry('╨',"&boxhU;","&#x02568;","&#9576;"));
		entries.add(new Entry('╩',"&boxHU;","&#x02569;","&#9577;"));
		entries.add(new Entry('╪',"&boxvH;","&#x0256A;","&#9578;"));
		entries.add(new Entry('╫',"&boxVh;","&#x0256B;","&#9579;"));
		entries.add(new Entry('╬',"&boxVH;","&#x0256C;","&#9580;"));
		entries.add(new Entry('▀',"&uhblk;","&#x02580;","&#9600;"));
		entries.add(new Entry('▄',"&lhblk;","&#x02584;","&#9604;"));
		entries.add(new Entry('█',"&block;","&#x02588;","&#9608;"));
		entries.add(new Entry('░',"&blk14;","&#x02591;","&#9617;"));
		entries.add(new Entry('▒',"&blk12;","&#x02592;","&#9618;"));
		entries.add(new Entry('▓',"&blk34;","&#x02593;","&#9619;"));
		entries.add(new Entry('□',"&squ;","&square;","&Square;","&#x025A1;","&#9633;"));
		entries.add(new Entry('▪',"&squf;","&squarf;","&blacksquare;","&FilledVerySmallSquare;","&#x025AA;","&#9642;"));
		entries.add(new Entry('▫',"&EmptyVerySmallSquare;","&#x025AB;","&#9643;"));
		entries.add(new Entry('▭',"&rect;","&#x025AD;","&#9645;"));
		entries.add(new Entry('▮',"&marker;","&#x025AE;","&#9646;"));
		entries.add(new Entry('▱',"&fltns;","&#x025B1;","&#9649;"));
		entries.add(new Entry('△',"&xutri;","&bigtriangleup;","&#x025B3;","&#9651;"));
		entries.add(new Entry('▴',"&utrif;","&blacktriangle;","&#x025B4;","&#9652;"));
		entries.add(new Entry('▵',"&utri;","&triangle;","&#x025B5;","&#9653;"));
		entries.add(new Entry('▸',"&rtrif;","&blacktriangleright;","&#x025B8;","&#9656;"));
		entries.add(new Entry('▹',"&rtri;","&triangleright;","&#x025B9;","&#9657;"));
		entries.add(new Entry('▽',"&xdtri;","&bigtriangledown;","&#x025BD;","&#9661;"));
		entries.add(new Entry('▾',"&dtrif;","&blacktriangledown;","&#x025BE;","&#9662;"));
		entries.add(new Entry('▿',"&dtri;","&triangledown;","&#x025BF;","&#9663;"));
		entries.add(new Entry('◂',"&ltrif;","&blacktriangleleft;","&#x025C2;","&#9666;"));
		entries.add(new Entry('◃',"&ltri;","&triangleleft;","&#x025C3;","&#9667;"));
		entries.add(new Entry('◊',"&loz;","&lozenge;","&#x025CA;","&#9674;"));
		entries.add(new Entry('○',"&cir;","&#x025CB;","&#9675;"));
		entries.add(new Entry('◬',"&tridot;","&#x025EC;","&#9708;"));
		entries.add(new Entry('◯',"&xcirc;","&bigcirc;","&#x025EF;","&#9711;"));
		entries.add(new Entry('◸',"&ultri;","&#x025F8;","&#9720;"));
		entries.add(new Entry('◹',"&urtri;","&#x025F9;","&#9721;"));
		entries.add(new Entry('◺',"&lltri;","&#x025FA;","&#9722;"));
		entries.add(new Entry('◻',"&EmptySmallSquare;","&#x025FB;","&#9723;"));
		entries.add(new Entry('◼',"&FilledSmallSquare;","&#x025FC;","&#9724;"));
		entries.add(new Entry('★',"&starf;","&bigstar;","&#x02605;","&#9733;"));
		entries.add(new Entry('☆',"&star;","&#x02606;","&#9734;"));
		entries.add(new Entry('☎',"&phone;","&#x0260E;","&#9742;"));
		entries.add(new Entry('♀',"&female;","&#x02640;","&#9792;"));
		entries.add(new Entry('♂',"&male;","&#x02642;","&#9794;"));
		entries.add(new Entry('♠',"&spades;","&spadesuit;","&#x02660;","&#9824;"));
		entries.add(new Entry('♣',"&clubs;","&clubsuit;","&#x02663;","&#9827;"));
		entries.add(new Entry('♥',"&hearts;","&heartsuit;","&#x02665;","&#9829;"));
		entries.add(new Entry('♦',"&diams;","&diamondsuit;","&#x02666;","&#9830;"));
		entries.add(new Entry('♪',"&sung;","&#x0266A;","&#9834;"));
		entries.add(new Entry('♭',"&flat;","&#x0266D;","&#9837;"));
		entries.add(new Entry('♮',"&natur;","&natural;","&#x0266E;","&#9838;"));
		entries.add(new Entry('♯',"&sharp;","&#x0266F;","&#9839;"));
		entries.add(new Entry('✓',"&check;","&checkmark;","&#x02713;","&#10003;"));
		entries.add(new Entry('✗',"&cross;","&#x02717;","&#10007;"));
		entries.add(new Entry('✠',"&malt;","&maltese;","&#x02720;","&#10016;"));
		entries.add(new Entry('✶',"&sext;","&#x02736;","&#10038;"));
		entries.add(new Entry('❘',"&VerticalSeparator;","&#x02758;","&#10072;"));
		entries.add(new Entry('❲',"&lbbrk;","&#x02772;","&#10098;"));
		entries.add(new Entry('❳',"&rbbrk;","&#x02773;","&#10099;"));
		entries.add(new Entry('⟦',"&lobrk;","&LeftDoubleBracket;","&#x027E6;","&#10214;"));
		entries.add(new Entry('⟧',"&robrk;","&RightDoubleBracket;","&#x027E7;","&#10215;"));
		entries.add(new Entry('⟨',"&lang;","&LeftAngleBracket;","&langle;","&#x027E8;","&#10216;"));
		entries.add(new Entry('⟩',"&rang;","&RightAngleBracket;","&rangle;","&#x027E9;","&#10217;"));
		entries.add(new Entry('⟪',"&Lang;","&#x027EA;","&#10218;"));
		entries.add(new Entry('⟫',"&Rang;","&#x027EB;","&#10219;"));
		entries.add(new Entry('⟬',"&loang;","&#x027EC;","&#10220;"));
		entries.add(new Entry('⟭',"&roang;","&#x027ED;","&#10221;"));
		entries.add(new Entry('⟵',"&xlarr;","&longleftarrow;","&LongLeftArrow;","&#x027F5;","&#10229;"));
		entries.add(new Entry('⟶',"&xrarr;","&longrightarrow;","&LongRightArrow;","&#x027F6;","&#10230;"));
		entries.add(new Entry('⟷',"&xharr;","&longleftrightarrow;","&LongLeftRightArrow;","&#x027F7;","&#10231;"));
		entries.add(new Entry('⟸',"&xlArr;","&Longleftarrow;","&DoubleLongLeftArrow;","&#x027F8;","&#10232;"));
		entries.add(new Entry('⟹',"&xrArr;","&Longrightarrow;","&DoubleLongRightArrow;","&#x027F9;","&#10233;"));
		entries.add(new Entry('⟺',"&xhArr;","&Longleftrightarrow;","&DoubleLongLeftRightArrow;","&#x027FA;","&#10234;"));
		entries.add(new Entry('⟼',"&xmap;","&longmapsto;","&#x027FC;","&#10236;"));
		entries.add(new Entry('⟿',"&dzigrarr;","&#x027FF;","&#10239;"));
		entries.add(new Entry('⤂',"&nvlArr;","&#x02902;","&#10498;"));
		entries.add(new Entry('⤃',"&nvrArr;","&#x02903;","&#10499;"));
		entries.add(new Entry('⤄',"&nvHarr;","&#x02904;","&#10500;"));
		entries.add(new Entry('⤅',"&Map;","&#x02905;","&#10501;"));
		entries.add(new Entry('⤌',"&lbarr;","&#x0290C;","&#10508;"));
		entries.add(new Entry('⤍',"&rbarr;","&bkarow;","&#x0290D;","&#10509;"));
		entries.add(new Entry('⤎',"&lBarr;","&#x0290E;","&#10510;"));
		entries.add(new Entry('⤏',"&rBarr;","&dbkarow;","&#x0290F;","&#10511;"));
		entries.add(new Entry('⤐',"&RBarr;","&drbkarow;","&#x02910;","&#10512;"));
		entries.add(new Entry('⤑',"&DDotrahd;","&#x02911;","&#10513;"));
		entries.add(new Entry('⤒',"&UpArrowBar;","&#x02912;","&#10514;"));
		entries.add(new Entry('⤓',"&DownArrowBar;","&#x02913;","&#10515;"));
		entries.add(new Entry('⤖',"&Rarrtl;","&#x02916;","&#10518;"));
		entries.add(new Entry('⤙',"&latail;","&#x02919;","&#10521;"));
		entries.add(new Entry('⤚',"&ratail;","&#x0291A;","&#10522;"));
		entries.add(new Entry('⤛',"&lAtail;","&#x0291B;","&#10523;"));
		entries.add(new Entry('⤜',"&rAtail;","&#x0291C;","&#10524;"));
		entries.add(new Entry('⤝',"&larrfs;","&#x0291D;","&#10525;"));
		entries.add(new Entry('⤞',"&rarrfs;","&#x0291E;","&#10526;"));
		entries.add(new Entry('⤟',"&larrbfs;","&#x0291F;","&#10527;"));
		entries.add(new Entry('⤠',"&rarrbfs;","&#x02920;","&#10528;"));
		entries.add(new Entry('⤣',"&nwarhk;","&#x02923;","&#10531;"));
		entries.add(new Entry('⤤',"&nearhk;","&#x02924;","&#10532;"));
		entries.add(new Entry('⤥',"&searhk;","&hksearow;","&#x02925;","&#10533;"));
		entries.add(new Entry('⤦',"&swarhk;","&hkswarow;","&#x02926;","&#10534;"));
		entries.add(new Entry('⤧',"&nwnear;","&#x02927;","&#10535;"));
		entries.add(new Entry('⤨',"&nesear;","&toea;","&#x02928;","&#10536;"));
		entries.add(new Entry('⤩',"&seswar;","&tosa;","&#x02929;","&#10537;"));
		entries.add(new Entry('⤪',"&swnwar;","&#x0292A;","&#10538;"));
		entries.add(new Entry('⤳',"&rarrc;","&#x02933;","&#10547;"));
		entries.add(new Entry('⤵',"&cudarrr;","&#x02935;","&#10549;"));
		entries.add(new Entry('⤶',"&ldca;","&#x02936;","&#10550;"));
		entries.add(new Entry('⤷',"&rdca;","&#x02937;","&#10551;"));
		entries.add(new Entry('⤸',"&cudarrl;","&#x02938;","&#10552;"));
		entries.add(new Entry('⤹',"&larrpl;","&#x02939;","&#10553;"));
		entries.add(new Entry('⤼',"&curarrm;","&#x0293C;","&#10556;"));
		entries.add(new Entry('⤽',"&cularrp;","&#x0293D;","&#10557;"));
		entries.add(new Entry('⥅',"&rarrpl;","&#x02945;","&#10565;"));
		entries.add(new Entry('⥈',"&harrcir;","&#x02948;","&#10568;"));
		entries.add(new Entry('⥉',"&Uarrocir;","&#x02949;","&#10569;"));
		entries.add(new Entry('⥊',"&lurdshar;","&#x0294A;","&#10570;"));
		entries.add(new Entry('⥋',"&ldrushar;","&#x0294B;","&#10571;"));
		entries.add(new Entry('⥎',"&LeftRightVector;","&#x0294E;","&#10574;"));
		entries.add(new Entry('⥏',"&RightUpDownVector;","&#x0294F;","&#10575;"));
		entries.add(new Entry('⥐',"&DownLeftRightVector;","&#x02950;","&#10576;"));
		entries.add(new Entry('⥑',"&LeftUpDownVector;","&#x02951;","&#10577;"));
		entries.add(new Entry('⥒',"&LeftVectorBar;","&#x02952;","&#10578;"));
		entries.add(new Entry('⥓',"&RightVectorBar;","&#x02953;","&#10579;"));
		entries.add(new Entry('⥔',"&RightUpVectorBar;","&#x02954;","&#10580;"));
		entries.add(new Entry('⥕',"&RightDownVectorBar;","&#x02955;","&#10581;"));
		entries.add(new Entry('⥖',"&DownLeftVectorBar;","&#x02956;","&#10582;"));
		entries.add(new Entry('⥗',"&DownRightVectorBar;","&#x02957;","&#10583;"));
		entries.add(new Entry('⥘',"&LeftUpVectorBar;","&#x02958;","&#10584;"));
		entries.add(new Entry('⥙',"&LeftDownVectorBar;","&#x02959;","&#10585;"));
		entries.add(new Entry('⥚',"&LeftTeeVector;","&#x0295A;","&#10586;"));
		entries.add(new Entry('⥛',"&RightTeeVector;","&#x0295B;","&#10587;"));
		entries.add(new Entry('⥜',"&RightUpTeeVector;","&#x0295C;","&#10588;"));
		entries.add(new Entry('⥝',"&RightDownTeeVector;","&#x0295D;","&#10589;"));
		entries.add(new Entry('⥞',"&DownLeftTeeVector;","&#x0295E;","&#10590;"));
		entries.add(new Entry('⥟',"&DownRightTeeVector;","&#x0295F;","&#10591;"));
		entries.add(new Entry('⥠',"&LeftUpTeeVector;","&#x02960;","&#10592;"));
		entries.add(new Entry('⥡',"&LeftDownTeeVector;","&#x02961;","&#10593;"));
		entries.add(new Entry('⥢',"&lHar;","&#x02962;","&#10594;"));
		entries.add(new Entry('⥣',"&uHar;","&#x02963;","&#10595;"));
		entries.add(new Entry('⥤',"&rHar;","&#x02964;","&#10596;"));
		entries.add(new Entry('⥥',"&dHar;","&#x02965;","&#10597;"));
		entries.add(new Entry('⥦',"&luruhar;","&#x02966;","&#10598;"));
		entries.add(new Entry('⥧',"&ldrdhar;","&#x02967;","&#10599;"));
		entries.add(new Entry('⥨',"&ruluhar;","&#x02968;","&#10600;"));
		entries.add(new Entry('⥩',"&rdldhar;","&#x02969;","&#10601;"));
		entries.add(new Entry('⥪',"&lharul;","&#x0296A;","&#10602;"));
		entries.add(new Entry('⥫',"&llhard;","&#x0296B;","&#10603;"));
		entries.add(new Entry('⥬',"&rharul;","&#x0296C;","&#10604;"));
		entries.add(new Entry('⥭',"&lrhard;","&#x0296D;","&#10605;"));
		entries.add(new Entry('⥮',"&udhar;","&UpEquilibrium;","&#x0296E;","&#10606;"));
		entries.add(new Entry('⥯',"&duhar;","&ReverseUpEquilibrium;","&#x0296F;","&#10607;"));
		entries.add(new Entry('⥰',"&RoundImplies;","&#x02970;","&#10608;"));
		entries.add(new Entry('⥱',"&erarr;","&#x02971;","&#10609;"));
		entries.add(new Entry('⥲',"&simrarr;","&#x02972;","&#10610;"));
		entries.add(new Entry('⥳',"&larrsim;","&#x02973;","&#10611;"));
		entries.add(new Entry('⥴',"&rarrsim;","&#x02974;","&#10612;"));
		entries.add(new Entry('⥵',"&rarrap;","&#x02975;","&#10613;"));
		entries.add(new Entry('⥶',"&ltlarr;","&#x02976;","&#10614;"));
		entries.add(new Entry('⥸',"&gtrarr;","&#x02978;","&#10616;"));
		entries.add(new Entry('⥹',"&subrarr;","&#x02979;","&#10617;"));
		entries.add(new Entry('⥻',"&suplarr;","&#x0297B;","&#10619;"));
		entries.add(new Entry('⥼',"&lfisht;","&#x0297C;","&#10620;"));
		entries.add(new Entry('⥽',"&rfisht;","&#x0297D;","&#10621;"));
		entries.add(new Entry('⥾',"&ufisht;","&#x0297E;","&#10622;"));
		entries.add(new Entry('⥿',"&dfisht;","&#x0297F;","&#10623;"));
		entries.add(new Entry('⦅',"&lopar;","&#x02985;","&#10629;"));
		entries.add(new Entry('⦆',"&ropar;","&#x02986;","&#10630;"));
		entries.add(new Entry('⦋',"&lbrke;","&#x0298B;","&#10635;"));
		entries.add(new Entry('⦌',"&rbrke;","&#x0298C;","&#10636;"));
		entries.add(new Entry('⦍',"&lbrkslu;","&#x0298D;","&#10637;"));
		entries.add(new Entry('⦎',"&rbrksld;","&#x0298E;","&#10638;"));
		entries.add(new Entry('⦏',"&lbrksld;","&#x0298F;","&#10639;"));
		entries.add(new Entry('⦐',"&rbrkslu;","&#x02990;","&#10640;"));
		entries.add(new Entry('⦑',"&langd;","&#x02991;","&#10641;"));
		entries.add(new Entry('⦒',"&rangd;","&#x02992;","&#10642;"));
		entries.add(new Entry('⦓',"&lparlt;","&#x02993;","&#10643;"));
		entries.add(new Entry('⦔',"&rpargt;","&#x02994;","&#10644;"));
		entries.add(new Entry('⦕',"&gtlPar;","&#x02995;","&#10645;"));
		entries.add(new Entry('⦖',"&ltrPar;","&#x02996;","&#10646;"));
		entries.add(new Entry('⦚',"&vzigzag;","&#x0299A;","&#10650;"));
		entries.add(new Entry('⦜',"&vangrt;","&#x0299C;","&#10652;"));
		entries.add(new Entry('⦝',"&angrtvbd;","&#x0299D;","&#10653;"));
		entries.add(new Entry('⦤',"&ange;","&#x029A4;","&#10660;"));
		entries.add(new Entry('⦥',"&range;","&#x029A5;","&#10661;"));
		entries.add(new Entry('⦦',"&dwangle;","&#x029A6;","&#10662;"));
		entries.add(new Entry('⦧',"&uwangle;","&#x029A7;","&#10663;"));
		entries.add(new Entry('⦨',"&angmsdaa;","&#x029A8;","&#10664;"));
		entries.add(new Entry('⦩',"&angmsdab;","&#x029A9;","&#10665;"));
		entries.add(new Entry('⦪',"&angmsdac;","&#x029AA;","&#10666;"));
		entries.add(new Entry('⦫',"&angmsdad;","&#x029AB;","&#10667;"));
		entries.add(new Entry('⦬',"&angmsdae;","&#x029AC;","&#10668;"));
		entries.add(new Entry('⦭',"&angmsdaf;","&#x029AD;","&#10669;"));
		entries.add(new Entry('⦮',"&angmsdag;","&#x029AE;","&#10670;"));
		entries.add(new Entry('⦯',"&angmsdah;","&#x029AF;","&#10671;"));
		entries.add(new Entry('⦰',"&bemptyv;","&#x029B0;","&#10672;"));
		entries.add(new Entry('⦱',"&demptyv;","&#x029B1;","&#10673;"));
		entries.add(new Entry('⦲',"&cemptyv;","&#x029B2;","&#10674;"));
		entries.add(new Entry('⦳',"&raemptyv;","&#x029B3;","&#10675;"));
		entries.add(new Entry('⦴',"&laemptyv;","&#x029B4;","&#10676;"));
		entries.add(new Entry('⦵',"&ohbar;","&#x029B5;","&#10677;"));
		entries.add(new Entry('⦶',"&omid;","&#x029B6;","&#10678;"));
		entries.add(new Entry('⦷',"&opar;","&#x029B7;","&#10679;"));
		entries.add(new Entry('⦹',"&operp;","&#x029B9;","&#10681;"));
		entries.add(new Entry('⦻',"&olcross;","&#x029BB;","&#10683;"));
		entries.add(new Entry('⦼',"&odsold;","&#x029BC;","&#10684;"));
		entries.add(new Entry('⦾',"&olcir;","&#x029BE;","&#10686;"));
		entries.add(new Entry('⦿',"&ofcir;","&#x029BF;","&#10687;"));
		entries.add(new Entry('⧀',"&olt;","&#x029C0;","&#10688;"));
		entries.add(new Entry('⧁',"&ogt;","&#x029C1;","&#10689;"));
		entries.add(new Entry('⧂',"&cirscir;","&#x029C2;","&#10690;"));
		entries.add(new Entry('⧃',"&cirE;","&#x029C3;","&#10691;"));
		entries.add(new Entry('⧄',"&solb;","&#x029C4;","&#10692;"));
		entries.add(new Entry('⧅',"&bsolb;","&#x029C5;","&#10693;"));
		entries.add(new Entry('⧉',"&boxbox;","&#x029C9;","&#10697;"));
		entries.add(new Entry('⧍',"&trisb;","&#x029CD;","&#10701;"));
		entries.add(new Entry('⧎',"&rtriltri;","&#x029CE;","&#10702;"));
		entries.add(new Entry('⧏',"&LeftTriangleBar;","&#x029CF;","&#10703;"));
		entries.add(new Entry('⧐',"&RightTriangleBar;","&#x029D0;","&#10704;"));
		entries.add(new Entry('⧚',"&race;","&#x029DA;","&#10714;"));
		entries.add(new Entry('⧜',"&iinfin;","&#x029DC;","&#10716;"));
		entries.add(new Entry('⧝',"&infintie;","&#x029DD;","&#10717;"));
		entries.add(new Entry('⧞',"&nvinfin;","&#x029DE;","&#10718;"));
		entries.add(new Entry('⧣',"&eparsl;","&#x029E3;","&#10723;"));
		entries.add(new Entry('⧤',"&smeparsl;","&#x029E4;","&#10724;"));
		entries.add(new Entry('⧥',"&eqvparsl;","&#x029E5;","&#10725;"));
		entries.add(new Entry('⧫',"&lozf;","&blacklozenge;","&#x029EB;","&#10731;"));
		entries.add(new Entry('⧴',"&RuleDelayed;","&#x029F4;","&#10740;"));
		entries.add(new Entry('⧶',"&dsol;","&#x029F6;","&#10742;"));
		entries.add(new Entry('⨀',"&xodot;","&bigodot;","&#x02A00;","&#10752;"));
		entries.add(new Entry('⨁',"&xoplus;","&bigoplus;","&#x02A01;","&#10753;"));
		entries.add(new Entry('⨂',"&xotime;","&bigotimes;","&#x02A02;","&#10754;"));
		entries.add(new Entry('⨄',"&xuplus;","&biguplus;","&#x02A04;","&#10756;"));
		entries.add(new Entry('⨆',"&xsqcup;","&bigsqcup;","&#x02A06;","&#10758;"));
		entries.add(new Entry('⨌',"&qint;","&iiiint;","&#x02A0C;","&#10764;"));
		entries.add(new Entry('⨍',"&fpartint;","&#x02A0D;","&#10765;"));
		entries.add(new Entry('⨐',"&cirfnint;","&#x02A10;","&#10768;"));
		entries.add(new Entry('⨑',"&awint;","&#x02A11;","&#10769;"));
		entries.add(new Entry('⨒',"&rppolint;","&#x02A12;","&#10770;"));
		entries.add(new Entry('⨓',"&scpolint;","&#x02A13;","&#10771;"));
		entries.add(new Entry('⨔',"&npolint;","&#x02A14;","&#10772;"));
		entries.add(new Entry('⨕',"&pointint;","&#x02A15;","&#10773;"));
		entries.add(new Entry('⨖',"&quatint;","&#x02A16;","&#10774;"));
		entries.add(new Entry('⨗',"&intlarhk;","&#x02A17;","&#10775;"));
		entries.add(new Entry('⨢',"&pluscir;","&#x02A22;","&#10786;"));
		entries.add(new Entry('⨣',"&plusacir;","&#x02A23;","&#10787;"));
		entries.add(new Entry('⨤',"&simplus;","&#x02A24;","&#10788;"));
		entries.add(new Entry('⨥',"&plusdu;","&#x02A25;","&#10789;"));
		entries.add(new Entry('⨦',"&plussim;","&#x02A26;","&#10790;"));
		entries.add(new Entry('⨧',"&plustwo;","&#x02A27;","&#10791;"));
		entries.add(new Entry('⨩',"&mcomma;","&#x02A29;","&#10793;"));
		entries.add(new Entry('⨪',"&minusdu;","&#x02A2A;","&#10794;"));
		entries.add(new Entry('⨭',"&loplus;","&#x02A2D;","&#10797;"));
		entries.add(new Entry('⨮',"&roplus;","&#x02A2E;","&#10798;"));
		entries.add(new Entry('⨯',"&Cross;","&#x02A2F;","&#10799;"));
		entries.add(new Entry('⨰',"&timesd;","&#x02A30;","&#10800;"));
		entries.add(new Entry('⨱',"&timesbar;","&#x02A31;","&#10801;"));
		entries.add(new Entry('⨳',"&smashp;","&#x02A33;","&#10803;"));
		entries.add(new Entry('⨴',"&lotimes;","&#x02A34;","&#10804;"));
		entries.add(new Entry('⨵',"&rotimes;","&#x02A35;","&#10805;"));
		entries.add(new Entry('⨶',"&otimesas;","&#x02A36;","&#10806;"));
		entries.add(new Entry('⨷',"&Otimes;","&#x02A37;","&#10807;"));
		entries.add(new Entry('⨸',"&odiv;","&#x02A38;","&#10808;"));
		entries.add(new Entry('⨹',"&triplus;","&#x02A39;","&#10809;"));
		entries.add(new Entry('⨺',"&triminus;","&#x02A3A;","&#10810;"));
		entries.add(new Entry('⨻',"&tritime;","&#x02A3B;","&#10811;"));
		entries.add(new Entry('⨼',"&iprod;","&intprod;","&#x02A3C;","&#10812;"));
		entries.add(new Entry('⨿',"&amalg;","&#x02A3F;","&#10815;"));
		entries.add(new Entry('⩀',"&capdot;","&#x02A40;","&#10816;"));
		entries.add(new Entry('⩂',"&ncup;","&#x02A42;","&#10818;"));
		entries.add(new Entry('⩃',"&ncap;","&#x02A43;","&#10819;"));
		entries.add(new Entry('⩄',"&capand;","&#x02A44;","&#10820;"));
		entries.add(new Entry('⩅',"&cupor;","&#x02A45;","&#10821;"));
		entries.add(new Entry('⩆',"&cupcap;","&#x02A46;","&#10822;"));
		entries.add(new Entry('⩇',"&capcup;","&#x02A47;","&#10823;"));
		entries.add(new Entry('⩈',"&cupbrcap;","&#x02A48;","&#10824;"));
		entries.add(new Entry('⩉',"&capbrcup;","&#x02A49;","&#10825;"));
		entries.add(new Entry('⩊',"&cupcup;","&#x02A4A;","&#10826;"));
		entries.add(new Entry('⩋',"&capcap;","&#x02A4B;","&#10827;"));
		entries.add(new Entry('⩌',"&ccups;","&#x02A4C;","&#10828;"));
		entries.add(new Entry('⩍',"&ccaps;","&#x02A4D;","&#10829;"));
		entries.add(new Entry('⩐',"&ccupssm;","&#x02A50;","&#10832;"));
		entries.add(new Entry('⩓',"&And;","&#x02A53;","&#10835;"));
		entries.add(new Entry('⩔',"&Or;","&#x02A54;","&#10836;"));
		entries.add(new Entry('⩕',"&andand;","&#x02A55;","&#10837;"));
		entries.add(new Entry('⩖',"&oror;","&#x02A56;","&#10838;"));
		entries.add(new Entry('⩗',"&orslope;","&#x02A57;","&#10839;"));
		entries.add(new Entry('⩘',"&andslope;","&#x02A58;","&#10840;"));
		entries.add(new Entry('⩚',"&andv;","&#x02A5A;","&#10842;"));
		entries.add(new Entry('⩛',"&orv;","&#x02A5B;","&#10843;"));
		entries.add(new Entry('⩜',"&andd;","&#x02A5C;","&#10844;"));
		entries.add(new Entry('⩝',"&ord;","&#x02A5D;","&#10845;"));
		entries.add(new Entry('⩟',"&wedbar;","&#x02A5F;","&#10847;"));
		entries.add(new Entry('⩦',"&sdote;","&#x02A66;","&#10854;"));
		entries.add(new Entry('⩪',"&simdot;","&#x02A6A;","&#10858;"));
		entries.add(new Entry('⩭',"&congdot;","&#x02A6D;","&#10861;"));
		entries.add(new Entry('⩮',"&easter;","&#x02A6E;","&#10862;"));
		entries.add(new Entry('⩯',"&apacir;","&#x02A6F;","&#10863;"));
		entries.add(new Entry('⩰',"&apE;","&#x02A70;","&#10864;"));
		entries.add(new Entry('⩱',"&eplus;","&#x02A71;","&#10865;"));
		entries.add(new Entry('⩲',"&pluse;","&#x02A72;","&#10866;"));
		entries.add(new Entry('⩳',"&Esim;","&#x02A73;","&#10867;"));
		entries.add(new Entry('⩴',"&Colone;","&#x02A74;","&#10868;"));
		entries.add(new Entry('⩵',"&Equal;","&#x02A75;","&#10869;"));
		entries.add(new Entry('⩷',"&eDDot;","&ddotseq;","&#x02A77;","&#10871;"));
		entries.add(new Entry('⩸',"&equivDD;","&#x02A78;","&#10872;"));
		entries.add(new Entry('⩹',"&ltcir;","&#x02A79;","&#10873;"));
		entries.add(new Entry('⩺',"&gtcir;","&#x02A7A;","&#10874;"));
		entries.add(new Entry('⩻',"&ltquest;","&#x02A7B;","&#10875;"));
		entries.add(new Entry('⩼',"&gtquest;","&#x02A7C;","&#10876;"));
		entries.add(new Entry('⩽',"&les;","&LessSlantEqual;","&leqslant;","&#x02A7D;","&#10877;"));
		entries.add(new Entry('⩾',"&ges;","&GreaterSlantEqual;","&geqslant;","&#x02A7E;","&#10878;"));
		entries.add(new Entry('⩿',"&lesdot;","&#x02A7F;","&#10879;"));
		entries.add(new Entry('⪀',"&gesdot;","&#x02A80;","&#10880;"));
		entries.add(new Entry('⪁',"&lesdoto;","&#x02A81;","&#10881;"));
		entries.add(new Entry('⪂',"&gesdoto;","&#x02A82;","&#10882;"));
		entries.add(new Entry('⪃',"&lesdotor;","&#x02A83;","&#10883;"));
		entries.add(new Entry('⪄',"&gesdotol;","&#x02A84;","&#10884;"));
		entries.add(new Entry('⪅',"&lap;","&lessapprox;","&#x02A85;","&#10885;"));
		entries.add(new Entry('⪆',"&gap;","&gtrapprox;","&#x02A86;","&#10886;"));
		entries.add(new Entry('⪇',"&lne;","&lneq;","&#x02A87;","&#10887;"));
		entries.add(new Entry('⪈',"&gne;","&gneq;","&#x02A88;","&#10888;"));
		entries.add(new Entry('⪉',"&lnap;","&lnapprox;","&#x02A89;","&#10889;"));
		entries.add(new Entry('⪊',"&gnap;","&gnapprox;","&#x02A8A;","&#10890;"));
		entries.add(new Entry('⪋',"&lEg;","&lesseqqgtr;","&#x02A8B;","&#10891;"));
		entries.add(new Entry('⪌',"&gEl;","&gtreqqless;","&#x02A8C;","&#10892;"));
		entries.add(new Entry('⪍',"&lsime;","&#x02A8D;","&#10893;"));
		entries.add(new Entry('⪎',"&gsime;","&#x02A8E;","&#10894;"));
		entries.add(new Entry('⪏',"&lsimg;","&#x02A8F;","&#10895;"));
		entries.add(new Entry('⪐',"&gsiml;","&#x02A90;","&#10896;"));
		entries.add(new Entry('⪑',"&lgE;","&#x02A91;","&#10897;"));
		entries.add(new Entry('⪒',"&glE;","&#x02A92;","&#10898;"));
		entries.add(new Entry('⪓',"&lesges;","&#x02A93;","&#10899;"));
		entries.add(new Entry('⪔',"&gesles;","&#x02A94;","&#10900;"));
		entries.add(new Entry('⪕',"&els;","&eqslantless;","&#x02A95;","&#10901;"));
		entries.add(new Entry('⪖',"&egs;","&eqslantgtr;","&#x02A96;","&#10902;"));
		entries.add(new Entry('⪗',"&elsdot;","&#x02A97;","&#10903;"));
		entries.add(new Entry('⪘',"&egsdot;","&#x02A98;","&#10904;"));
		entries.add(new Entry('⪙',"&el;","&#x02A99;","&#10905;"));
		entries.add(new Entry('⪚',"&eg;","&#x02A9A;","&#10906;"));
		entries.add(new Entry('⪝',"&siml;","&#x02A9D;","&#10909;"));
		entries.add(new Entry('⪞',"&simg;","&#x02A9E;","&#10910;"));
		entries.add(new Entry('⪟',"&simlE;","&#x02A9F;","&#10911;"));
		entries.add(new Entry('⪠',"&simgE;","&#x02AA0;","&#10912;"));
		entries.add(new Entry('⪡',"&LessLess;","&#x02AA1;","&#10913;"));
		entries.add(new Entry('⪢',"&GreaterGreater;","&#x02AA2;","&#10914;"));
		entries.add(new Entry('⪤',"&glj;","&#x02AA4;","&#10916;"));
		entries.add(new Entry('⪥',"&gla;","&#x02AA5;","&#10917;"));
		entries.add(new Entry('⪦',"&ltcc;","&#x02AA6;","&#10918;"));
		entries.add(new Entry('⪧',"&gtcc;","&#x02AA7;","&#10919;"));
		entries.add(new Entry('⪨',"&lescc;","&#x02AA8;","&#10920;"));
		entries.add(new Entry('⪩',"&gescc;","&#x02AA9;","&#10921;"));
		entries.add(new Entry('⪪',"&smt;","&#x02AAA;","&#10922;"));
		entries.add(new Entry('⪫',"&lat;","&#x02AAB;","&#10923;"));
		entries.add(new Entry('⪬',"&smte;","&#x02AAC;","&#10924;"));
		entries.add(new Entry('⪭',"&late;","&#x02AAD;","&#10925;"));
		entries.add(new Entry('⪮',"&bumpE;","&#x02AAE;","&#10926;"));
		entries.add(new Entry('⪯',"&pre;","&preceq;","&PrecedesEqual;","&#x02AAF;","&#10927;"));
		entries.add(new Entry('⪰',"&sce;","&succeq;","&SucceedsEqual;","&#x02AB0;","&#10928;"));
		entries.add(new Entry('⪳',"&prE;","&#x02AB3;","&#10931;"));
		entries.add(new Entry('⪴',"&scE;","&#x02AB4;","&#10932;"));
		entries.add(new Entry('⪵',"&prnE;","&precneqq;","&#x02AB5;","&#10933;"));
		entries.add(new Entry('⪶',"&scnE;","&succneqq;","&#x02AB6;","&#10934;"));
		entries.add(new Entry('⪷',"&prap;","&precapprox;","&#x02AB7;","&#10935;"));
		entries.add(new Entry('⪸',"&scap;","&succapprox;","&#x02AB8;","&#10936;"));
		entries.add(new Entry('⪹',"&prnap;","&precnapprox;","&#x02AB9;","&#10937;"));
		entries.add(new Entry('⪺',"&scnap;","&succnapprox;","&#x02ABA;","&#10938;"));
		entries.add(new Entry('⪻',"&Pr;","&#x02ABB;","&#10939;"));
		entries.add(new Entry('⪼',"&Sc;","&#x02ABC;","&#10940;"));
		entries.add(new Entry('⪽',"&subdot;","&#x02ABD;","&#10941;"));
		entries.add(new Entry('⪾',"&supdot;","&#x02ABE;","&#10942;"));
		entries.add(new Entry('⪿',"&subplus;","&#x02ABF;","&#10943;"));
		entries.add(new Entry('⫀',"&supplus;","&#x02AC0;","&#10944;"));
		entries.add(new Entry('⫁',"&submult;","&#x02AC1;","&#10945;"));
		entries.add(new Entry('⫂',"&supmult;","&#x02AC2;","&#10946;"));
		entries.add(new Entry('⫃',"&subedot;","&#x02AC3;","&#10947;"));
		entries.add(new Entry('⫄',"&supedot;","&#x02AC4;","&#10948;"));
		entries.add(new Entry('⫅',"&subE;","&subseteqq;","&#x02AC5;","&#10949;"));
		entries.add(new Entry('⫆',"&supE;","&supseteqq;","&#x02AC6;","&#10950;"));
		entries.add(new Entry('⫇',"&subsim;","&#x02AC7;","&#10951;"));
		entries.add(new Entry('⫈',"&supsim;","&#x02AC8;","&#10952;"));
		entries.add(new Entry('⫋',"&subnE;","&subsetneqq;","&#x02ACB;","&#10955;"));
		entries.add(new Entry('⫌',"&supnE;","&supsetneqq;","&#x02ACC;","&#10956;"));
		entries.add(new Entry('⫏',"&csub;","&#x02ACF;","&#10959;"));
		entries.add(new Entry('⫐',"&csup;","&#x02AD0;","&#10960;"));
		entries.add(new Entry('⫑',"&csube;","&#x02AD1;","&#10961;"));
		entries.add(new Entry('⫒',"&csupe;","&#x02AD2;","&#10962;"));
		entries.add(new Entry('⫓',"&subsup;","&#x02AD3;","&#10963;"));
		entries.add(new Entry('⫔',"&supsub;","&#x02AD4;","&#10964;"));
		entries.add(new Entry('⫕',"&subsub;","&#x02AD5;","&#10965;"));
		entries.add(new Entry('⫖',"&supsup;","&#x02AD6;","&#10966;"));
		entries.add(new Entry('⫗',"&suphsub;","&#x02AD7;","&#10967;"));
		entries.add(new Entry('⫘',"&supdsub;","&#x02AD8;","&#10968;"));
		entries.add(new Entry('⫙',"&forkv;","&#x02AD9;","&#10969;"));
		entries.add(new Entry('⫚',"&topfork;","&#x02ADA;","&#10970;"));
		entries.add(new Entry('⫛',"&mlcp;","&#x02ADB;","&#10971;"));
		entries.add(new Entry('⫤',"&Dashv;","&DoubleLeftTee;","&#x02AE4;","&#10980;"));
		entries.add(new Entry('⫦',"&Vdashl;","&#x02AE6;","&#10982;"));
		entries.add(new Entry('⫧',"&Barv;","&#x02AE7;","&#10983;"));
		entries.add(new Entry('⫨',"&vBar;","&#x02AE8;","&#10984;"));
		entries.add(new Entry('⫩',"&vBarv;","&#x02AE9;","&#10985;"));
		entries.add(new Entry('⫫',"&Vbar;","&#x02AEB;","&#10987;"));
		entries.add(new Entry('⫬',"&Not;","&#x02AEC;","&#10988;"));
		entries.add(new Entry('⫭',"&bNot;","&#x02AED;","&#10989;"));
		entries.add(new Entry('⫮',"&rnmid;","&#x02AEE;","&#10990;"));
		entries.add(new Entry('⫯',"&cirmid;","&#x02AEF;","&#10991;"));
		entries.add(new Entry('⫰',"&midcir;","&#x02AF0;","&#10992;"));
		entries.add(new Entry('⫱',"&topcir;","&#x02AF1;","&#10993;"));
		entries.add(new Entry('⫲',"&nhpar;","&#x02AF2;","&#10994;"));
		entries.add(new Entry('⫳',"&parsim;","&#x02AF3;","&#10995;"));
		entries.add(new Entry('⫽',"&parsl;","&#x02AFD;","&#11005;"));
		entries.add(new Entry('ﬀ',"&fflig;","&#x0FB00;","&#64256;"));
		entries.add(new Entry('ﬁ',"&filig;","&#x0FB01;","&#64257;"));
		entries.add(new Entry('ﬂ',"&fllig;","&#x0FB02;","&#64258;"));
		entries.add(new Entry('ﬃ',"&ffilig;","&#x0FB03;","&#64259;"));
		entries.add(new Entry('ﬄ',"&ffllig;","&#x0FB04;","&#64260;"));
		
		entries.add(new Entry(((char)0x1D49C),"&Ascr;","&#x1D49C;","&#119964;"));
		entries.add(new Entry(((char)0x1D49E),"&Cscr;","&#x1D49E;","&#119966;"));
		entries.add(new Entry(((char)0x1D49F),"&Dscr;","&#x1D49F;","&#119967;"));
		entries.add(new Entry(((char)0x1D4A2),"&Gscr;","&#x1D4A2;","&#119970;"));
		entries.add(new Entry(((char)0x1D4A5),"&Jscr;","&#x1D4A5;","&#119973;"));
		entries.add(new Entry(((char)0x1D4A6),"&Kscr;","&#x1D4A6;","&#119974;"));
		entries.add(new Entry(((char)0x1D4A9),"&Nscr;","&#x1D4A9;","&#119977;"));
		entries.add(new Entry(((char)0x1D4AA),"&Oscr;","&#x1D4AA;","&#119978;"));
		entries.add(new Entry(((char)0x1D4AB),"&Pscr;","&#x1D4AB;","&#119979;"));
		entries.add(new Entry(((char)0x1D4AC),"&Qscr;","&#x1D4AC;","&#119980;"));
		entries.add(new Entry(((char)0x1D4AE),"&Sscr;","&#x1D4AE;","&#119982;"));
		entries.add(new Entry(((char)0x1D4AF),"&Tscr;","&#x1D4AF;","&#119983;"));
		entries.add(new Entry(((char)0x1D4B0),"&Uscr;","&#x1D4B0;","&#119984;"));
		entries.add(new Entry(((char)0x1D4B1),"&Vscr;","&#x1D4B1;","&#119985;"));
		entries.add(new Entry(((char)0x1D4B2),"&Wscr;","&#x1D4B2;","&#119986;"));
		entries.add(new Entry(((char)0x1D4B3),"&Xscr;","&#x1D4B3;","&#119987;"));
		entries.add(new Entry(((char)0x1D4B4),"&Yscr;","&#x1D4B4;","&#119988;"));
		entries.add(new Entry(((char)0x1D4B5),"&Zscr;","&#x1D4B5;","&#119989;"));
		entries.add(new Entry(((char)0x1D4B6),"&ascr;","&#x1D4B6;","&#119990;"));
		entries.add(new Entry(((char)0x1D4B7),"&bscr;","&#x1D4B7;","&#119991;"));
		entries.add(new Entry(((char)0x1D4B8),"&cscr;","&#x1D4B8;","&#119992;"));
		entries.add(new Entry(((char)0x1D4B9),"&dscr;","&#x1D4B9;","&#119993;"));
		entries.add(new Entry(((char)0x1D4BB),"&fscr;","&#x1D4BB;","&#119995;"));
		entries.add(new Entry(((char)0x1D4BD),"&hscr;","&#x1D4BD;","&#119997;"));
		entries.add(new Entry(((char)0x1D4BE),"&iscr;","&#x1D4BE;","&#119998;"));
		entries.add(new Entry(((char)0x1D4BF),"&jscr;","&#x1D4BF;","&#119999;"));
		entries.add(new Entry(((char)0x1D4C0),"&kscr;","&#x1D4C0;","&#120000;"));
		entries.add(new Entry(((char)0x1D4C1),"&lscr;","&#x1D4C1;","&#120001;"));
		entries.add(new Entry(((char)0x1D4C2),"&mscr;","&#x1D4C2;","&#120002;"));
		entries.add(new Entry(((char)0x1D4C3),"&nscr;","&#x1D4C3;","&#120003;"));
		entries.add(new Entry(((char)0x1D4C5),"&pscr;","&#x1D4C5;","&#120005;"));
		entries.add(new Entry(((char)0x1D4C6),"&qscr;","&#x1D4C6;","&#120006;"));
		entries.add(new Entry(((char)0x1D4C7),"&rscr;","&#x1D4C7;","&#120007;"));
		entries.add(new Entry(((char)0x1D4C8),"&sscr;","&#x1D4C8;","&#120008;"));
		entries.add(new Entry(((char)0x1D4C9),"&tscr;","&#x1D4C9;","&#120009;"));
		entries.add(new Entry(((char)0x1D4CA),"&uscr;","&#x1D4CA;","&#120010;"));
		entries.add(new Entry(((char)0x1D4CB),"&vscr;","&#x1D4CB;","&#120011;"));
		entries.add(new Entry(((char)0x1D4CC),"&wscr;","&#x1D4CC;","&#120012;"));
		entries.add(new Entry(((char)0x1D4CD),"&xscr;","&#x1D4CD;","&#120013;"));
		entries.add(new Entry(((char)0x1D4CE),"&yscr;","&#x1D4CE;","&#120014;"));
		entries.add(new Entry(((char)0x1D4CF),"&zscr;","&#x1D4CF;","&#120015;"));
		entries.add(new Entry(((char)0x1D504),"&Afr;","&#x1D504;","&#120068;"));
		entries.add(new Entry(((char)0x1D505),"&Bfr;","&#x1D505;","&#120069;"));
		entries.add(new Entry(((char)0x1D507),"&Dfr;","&#x1D507;","&#120071;"));
		entries.add(new Entry(((char)0x1D508),"&Efr;","&#x1D508;","&#120072;"));
		entries.add(new Entry(((char)0x1D509),"&Ffr;","&#x1D509;","&#120073;"));
		entries.add(new Entry(((char)0x1D50A),"&Gfr;","&#x1D50A;","&#120074;"));
		entries.add(new Entry(((char)0x1D50D),"&Jfr;","&#x1D50D;","&#120077;"));
		entries.add(new Entry(((char)0x1D50E),"&Kfr;","&#x1D50E;","&#120078;"));
		entries.add(new Entry(((char)0x1D50F),"&Lfr;","&#x1D50F;","&#120079;"));
		entries.add(new Entry(((char)0x1D510),"&Mfr;","&#x1D510;","&#120080;"));
		entries.add(new Entry(((char)0x1D511),"&Nfr;","&#x1D511;","&#120081;"));
		entries.add(new Entry(((char)0x1D512),"&Ofr;","&#x1D512;","&#120082;"));
		entries.add(new Entry(((char)0x1D513),"&Pfr;","&#x1D513;","&#120083;"));
		entries.add(new Entry(((char)0x1D514),"&Qfr;","&#x1D514;","&#120084;"));
		entries.add(new Entry(((char)0x1D516),"&Sfr;","&#x1D516;","&#120086;"));
		entries.add(new Entry(((char)0x1D517),"&Tfr;","&#x1D517;","&#120087;"));
		entries.add(new Entry(((char)0x1D518),"&Ufr;","&#x1D518;","&#120088;"));
		entries.add(new Entry(((char)0x1D519),"&Vfr;","&#x1D519;","&#120089;"));
		entries.add(new Entry(((char)0x1D51A),"&Wfr;","&#x1D51A;","&#120090;"));
		entries.add(new Entry(((char)0x1D51B),"&Xfr;","&#x1D51B;","&#120091;"));
		entries.add(new Entry(((char)0x1D51C),"&Yfr;","&#x1D51C;","&#120092;"));
		entries.add(new Entry(((char)0x1D51E),"&afr;","&#x1D51E;","&#120094;"));
		entries.add(new Entry(((char)0x1D51F),"&bfr;","&#x1D51F;","&#120095;"));
		entries.add(new Entry(((char)0x1D520),"&cfr;","&#x1D520;","&#120096;"));
		entries.add(new Entry(((char)0x1D521),"&dfr;","&#x1D521;","&#120097;"));
		entries.add(new Entry(((char)0x1D522),"&efr;","&#x1D522;","&#120098;"));
		entries.add(new Entry(((char)0x1D523),"&ffr;","&#x1D523;","&#120099;"));
		entries.add(new Entry(((char)0x1D524),"&gfr;","&#x1D524;","&#120100;"));
		entries.add(new Entry(((char)0x1D525),"&hfr;","&#x1D525;","&#120101;"));
		entries.add(new Entry(((char)0x1D526),"&ifr;","&#x1D526;","&#120102;"));
		entries.add(new Entry(((char)0x1D527),"&jfr;","&#x1D527;","&#120103;"));
		entries.add(new Entry(((char)0x1D528),"&kfr;","&#x1D528;","&#120104;"));
		entries.add(new Entry(((char)0x1D529),"&lfr;","&#x1D529;","&#120105;"));
		entries.add(new Entry(((char)0x1D52A),"&mfr;","&#x1D52A;","&#120106;"));
		entries.add(new Entry(((char)0x1D52B),"&nfr;","&#x1D52B;","&#120107;"));
		entries.add(new Entry(((char)0x1D52C),"&ofr;","&#x1D52C;","&#120108;"));
		entries.add(new Entry(((char)0x1D52D),"&pfr;","&#x1D52D;","&#120109;"));
		entries.add(new Entry(((char)0x1D52E),"&qfr;","&#x1D52E;","&#120110;"));
		entries.add(new Entry(((char)0x1D52F),"&rfr;","&#x1D52F;","&#120111;"));
		entries.add(new Entry(((char)0x1D530),"&sfr;","&#x1D530;","&#120112;"));
		entries.add(new Entry(((char)0x1D531),"&tfr;","&#x1D531;","&#120113;"));
		entries.add(new Entry(((char)0x1D532),"&ufr;","&#x1D532;","&#120114;"));
		entries.add(new Entry(((char)0x1D533),"&vfr;","&#x1D533;","&#120115;"));
		entries.add(new Entry(((char)0x1D534),"&wfr;","&#x1D534;","&#120116;"));
		entries.add(new Entry(((char)0x1D535),"&xfr;","&#x1D535;","&#120117;"));
		entries.add(new Entry(((char)0x1D536),"&yfr;","&#x1D536;","&#120118;"));
		entries.add(new Entry(((char)0x1D537),"&zfr;","&#x1D537;","&#120119;"));
		
//		entries.add(new Entry(((char)0x1D538),"&Aopf;","&#x1D538;","&#120120;"));
//		entries.add(new Entry(((char)0x1D539),"&Bopf;","&#x1D539;","&#120121;"));
//		entries.add(new Entry(((char)0x1D53B),"&Dopf;","&#x1D53B;","&#120123;"));
//		entries.add(new Entry(((char)0x1D53C),"&Eopf;","&#x1D53C;","&#120124;"));
//		entries.add(new Entry(((char)0x1D53D),"&Fopf;","&#x1D53D;","&#120125;"));
//		entries.add(new Entry(((char)0x1D53E),"&Gopf;","&#x1D53E;","&#120126;"));
//		entries.add(new Entry(((char)0x1D540),"&Iopf;","&#x1D540;","&#120128;"));
//		entries.add(new Entry(((char)0x1D541),"&Jopf;","&#x1D541;","&#120129;"));
//		entries.add(new Entry(((char)0x1D542),"&Kopf;","&#x1D542;","&#120130;"));
//		entries.add(new Entry(((char)0x1D543),"&Lopf;","&#x1D543;","&#120131;"));
//		entries.add(new Entry(((char)0x1D544),"&Mopf;","&#x1D544;","&#120132;"));
//		entries.add(new Entry(((char)0x1D546),"&Oopf;","&#x1D546;","&#120134;"));
//		entries.add(new Entry(((char)0x1D54A),"&Sopf;","&#x1D54A;","&#120138;"));
//		entries.add(new Entry(((char)0x1D54B),"&Topf;","&#x1D54B;","&#120139;"));
//		entries.add(new Entry(((char)0x1D54C),"&Uopf;","&#x1D54C;","&#120140;"));
//		entries.add(new Entry(((char)0x1D54D),"&Vopf;","&#x1D54D;","&#120141;"));
//		entries.add(new Entry(((char)0x1D54E),"&Wopf;","&#x1D54E;","&#120142;"));
//		entries.add(new Entry(((char)0x1D54F),"&Xopf;","&#x1D54F;","&#120143;"));
//		entries.add(new Entry(((char)0x1D550),"&Yopf;","&#x1D550;","&#120144;"));
//		entries.add(new Entry(((char)0x1D552),"&aopf;","&#x1D552;","&#120146;"));
//		entries.add(new Entry(((char)0x1D553),"&bopf;","&#x1D553;","&#120147;"));
//		entries.add(new Entry(((char)0x1D554),"&copf;","&#x1D554;","&#120148;"));
//		entries.add(new Entry(((char)0x1D555),"&dopf;","&#x1D555;","&#120149;"));
//		entries.add(new Entry(((char)0x1D556),"&eopf;","&#x1D556;","&#120150;"));
//		entries.add(new Entry(((char)0x1D557),"&fopf;","&#x1D557;","&#120151;"));
//		entries.add(new Entry(((char)0x1D558),"&gopf;","&#x1D558;","&#120152;"));
//		entries.add(new Entry(((char)0x1D559),"&hopf;","&#x1D559;","&#120153;"));
//		entries.add(new Entry(((char)0x1D55A),"&iopf;","&#x1D55A;","&#120154;"));
//		entries.add(new Entry(((char)0x1D55B),"&jopf;","&#x1D55B;","&#120155;"));
//		entries.add(new Entry(((char)0x1D55C),"&kopf;","&#x1D55C;","&#120156;"));
//		entries.add(new Entry(((char)0x1D55D),"&lopf;","&#x1D55D;","&#120157;"));
//		entries.add(new Entry(((char)0x1D55E),"&mopf;","&#x1D55E;","&#120158;"));
//		entries.add(new Entry(((char)0x1D55F),"&nopf;","&#x1D55F;","&#120159;"));
//		entries.add(new Entry(((char)0x1D560),"&oopf;","&#x1D560;","&#120160;"));
//		entries.add(new Entry(((char)0x1D561),"&popf;","&#x1D561;","&#120161;"));
//		entries.add(new Entry(((char)0x1D562),"&qopf;","&#x1D562;","&#120162;"));
//		entries.add(new Entry(((char)0x1D563),"&ropf;","&#x1D563;","&#120163;"));
//		entries.add(new Entry(((char)0x1D564),"&sopf;","&#x1D564;","&#120164;"));
//		entries.add(new Entry(((char)0x1D565),"&topf;","&#x1D565;","&#120165;"));
//		entries.add(new Entry(((char)0x1D566),"&uopf;","&#x1D566;","&#120166;"));
//		entries.add(new Entry(((char)0x1D567),"&vopf;","&#x1D567;","&#120167;"));
//		entries.add(new Entry(((char)0x1D568),"&wopf;","&#x1D568;","&#120168;"));
//		entries.add(new Entry(((char)0x1D569),"&xopf;","&#x1D569;","&#120169;"));
//		entries.add(new Entry(((char)0x1D56A),"&yopf;","&#x1D56A;","&#120170;"));
//		entries.add(new Entry(((char)0x1D56B),"&zopf;","&#x1D56B;","&#120171;"));
		
		for (Entry entry : entries) {
			if (entriesByChar.containsKey(entry.getCharacter())) {
				boolean fixed = false;
				for (String entity : entry.getEntities()) {
					if (entity.startsWith("&#x")) {
						String hexString = entity.substring(3, entity.length() - 1);
						long l = Long.parseLong(hexString, 16);
						if (l > Integer.MAX_VALUE) {
							throw new IllegalStateException("double bound character: " + entry.getCharacter());
						}
						int i = (int) l;
						char c = (char) i;
						if (entriesByChar.containsKey(c)) {
							throw new IllegalStateException("double bound character: " + entry.getCharacter());
						}
						entriesByChar.put(c, entry);
						fixed = true;
					}
				}
				if (!fixed) {
					throw new IllegalStateException("double bound character: " + entry.getCharacter());
				}
			} else {
				entriesByChar.put(entry.getCharacter(), entry);
			}
			for (String entity : entry.getEntities()) {
				if (entriesByEntity.containsKey(entity)) {
					throw new IllegalStateException("double bound entity: " + entity);
				}
				entriesByEntity.put(entity, entry);
			}
		}
	}
	
}
