/**
 * Copyright (C) 2019 Bonitasoft S.A.
 * Bonitasoft, 32 rue Gustave Eiffel - 38000 Grenoble
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 **/
package org.bonitasoft.engine.scheduler;

import java.util.Date;
import java.util.List;

import org.bonitasoft.engine.scheduler.exception.SSchedulerException;
import org.bonitasoft.engine.scheduler.impl.SchedulerServiceImpl;
import org.bonitasoft.engine.scheduler.trigger.Trigger;

/**
 * @author Matthieu Chaffotte
 * @author Celine Souchet
 */
public interface SchedulerExecutor {

    boolean isStarted() throws SSchedulerException;

    boolean isShutdown() throws SSchedulerException;

    /**
     * Note that once a scheduler is shutdown, it cannot be restarted without being re-instantiated.
     *
     * @throws SSchedulerException
     * @since 6.4.0
     */
    void start() throws SSchedulerException;

    /**
     * Note that once a scheduler is shutdown, it cannot be restarted without being re-instantiated.
     *
     * @throws SSchedulerException
     * @since 6.4.0
     */
    void shutdown() throws SSchedulerException;

    boolean mayFireAgain(String jobName) throws SSchedulerException;

    void rescheduleErroneousTriggers() throws SSchedulerException;

    boolean delete(String jobName) throws SSchedulerException;

    void deleteJobs() throws SSchedulerException;

    List<String> getJobs() throws SSchedulerException;

    void setBOSSchedulerService(SchedulerServiceImpl schedulerService);

    void schedule(long jobId, String jobName, Trigger trigger, boolean disallowConcurrentExecution)
            throws SSchedulerException;

    void executeAgain(long jobId, String jobName, boolean disallowConcurrentExecution,
            int delayInMillis) throws SSchedulerException;

    void pauseJobs() throws SSchedulerException;

    void resumeJobs() throws SSchedulerException;

    /**
     * Remove (delete) the <code>{@link org.quartz.Trigger}</code> with the given key, and store the new given one -
     * which must be associated
     * with the same job (the new trigger must have the job name specified)
     * - however, the new trigger need not have the same name as the old trigger.
     *
     * @param triggerName
     *        The name of the trigger to replace
     * @param triggerStartTime
     *        The start date of the new trigger
     * @return <code>null</code> if a <code>Trigger</code> with the given
     *         name was not found and removed from the store (and the
     *         new trigger is therefore not stored), otherwise
     *         the first fire time of the newly scheduled trigger is returned.
     * @throws SSchedulerException
     * @since 6.4.0
     */
    Date rescheduleJob(String triggerName, Date triggerStartTime) throws SSchedulerException;

    /**
     * Check if a job exists.
     *
     * @param jobName
     *        The name of the job
     * @return True if the job exists, else False.
     * @throws SSchedulerException
     * @since 6.4.0
     */
    boolean isExistingJob(String jobName) throws SSchedulerException;
}
